// $Id: grid.cpp,v 1.1 2005/02/18 10:26:43 grosskur Exp $
//
// Copyright (C) 2003 Alan Grosskurth
//
// This file is part of Spatter.
//
// Spatter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Spatter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Spatter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

#include "grid.hpp"
#include "dsvfile.hpp"
#include "stringparser.hpp"

#include <fstream>

static double area(const std::vector< lti::point >& points);

static lti::dpoint centroid(const std::vector< lti::point >& points);


//
// Grid
//

Grid::Grid() {}

Grid::Grid(const lti::point& size, const lti::dpoint& delta,
           const lti::point& subgrid_size, const lti::dpoint& subgrid_delta)
    : m_size(size), m_delta(delta),
      m_subgrid_size(subgrid_size), m_subgrid_delta(subgrid_delta),
      m_origin()
{
    for (int r = 0; r < this->size()[0]; ++r) {
        for (int c = 0; c < this->size()[1]; ++c) {
            m_subgrid_origins.push_back(
                lti::point(lti::dpoint(r, c) * this->delta()));
        }
    }
}

Grid::Grid(std::istream& in)
{
    this->load(in);
}

#if 0
Grid::Grid(std::vector< SubGrid > subgrids,
           const Geometry& geo, const Geometry& subgeo)
    : ArrayGeometry(geo, subgeo), m_subgrids(subgrids)
{
    if (subgrids.empty()) {
        m_origin = lti::point(0,0);
    } else {
        m_origin = subgrids[0].origin();
    }
}
#endif

Grid::~Grid()
{
}

const lti::point
Grid::size() const
{
    return m_size;
}

const lti::dpoint
Grid::delta() const
{
    return m_delta;
}

const lti::point
Grid::subgrid_size() const
{
    return m_subgrid_size;
}

const lti::dpoint
Grid::subgrid_delta() const
{
    return m_subgrid_delta;
}

void
Grid::move(const lti::point& dist)
{
    m_origin += dist;
}

const lti::point&
Grid::origin() const
{
    return m_origin;
}

const lti::point
Grid::pixel_size() const
{
    return lti::point(lti::dpoint(size()) * delta());
}

void
Grid::subgrid_move(int row, int column, const lti::point& dist)
{
    m_subgrid_origins[row*size()[1] + column] += dist;
}

const lti::point
Grid::subgrid_origin(int row, int column) const
{
    return origin() + m_subgrid_origins[row*size()[1] + column];
}

const lti::point
Grid::subgrid_node(int row, int column, const lti::point& coord) const
{
    return subgrid_origin(row, column)
        + lti::point(lti::dpoint(coord) * subgrid_delta());
}

const lti::point
Grid::subgrid_node(int row, int column,
                   int subgrid_row, int subgrid_column) const
{
    lti::point coord(subgrid_row, subgrid_column);
    return subgrid_node(row, column, coord);
}


const lti::point
Grid::subgrid_pixel_size() const
{
    return lti::point(lti::dpoint(subgrid_size()) * subgrid_delta());
}

int
Grid::subgrid_count() const
{
    return m_subgrid_origins.size();
}

const lti::point
Grid::spot_center(int row, int column,
                  int subgrid_row, int subgrid_column) const
{
    std::vector< lti::point > points;

    points.push_back(
        subgrid_node(row, column,
                     lti::point(subgrid_row,     subgrid_column    )));

    points.push_back(
        subgrid_node(row, column,
                     lti::point(subgrid_row + 1, subgrid_column    )));

    points.push_back(
        subgrid_node(row, column,
                     lti::point(subgrid_row + 1, subgrid_column + 1)));

    points.push_back(
        subgrid_node(row, column,
                     lti::point(subgrid_row,     subgrid_column + 1)));

    return centroid(points);
}


const lti::point
Grid::spot_origin(int row, int column,
                  int subgrid_row, int subgrid_column) const
{
    return spot_center(row, column, subgrid_row, subgrid_column)
        - subgrid_delta()/2;
}

int
Grid::spot_count() const
{
    return size()[0] * size()[1] * subgrid_size()[0] * subgrid_size()[1];
}


void
Grid::load(std::istream& in)
{
    DsvFileReader tab_file(in, ':');

    std::vector< std::string > data;

    // Read the array geometry
    tab_file.read_line(data);

    if (data.size() != 8) {
        std::cout << "Invalid grid file" << std::endl;
        return;
    }

    m_size = lti::point(spatter::parse_int(data[0]),
                        spatter::parse_int(data[1]));
    m_delta = lti::dpoint(spatter::parse_double(data[2]),
                          spatter::parse_double(data[3]));
    
    m_subgrid_size = lti::point(spatter::parse_int(data[4]),
                                spatter::parse_int(data[5]));
    m_subgrid_delta = lti::dpoint(spatter::parse_double(data[6]),
                                  spatter::parse_double(data[7]));

    data.clear();

    // Read the subgrid origins
    while (tab_file.read_line(data) != -1) {
        lti::point subgrid_origin = lti::point(
            spatter::parse_int(data[0]),
            spatter::parse_int(data[1]));
        m_subgrid_origins.push_back(subgrid_origin);
        data.clear();
    }
}

void
Grid::save(std::ostream& out) const
{
    DsvFileWriter tab_file(out, ':');

    std::vector< std::string > data;

    // Write the array geometry
    data.push_back(spatter::unparse_int(this->size()[0]));
    data.push_back(spatter::unparse_int(this->size()[1]));
    data.push_back(spatter::unparse_double(this->delta()[0]));
    data.push_back(spatter::unparse_double(this->delta()[1]));
    data.push_back(spatter::unparse_int(this->subgrid_size()[0]));
    data.push_back(spatter::unparse_int(this->subgrid_size()[1]));
    data.push_back(spatter::unparse_double(this->subgrid_delta()[0]));
    data.push_back(spatter::unparse_double(this->subgrid_delta()[1]));

    tab_file.write_line(data);

    data.clear();

    // Write the subgrid origins
    for (int i = 0; i < size()[0]; ++i) {
        for (int j = 0; j < size()[1]; ++j) {
            data.push_back(spatter::unparse_int(subgrid_origin(i,j)[0]));
            data.push_back(spatter::unparse_int(subgrid_origin(i,j)[1]));

            tab_file.write_line(data);

            data.clear();
        }
    }
}

// STATIC FUNCTIONS

double
area(const std::vector< lti::point >& points)
{
    double sum = 0.0;
    int N = points.size();
    for (int i = 0; i < N; ++i) {
        lti::point p = points[i];
        lti::point q = points[(i+1) % N];
        sum += p[0]*q[1] - q[0]*p[1];
    }

    return sum / 2.0;
}

lti::dpoint
centroid(const std::vector< lti::point >& points)
{
    lti::dpoint sum(0,0);
    int N = points.size();
    for (int i = 0; i < N; ++i) {
        lti::point p = points[i];
        lti::point q = points[(i+1) % N];
        double temp = p[0]*q[1] - q[0]*p[1];
        sum[0] += (p[0] + q[0]) * temp;
        sum[1] += (p[1] + q[1]) * temp;
    }
    sum = sum / (6 * area(points));
    return sum;
}
