#!/usr/bin/env python
#
# $Id$
#
# Copyright (C) 2003 Alan Grosskurth
#
# This file is part of Spatter.
#
# Spatter is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Spatter is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Spatter; if not, write to the Free Software Foundation,
# Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

import Image, ImageDraw, ImageFilter, ImageStat, ImageChops
import numarray
import arrayutils

class Geometry:
    def __init__(self, rows, cols, row_delta, col_delta):
        self.size = numarray.array([rows, cols])
        self.delta = numarray.array([row_delta, col_delta])

def load_array_geometry(fname):
    import re
    lines = open(fname).readlines()
    if len(lines) != 1:
        return None
    r = re.compile(
        '(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)$')

    # Find the array geometry
    m1 = r.match(lines[0])
    if not m1:
        return None
    geo = Geometry(int(m1.group(1)), int(m1.group(2)),
                   int(m1.group(3)), int(m1.group(4)))
    subgeo = Geometry(int(m1.group(5)), int(m1.group(6)),
                      int(m1.group(7)), int(m1.group(8)))

    return (geo, subgeo)

def save_grid(grid, fname):
    lines = []
    for subgrid in grid.subgrids:
        str = '%d\t%d\t%f\t%f\t%f\t%f\n' \
              % (subgrid.origin[0], subgrid.origin[1],
                 subgrid.transf[0,0], subgrid.transf[0,1],
                 subgrid.transf[1,0], subgrid.transf[1,1])
        lines.append(str)
    out = open(fname, 'w')
    out.writelines(lines)
    out.close()

def load_grid(fname, geo, subgeo):
    import re
    r = re.compile('(\d+)\s+(\d+)')
    file = open(fname)
    subgrids = []
    for line in file.readlines():
        m = r.search(line)
        if m:
            row = int(m.group(1))
            col = int(m.group(2))
            origin = numarray.array([row,col])
            subgrids.append(SubGrid(origin, subgeo))
        else:
            return None
    grid = Grid(None, geo, subgeo, subgrids)
    return grid

def load_spots(fname):
    import re
    lines = open(fname).readlines()
    r = re.compile('(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)$')
    spots = []
    
    for line in lines:
        m = r.match(line)
        origin = numarray.array([int(m.group(1)), int(m.group(2))])
        shape = (int(m.group(3)), int(m.group(4)))
        mask_string = m.group(5)
        mask = numarray.zeros(shape[0]*shape[1])
        for i in range(len(mask_string)):
            mask[i] = int(mask_string[i])
        mask.shape = shape
        spots.append(Spot(origin=origin, mask=mask))

    return spots

def load_spots_test():
    spots = load_spots('12244189.spots.txt')

class Grid:
    def __init__(self, origin, geo, subgeo, subgrids=None):
        if not origin:
            if subgrids:
                origin = subgrids[0].origin
            else:
                origin = numarray.zeros(2)
            
        self.origin = origin
        self.geo = geo
        self.subgeo = subgeo

        if subgrids:
            self.subgrids = subgrids
        else:
            self.subgrids = []
            for r in range(self.geo.size[0]):
                for c in range(self.geo.size[1]):
                    sub_origin = self.origin + \
                                 numarray.array([r,c]) * self.geo.delta
                    self.subgrids.append(SubGrid(sub_origin, self.subgeo))

    def size(self):
        return self.geo.size * self.geo.delta

    def move(self, dist):
        self.origin += dist
        for subgrid in self.subgrids:
            subgrid.move(dist)

    def draw(self, image, color):
        for subgrid in self.subgrids:
            subgrid.draw(image, color)

    def draw_blocky(self, image, color):
        for subgrid in self.subgrids:
            subgrid.draw_blocky(image, color)

    def save(self, filename):
        file = open(filename, 'w')
        i = 0
        for subgrid in self.subgrids:
            subgrid.save(file, i / self.geo.size[0], i % self.geo.size[0])
            i = i + 1

class SubGrid:
    def __init__(self, origin, geo, transf=None):
        self.origin = origin
        self.geo = geo
        if not transf:
            self.transf = numarray.identity(2, numarray.Float)
        else:
            self.transf = transf

    def origin(self):
        return self.origin

    def move(self, dist):
        self.origin += dist

    def size(self):
        return (self.geo.size - numarray.array([1,1])) * self.geo.delta \
               + numarray.array([1,1])

    def num_cells(self):
        return product(self.geo.size)

    def get_node(self, coord):
        return numarray.matrixmultiply(self.transf,
                                       self.origin + coord*self.geo.delta)

    def get_array(self):
        a = numarray.zeros(self.size())
        for i in range(self.geo.size[0]):
            for j in range(self.geo.size[1]):
                a[int(i*self.geo.delta[0]), :] = 1
                a[:, int(j*self.geo.delta[1])] = 1
        return a

    def draw(self, image, outline_color=1):
        draw = ImageDraw.Draw(image)
        for r in range(self.geo.size[0]):
            for c in range(self.geo.size[1]):
                draw.rectangle((self.get_node((r,c))[1],
                                self.get_node((r,c))[0],
                                self.get_node((r+1,c+1))[1],
                                self.get_node((r+1,c+1))[0]),
                               outline=outline_color)

    def draw_blocky(self, image, fill_color=1):
        draw = ImageDraw.Draw(image)
        draw.rectangle((self.origin[1],
                        self.origin[0],
                        self.origin[1] + self.size()[1],
                        self.origin[0] + self.size()[0]),
                        fill=fill_color)

    def draw_array(self, a, color=1):
        b = numarray.zeros(self.size())
        
        
    def save(self, file, row, col):
        lines = []
        for r in range(self.geo.size[0]):
            for c in range(self.geo.size[1]):
                node = self.get_node((r,c))
                lines.append("%d\t%d\t%d\t%d\t%d\t%d\n" %
                             (row, col, r, c, node[0], node[1]))
        file.writelines(lines)

class Spot:
    def __init__(self, origin, mask):
        self.origin = origin
        self.centre = origin - numarray.array([mask.shape[0]/2,
                                               mask.shape[1]/2])
        self.mask = mask

    def get_patch(self, a):
        r0 = self.origin[0]
        r1 = self.origin[0] + self.mask.shape[0]
        c0 = self.origin[1]
        c1 = self.origin[1] + self.mask.shape[1]
        return a[r0:r1,c0:c1]

    def set_patch(self, a, b):
        r0 = self.origin[0]
        r1 = self.origin[0] + self.mask.shape[0]
        c0 = self.origin[1]
        c1 = self.origin[1] + self.mask.shape[1]
        a[r0:r1,c0:c1] = b

    def draw(self, a, fill=1):
        lines = fill*numarray.ones(self.mask.shape)
        region = self.get_patch(a).copy()
        numarray.putmask(region, arraydraw.outline(self.mask), lines)
        #numarray.putmask(region, self.mask, lines)
        self.set_patch(a, region)

def rank_filter(a, w, func):
    import MA
    b = numarray.zeros(a.shape, a.type())
    w_inv = 1 - w
    for r in range(0, a.shape[0] - w.shape[0] + 1):
        for c in range(0, a.shape[1] - w.shape[1] + 1):
            region = MA.array(a[r:r + w.shape[0], c:c + w.shape[1]],
                              mask=w_inv)
            if func == 'max':
                value = max(max(region.tolist()))
                b[r + w.shape[0]/2, c + w.shape[1]/2] = value
                #print region
                #print b[r + w.shape[0]/2, c + w.shape[1]/2]
            else:
                value = min(min(region.tolist()))
                b[r + w.shape[0]/2, c + w.shape[1]/2] = value
    return b

def rank_filter_test():
    a = numarray.identity(30)
    w = numarray.array([[0,1,0],
                        [1,1,1],
                        [0,1,0]])
    b = rank_filter(a,w,'max')
    print b

def opening(a, w):
    b = rank_filter(a,w,'min')
    print b
    c = rank_filter(b,w,'max')
    print c[2,2]
    return c

def opening_test():
    import numarray.random_array
    a = numarray.random_array.standard_normal((5,5))
    print a
    w = numarray.array([[0,1,0],
                        [1,1,1],
                        [0,1,0]])
    b = opening(a, w)
    print b

from microarray import *
def opening_test2():
    import Image
    im = Image.open('/h/45/grosskur/nserc/arrays/12244189-Cy3.Tif')
    a = image2array(im)
    w = numarray.array([[0,1,0],
                        [1,1,1],
                        [0,1,0]])
    b = opening(a, w)
    imout = array2image(b)
    imout.save('images/12244189-Cy3-opened.pgm')

def image2array_test():
    im = Image.open('images/12244189.pgm')
    #a = numarray.array(list(im.getdata()))
    a = image2array(im)
    im2 = array2image(a)
    print a
    im2.save('images/gtk2.pgm')

def load_array_geometry_test():
    (geo, subgeo) = load_array_geometry('uhn_h19k.geo')
    print geo.size
    print geo.delta
    print subgeo.size
    print subgeo.delta

def main():
    #rank_filter_test()
    #opening_test()
    #opening_test2()
    load_spots_test()

if __name__ == '__main__':
    main()
