// $Id$
//
// Copyright (C) 2003 Alan Grosskurth
//
// This file is part of Spatter.
//
// Spatter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Spatter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Spatter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

#include "tabdelimtextfile.hpp"

#include <sstream>
#include <fstream>


DelimTextFileReader::DelimTextFileReader(const char* fname)
    : m_fs(fname)
{
}

DelimTextFileReader::~DelimTextFileReader()
{
    if (m_fs.is_open()) {
        m_fs.close();
    }
}

int
DelimTextFileReader::read_line(std::vector< std::string >& data)
{
    assert(m_fs.is_open());

    std::string line;
    const std::string delims(" \t,;");

    if (!std::getline(m_fs, line).eof()) {
        std::string::size_type beg_idx = line.find_first_not_of(delims);

        while (beg_idx != std::string::npos) {
            std::string::size_type end_idx =
                line.find_first_of(delims, beg_idx);

            if (end_idx == std::string::npos) {
                end_idx = line.length();
            }

            data.push_back(line.substr(beg_idx, end_idx-beg_idx));

            beg_idx = line.find_first_not_of(delims, end_idx);
        }
    } else {
        return -1;
    }

    return 0;
}

DelimTextFileWriter::DelimTextFileWriter(const char* fname)
    : m_fs(fname)
{
}


DelimTextFileWriter::~DelimTextFileWriter()
{
    if (m_fs.is_open()) {
        m_fs.close();
    }
}


int
DelimTextFileWriter::write_line(std::vector< std::string >& data)
{
    assert(m_fs.is_open());

    std::vector< std::string >::const_iterator elem;
    for (elem = data.begin(); elem + 1 != data.end(); ++elem) {
        m_fs << *elem << '\t';
    }
    m_fs << *elem << '\n';

    return 0;
}


int
StringParser::parse_int(const std::string& str)
{
    int x;
    std::istringstream is(str);
    is >> x;
    return x;
}


double
StringParser::parse_double(const std::string& str)
{
    double x;
    std::istringstream is(str);
    is >> x;
    return x;
}

lti::imatrix
StringParser::parse_matrix(const std::string& str, int rows, int columns)
{
    int* data = new int[rows * columns];

    for (int i = 0; str[i] && i < rows * columns; ++i) {
        data[i] = parse_int(str.substr(i, 1));
    }

    lti::imatrix mat(rows, columns, data);

    return mat;
}


lti::ubyte*
StringParser::parse_int_array(const char* s, unsigned int len)
{
    std::vector< std::string > v;
    std::string cur = "";

    for (int i = 0; s[i]; ++i) {
        if (s[i] != ',') {
            cur += s[i];
        } else {
            v.push_back(cur);
            cur = "";
        }   
    }
    v.push_back(cur);

    // sanity check
    if (v.size() != len)
        return 0;

    lti::ubyte* data = new lti::ubyte[v.size()];

    for (unsigned int i = 0; i < v.size(); ++i) {
        data[i] = parse_int(v[i].c_str());
    }

    return data;
}

std::string
StringParser::unparse_int(int x)
{
    std::ostringstream oss;
    oss << x;

    return oss.str();
}

std::string
StringParser::unparse_double(double x)
{
    std::ostringstream oss;
    oss << x;

    return oss.str();
}

std::pair< std::string, std::string >
StringParser::unparse_point(lti::point point)
{
    return make_pair(unparse_int(point[0]), unparse_int(point[1]));
}

std::string
StringParser::unparse_matrix(lti::imatrix mat)
{
    std::string str = "";

    for (int r = 0; r < mat.rows(); ++r) {
        for (int c = 0; c < mat.columns(); ++c) {
            str += unparse_int(mat[r][c]);
        }
    }

    return str;
}

#if 0
void
save_tab_delimited_text_file(const char* fname,
                             std::vector< std::vector< std::string > >& data)
{
    std::ofstream outf(fname);

    std::vector< std::vector< std::string > >::const_iterator line;
    for (line = data.begin(); line != data.end(); ++line) {
        std::vector< std::string >::const_iterator elem;
        for (elem = line->begin(); elem + 1 != line->end(); ++elem) {
            outf << *elem << '\t';
        }
        outf << *elem << '\n';
    }
}



std::vector< std::vector< std::string > >
load_tab_delimited_text_file(const char* fname)
{
    std::ifstream inf(fname);
    std::string line;
    const std::string delims(" \t,;");

    std::vector< std::vector< std::string > > results;

    while (!std::getline(inf, line).eof()) {
        //std::cout << line << std::endl;
        std::string::size_type beg_idx = line.find_first_not_of(delims);
        //std::cout << "beg_idx: " << beg_idx << std::endl;
        std::vector< std::string > line_result;
        while (beg_idx != std::string::npos) {
            std::string::size_type end_idx =
                line.find_first_of(delims, beg_idx);

            if (end_idx == std::string::npos) {
                end_idx = line.length();
            }

            line_result.push_back(line.substr(beg_idx, end_idx-beg_idx));

            beg_idx = line.find_first_not_of(delims, end_idx);
        }
        results.push_back(line_result);
    }
    return results;
}


void
save_spots(const char* fname, const std::vector< Spot > spots)
{
    std::vector< std::vector< std::string > > data;
    for (unsigned int i = 0; i < spots.size(); ++i) {
        std::vector< std::string > line;

        std::pair< std::string, std::string > origin_pair =
            unparse_point(spots[i].origin());
        line.push_back(origin_pair.first);
        line.push_back(origin_pair.second);

        line.push_back(unparse_int(spots[i].mask().rows()));
        line.push_back(unparse_int(spots[i].mask().columns()));

        line.push_back(unparse_matrix(spots[i].mask()));

        data.push_back(line);
    }

    save_tab_delimited_text_file(fname, data);
}

std::pair< Geometry, Geometry >
load_geometry(const char* fname)
{
    std::vector< std::vector< std::string > > data =
        load_tab_delimited_text_file(fname);

    if (data.size() != 2
        || data[0].size() != 4
        || data[1].size() != 4) {
        std::cerr << "Invalid geometry file" << std::endl;
    }

    int rows = parse_int(data[0][0]);
    int cols = parse_int(data[0][1]);
    double row_delta = parse_double(data[0][2]);
    double col_delta = parse_double(data[0][3]);

    int sub_rows = parse_int(data[1][0]);
    int sub_cols = parse_int(data[1][1]);
    double sub_row_delta = parse_double(data[1][2]);
    double sub_col_delta = parse_double(data[1][3]);

    return std::pair< Geometry, Geometry >
        (Geometry(lti::point(rows, cols),
                  lti::dpoint(row_delta, col_delta)),
         Geometry(lti::point(sub_rows, sub_cols),
                  lti::dpoint(sub_row_delta, sub_col_delta)));
}


Grid
load_grid(const char* fname, const Geometry& geo, const Geometry& subgeo)
{
    std::vector< std::vector< std::string > > data =
        load_tab_delimited_text_file(fname);

    std::vector< SubGrid > subgrids;

    std::vector< std::vector< std::string > >::const_iterator pos;
    for (pos = data.begin(); pos != data.end(); ++pos) {
        if (pos->size() != 6) {
            std::cerr << "Invalid grid file" << std::endl;            
        }

        int x = parse_int((*pos)[0]);
        int y = parse_int((*pos)[1]);
        double transf[] = {
            parse_double((*pos)[2]),
            parse_double((*pos)[3]),
            parse_double((*pos)[3]),
            parse_double((*pos)[3])
        };

        subgrids.push_back(SubGrid(lti::point(x, y),
                                   lti::dmatrix(2, 2, transf),
                                   subgeo));
    }

    return Grid(subgrids, geo, subgeo);
}

void
save_grid(const char* fname, const Grid& grid)
{
    std::vector< std::vector< std::string > > data;

    for (int i = 0; i < grid.size()[0]; ++i) {
        for (int j = 0; j < grid.size()[1]; ++j) {
            std::vector< std::string > line;

            std::pair< std::string, std::string > origin_pair =
                unparse_point(grid.subgrid_origin(i, j));

            line.push_back(origin_pair.first);
            line.push_back(origin_pair.second);

            line.push_back("1");
            line.push_back("0");
            line.push_back("0");
            line.push_back("1");

            data.push_back(line);
        }
    }

    save_tab_delimited_text_file(fname, data);
}
#endif

std::string
common_prefix(std::string& s1, std::string& s2)
{
    unsigned int i;

    for (i = 0; i < s1.length() && i < s2.length(); ++i) {
        if (s1[i] != s2[i]) {
            break;
        }
    }

    return s1.substr(0, i);
}

std::string
basename(std::string& s)
{
    unsigned int i;

    for (i = s.length() - 1; i >= 0; --i) {
        if (s[i] == '/') {
            break;
        }
    }

    return s.substr(i + 1);
}

std::string
dirname(std::string& s)
{
    unsigned int i;

    for (i = s.length() - 1; i >= 0; --i) {
        if (s[i] == '/') {
            break;
        }
    }

    return s.substr(0, i + 1);
}
