// $Id$
//
// Copyright (C) 2003 Alan Grosskurth
//
// This file is part of Spatter.
//
// Spatter is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Spatter is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Spatter; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

#include "gridder.hpp"

#include <iostream>
#include <string>
#include <cstdlib>
#include <limits>

#include <ltiALLFunctor.h>
#include <ltiPoint.h>
#include <ltiMatrix.h>
#include <ltiImage.h>
#include <ltiDraw.h>


int
get_area(lti::channel8& im, lti::channel8& mat, int row, int col)
{
    mat.fill(im,
             0, 0,
             mat.rows(), mat.columns(),
             row, col);

    return mat.sumOfElements();
}

int
find_shift(int total, int a, int b, int c, int d)
{
    //int q1 = std::abs(a - b);
    //int q2 = std::abs(c - d);
    //int q3 = std::abs(a - d);
    //int q4 = std::abs(b - c);

    //int move = q1 + q2;
    //int stay = q3 + q4;

    //std::cout << total << " : "
    //          << a << "," << b << "," << c << "," << d << std::endl;

    int result = 0;

    if (b < c && b < d) {
        result = 1;
    } else if (c < b && c < a) {
        result = -1;
    }

    return result;
}

lti::channel8
get_grid_image(const Grid& grid)
{
    lti::channel8 grid_mat(grid.pixel_size()[0],
                           grid.pixel_size()[1], (lti::ubyte)0);

    // Generate the grid pattern
    for (int i = 0; i < grid.size()[0]; ++i) {
        for (int j = 0; j < grid.size()[1]; ++j) {
            int from_row = (int)(i * grid.delta()[0]);
            int from_col = (int)(j * grid.delta()[1]);
            int to_row = from_row + grid.subgrid_pixel_size()[0];
            int to_col = from_col + grid.subgrid_pixel_size()[1];
#if 0
            std::cout << "(" << i << "," << j << ")" << std::endl;
            std::cout << "  (" << from_row << "," << from_col << ")"
                      << std::endl;
            std::cout << "  (" << to_row << "," << to_col << ")"
                      << std::endl;
#endif
            grid_mat.fill(1, from_row, from_col, to_row, to_col);
        }
    }

    return grid_mat;
}

lti::channel8
get_reverse_grid_spot_image(const Grid& grid)
{
    lti::channel8 mat(grid.subgrid_pixel_size()[0],
                      grid.subgrid_pixel_size()[1], (lti::ubyte)1);

    lti::draw< lti::ubyte > drawer;
    drawer.use(mat);
    drawer.setColor((lti::ubyte)0);

    // Generate the grid pattern
    for (int i = 0; i < grid.subgrid_size()[0]; ++i) {
        for (int j = 0; j < grid.subgrid_size()[1]; ++j) {
            lti::point origin = lti::dpoint(i, j)*grid.subgrid_delta()
                + grid.subgrid_delta()/2;
            int radius = std::max((int)grid.subgrid_delta()[0],
                                  (int)grid.subgrid_delta()[1]) / 2 - 1;
            drawer.circle(origin[1], origin[0], radius, true);
        }
    }

    return mat;   
}

lti::channel8
get_subgrid_image(const Grid& grid)
{
    // Draw one subgrid
    lti::channel8 subgrid_mat(grid.subgrid_pixel_size()[0],
                              grid.subgrid_pixel_size()[1], (lti::ubyte)0);

    // draw the row lines
    for (int i = 0; i < grid.subgrid_size()[0]; ++i) {
        for (int j = 0; j < grid.subgrid_pixel_size()[1]; ++j) {
            subgrid_mat[(int)(i*grid.subgrid_delta()[0])][j] = 1;
        }
    }

    // draw the column lines
    for (int i = 0; i < grid.subgrid_pixel_size()[0]; ++i) {
        for (int j = 0; j < grid.subgrid_size()[1]; ++j) {
            subgrid_mat[i][(int)(j*grid.subgrid_delta()[1])] = 1;
        }
    }

    return subgrid_mat;
}

lti::channel8
get_image(const lti::image& im)
{
    lti::channel8 im8(im.rows(), im.columns(), (lti::ubyte)0);
    for (int i = 0; i < im.rows(); ++i) {
        for (int j = 0; j < im.columns(); ++j) {
            im8[i][j] = im[i][j].getRed();
        }
    }
    return im8;
}

void
adjust_coarse(Grid& grid, lti::channel8& im8, lti::channel8& grid_mat)
{
    lti::point im_size = lti::point(im8.size().y, im8.size().x);
    lti::point move_distance = im_size - grid.pixel_size();

    // Iterate over possible positions
    lti::channel8 region(grid.pixel_size()[0],
                         grid.pixel_size()[1],
                         (lti::ubyte)0);

    lti::channel8 inv_region(grid.pixel_size()[0],
                             grid.pixel_size()[1],
                             (lti::ubyte)0);

    int max_val = std::numeric_limits<int>::min();
    lti::point max_point(0,0);

    //std::cout << move_distance << std::endl;

    for (int i = 0; i < move_distance[0]; i += 16) {
        for (int j = 0; j < move_distance[1]; j += 16) {
            region.fill(im8,
                        0, 0, grid.pixel_size()[0], grid.pixel_size()[1],
                        i, j);

            region.emultiply(grid_mat);

            int curr_val = region.sumOfElements();

//            std::cout << lti::point(i, j) << ": " << curr_val << std::endl;

            if (curr_val > max_val) {
                max_val = curr_val;
                max_point = lti::point(i, j);
            }
        }
//        std::cout << i << " rows completed" << std::endl;
    }

//    std::cout << max_point << " " << max_val << std::endl;

    // Move the grid
    grid.move(max_point);
}

void
adjust_fine(Grid& grid, int i, int j,
            lti::channel8& im8, lti::channel8& subgrid_mat,
            int tol)
{
    int min_val = std::numeric_limits<int>::max();
    lti::point min_point(0,0);

    for (int r = -tol; r < tol; ++r) {
        for (int c = -tol; c < tol; ++c) {
            lti::point origin =
                grid.subgrid_origin(i, j) + lti::point(r, c);

            assert(origin[0] >= 0 && origin[1] >= 0);

            lti::channel8 subregion(grid.subgrid_pixel_size()[0],
                                    grid.subgrid_pixel_size()[1],
                                    (lti::ubyte)0);
            subregion.fill(im8, 0, 0,
                           grid.subgrid_pixel_size()[0],
                           grid.subgrid_pixel_size()[1],
                           origin[0], origin[1]);

            subregion.emultiply(subgrid_mat);

            int curr_val = subregion.sumOfElements();
            if (curr_val < min_val) {
                min_val = curr_val;
                min_point = lti::point(r, c);
            }
        }
    }

    // Move the grid
    grid.subgrid_move(i, j, min_point);
//    std::cout << "  Shift: " << min_point << std::endl;
//    std::cout << "  After: " << grid.subgrid_origin(i, j) << std::endl;
}

void
adjust_one_off(Grid& grid, int i, int j,
               lti::channel8& im8, lti::channel8& subgrid_mat,
               int num_iter)
{
    // Adjust for "one-spot-off" in either direction

    for (int k = 0; k < num_iter; ++k) {
        lti::point o(grid.subgrid_origin(i, j));

        lti::channel8 mat(grid.subgrid_pixel_size()[0],
                          grid.subgrid_pixel_size()[1],
                          (lti::ubyte)0);

        lti::channel8 col_mat(grid.subgrid_pixel_size()[0],
                              (int)(grid.subgrid_delta()[1]),
                              (lti::ubyte)0);

        int left_inside = get_area(im8, col_mat, o[0], o[1]);
        int left_outside = get_area(im8, col_mat, o[0],
                                    o[1] - grid.subgrid_delta()[1]);
        int right_inside = get_area(im8, col_mat, o[0],
                                    o[1]
                                    + (grid.subgrid_size()[1] - 1)
                                    *grid.subgrid_delta()[1]);
        int right_outside = get_area(im8, col_mat, o[0], o[1]
                                     + grid.subgrid_size()[1]
                                     *grid.subgrid_delta()[1]);

        int row_total = get_area(im8, mat, o[0], o[1])
            / grid.subgrid_size()[0];

        lti::channel8 row_mat(grid.subgrid_delta()[0],
                              grid.subgrid_pixel_size()[1],
                              (lti::ubyte)0);

        int top_inside = get_area(im8, row_mat, o[0], o[1]);
        int top_outside = get_area(im8, row_mat, o[0]
                                   - grid.subgrid_delta()[0], o[1]);
        int bottom_inside = get_area(im8, row_mat, o[0]
                                     + (grid.subgrid_size()[0]-1)
                                     *grid.subgrid_delta()[0], o[1]);
        int bottom_outside = get_area(im8, row_mat, o[0]
                                      + grid.subgrid_size()[0]
                                      *grid.subgrid_delta()[0], o[1]);

        int col_total = get_area(im8, mat, o[0], o[1])
            / grid.subgrid_size()[1];

        lti::point dir(find_shift(row_total,
                                  top_outside, top_inside,
                                  bottom_inside, bottom_outside),
                       find_shift(col_total,
                                  left_outside, left_inside,
                                  right_inside, right_outside));

        if (dir == lti::point(0, 0)) {
            break;
        }

        lti::point shift(grid.subgrid_delta()*lti::dpoint(dir));

        grid.subgrid_move(i, j, shift);

        //std::cout << "  Jump: " << dir << std::endl;
    }
}

Gridder::Gridder()
{
}


MultiStepGridder::MultiStepGridder()
{
}


Grid
MultiStepGridder::apply(const ArrayGeometry& geo, const lti::image& im)
{
    Grid grid(geo.size, geo.delta, geo.subgrid_size, geo.subgrid_delta);

    lti::channel8 grid_mat = get_grid_image(grid);
    lti::channel8 im8 = get_image(im);

    std::cout << "Finding grid origin..." << std::endl;
    adjust_coarse(grid, im8, grid_mat);
    std::cout << "Grid origin: " << grid.origin() << std::endl;

    lti::channel8 subgrid_mat = get_subgrid_image(grid);
    //lti::channel8 subgrid_mat = get_reverse_grid_spot_image(grid);

    std::cout << "Fine-tuning subgrid origins..." << std::endl;

    for (int i = 0; i < grid.size()[0]; ++i) {
        for (int j = 0; j < grid.size()[1]; ++j) {
//            std::cout << "Subgrid: " << lti::point(i, j) << std::endl;
//            std::cout << "  Origin: " << grid.subgrid_origin(i, j)
//                      << std::endl;
            adjust_fine(grid, i, j, im8, subgrid_mat, 16);
            adjust_one_off(grid, i, j, im8, subgrid_mat, 3);
            std::cout << i*grid.size()[1] + j + 1
                      << " of "
                      << grid.size()[0]*grid.size()[1]
                      << " subgrids found"
                      << std::endl;
            //std::cout << "  Final: " << grid.subgrid_origin(i, j)
            //          << std::endl;
        }
    }

    return grid;
}
