;;; spark-mode.el --- addon to the ada-mode for editing SPARK Ada
;;; sources

;; Copyright (C) 2010  Gaétan Allaert

;; Author: Gaétan Allaert <gaetan.allaert@belgacom.net>
;; Maintainer: Gaétan Allaert <gaetan.allaert@belgacom.net>
;; Keywords: languages SPARK ada

;; This file is not part of GNU Emacs.

;; This program is free software: you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation, either version 3 of the License, or
;; (at your option) any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with this program.  If not, see <http://www.gnu.org/licenses/>.

;;; Commentary:
;; This file contains the code for editing SPARK Ada source code. It
;; requires the full ada-mode for GNU Emacs and modifications in the
;; ada-mode.el and in the ada-xref.el.

;;; Usage:
;; The SPARK Ada mode:
;; - TAB                       : indent SPARK annotations;
;; - ENTER                     : add the SPARK annotation "--#" when you insert a newline inside the SPARK annotations;
;; - TYPE                      : adjust casing for identifier in SPARK annotations;
;; - ESC-/                     : auto-completion in SPARK annotation;
;; - C-c C-h                   : reformat/compress/sort derives annotation;
;; - C-c C-g                   : reformat/sort global annotations;
;; - C-c C-i                   : reformat/sort inherit annotations;
;; - C-c C-f                   : reformat proof functions;
;; - spark-currrent            : run SPARK on file
;; - spark-metafile            : run SPARK on metafile
;; - spark-fix-error           : automatic fix of SPARK semantic, flow and capitalization errors;
;; - C-c C-d                   : "Goto Declaration/Body" cross navigation in SPARK annotations;
;; - ada-find-any-references   : "List References" cross navigation in SPARK annotations using gnatfind or gps;
;; - AUTO                      : syntax highlighting of SPARK keywords in annotation;
;; - ada-rename-identifier     : renames identifier in Ada source code and in SPARK annotations;
;; - spark-pretty-print        : pretty print the SPARK source code;
;; - ada-format-call-paramlist : reformat subprogram call or aggregate.

(require 'ada-mode)

(defun spark-own-varlist-no-type-p (varlist)
  "Check if SPARK own variable list contains types"
  (let ((i (length varlist))
	(type-found nil))
    ;; loop until last parameter or type is found
    (while (not (or (zerop i) type-found))
      (setq i (1- i))
      (setq type-found (nth 5 (nth i varlist))))
    (not type-found)))

(defun spark-sort-global-varlist (el1 el2)
  "Sort SPARK own/global variable list"
  (string< (upcase (nth 4 el1)) (upcase (nth 4 el2))))

(defun spark-sort-derives-varlist (el1 el2)
  "Sort SPARK derives variable list"
  (cond
   ((string= (upcase (nth 0 (nth 0 el1))) "NULL")
    nil)
   ((string= (upcase (nth 0 (nth 0 el2))) "NULL")
    t)
   (t
    (string< (upcase (nth 0 (nth 0 el1))) (upcase (nth 0 (nth 0 el2)))))))

(defun spark-global-varlist-reformat (varlist)
  "Reformat the SPARK own/global variable list point is in."
  (let ((varlist-in nil)
	(varlist-in-sorted nil)
	(varlist-out nil)
	(varlist-out-sorted nil)
	(varlist-in-out nil)
	(varlist-in-out-sorted nil)
	(varlist-protected nil)
	(varlist-protected-sorted nil)
	(varlist-global nil)
	(varlist-global-sorted nil)
	(previous-varlist nil)
	(i (length varlist)))
    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (cond
       ((and (nth 1 (nth i varlist))
	     (not (nth 2 (nth i varlist))))
	(setq previous-varlist 'varlist-in)
	(setq varlist-in (append varlist-in (list (list "" ada-spark-explicit-mode nil nil (nth 4 (nth i varlist)) nil)))))

       ((and (not (nth 1 (nth i varlist)))
	     (nth 2 (nth i varlist)))
	(setq previous-varlist 'varlist-out)
	(setq varlist-out (append varlist-out (list (list "" nil ada-spark-explicit-mode nil (nth 4 (nth i varlist)) nil)))))

       ((and (nth 1 (nth i varlist))
	     (nth 2 (nth i varlist)))
	(setq previous-varlist 'varlist-in-out)
	(setq varlist-in-out (append varlist-in-out (list (list "" ada-spark-explicit-mode ada-spark-explicit-mode nil (nth 4 (nth i varlist)) nil)))))

       ((nth 3 (nth i varlist))
	(setq previous-varlist 'varlist-protected)
	(setq varlist-protected (append varlist-protected (list (list "" nil nil ada-spark-explicit-mode (nth 4 (nth i varlist)) nil)))))

       ((and (not (nth 1 (nth i varlist)))
	     (not (nth 2 (nth i varlist)))
	     (not (nth 3 (nth i varlist))))
	(cond
	 ((equal previous-varlist 'varlist-in)
	  (setq varlist-in (append varlist-in (list (list "" ada-spark-explicit-mode nil nil (nth 4 (nth i varlist)) nil)))))

	 ((equal previous-varlist 'varlist-out)
	  (setq varlist-out (append varlist-out (list (list "" nil ada-spark-explicit-mode nil (nth 4 (nth i varlist)) nil)))))

	 ((equal previous-varlist 'varlist-in-out)
	  (setq varlist-in-out (append varlist-in-out (list (list "" ada-spark-explicit-mode ada-spark-explicit-mode nil (nth 4 (nth i varlist)) nil)))))

	 ((equal previous-varlist 'varlist-protected)
	  (setq varlist-protected (append varlist-protected (list (list "" nil nil ada-spark-explicit-mode (nth 4 (nth i varlist)) nil)))))

	 ((not previous-varlist)
	  (setq varlist-global (append varlist-global (list (list "" nil nil nil (nth 4 (nth i varlist)) nil)))))))))

    (setq varlist-in-sorted        (sort varlist-in        'spark-sort-global-varlist))
    (setq varlist-out-sorted       (sort varlist-out       'spark-sort-global-varlist))
    (setq varlist-in-out-sorted    (sort varlist-in-out    'spark-sort-global-varlist))
    (setq varlist-protected-sorted (sort varlist-protected 'spark-sort-global-varlist))
    (setq varlist-global-sorted    (sort varlist-global    'spark-sort-global-varlist))
    (append (if varlist-global-sorted
		(reverse varlist-global-sorted))
	    (append (if varlist-protected-sorted
			(reverse (append (list (list "" nil nil t (nth 4 (car varlist-protected-sorted)) nil))
					 (cdr varlist-protected-sorted))))
		    (append (if varlist-out-sorted
				(reverse (append (list (list "" nil t nil (nth 4 (car varlist-out-sorted)) nil))
						 (cdr varlist-out-sorted))))
			    (append (if varlist-in-out-sorted
					(reverse (append (list (list "" t t nil (nth 4 (car varlist-in-out-sorted)) nil))
							 (cdr varlist-in-out-sorted))))
				    (if varlist-in-sorted
					(reverse (append (list (list "" t nil nil (nth 4 (car varlist-in-sorted))  nil))
							 (cdr varlist-in-sorted))))))))))

(defun spark-format-global ()
  "Reformat the SPARK own/global variable list point is in."
  (interactive)
  (let ((begin nil)
	(end nil)
	(paramlist nil)
	(paramlist-reformatted nil)
	(previous-syntax-table (syntax-table)))
    (unwind-protect
	(progn
	  (set-syntax-table ada-mode-symbol-syntax-table)

	  ;; check if really inside a own/global variable list
	  (or (ada-in-spark-global-varlist-p)
	      (error "Not in SPARK own/global variable list"))

	  ;; find start of current variable-list
	  (ada-search-ignore-string-comment "\\<own\\|global\\>" t nil nil nil t)
	  (save-excursion (beginning-of-line)
			  (ada-indent-current))
	  (forward-word 1)
	  (ada-goto-next-non-ws nil nil t)
	  (save-excursion (beginning-of-line)
			  (ada-indent-current))
	  (setq begin (point))

	  ;; find end of variable-list
	  (ada-search-ignore-string-comment ";" nil nil nil nil t)
	  (let ((p (point)))
	    (while (save-excursion (and
				    (ada-goto-next-non-ws nil nil t)
				    (or (re-search-forward "protected[ \t]" (+ (point) 10) t)
					(re-search-forward "in[ \t]" (+ (point) 3) t)
					(re-search-forward "out[ \t]" (+ (point) 4) t))
				    (setq p (point))))
	      (progn (goto-char p)
		     (ada-search-ignore-string-comment ";" nil nil nil nil t))))
	  (delete-char -1)
	  (insert "\n")

	  ;; find end of last variable-declaration
	  (setq end (point))

	  ;; build a list of all elements of the variable-list
	  (setq paramlist (ada-scan-paramlist begin (1- end) t t))

	  ;; delete the original variable-list
	  (delete-region begin end)

	  ;; reformat the variable-list
	  (setq paramlist-reformatted
		(if (spark-own-varlist-no-type-p paramlist)
		    (spark-global-varlist-reformat paramlist)
		  paramlist))

	  ;; insert the new variable-list
	  (goto-char begin)
	  (ada-insert-paramlist paramlist-reformatted t t))

      ;; restore syntax-table
      (set-syntax-table previous-syntax-table))))

;; Compare A and B (involve A, B, L3 with A not in L3 and B not in L3)

;;      derives A  from *           & B  from *               => derives A, B   from *
;;                                    B  from A               => derives A, B   from A
;;                                    B  from *, A            => derives A, B   from *, A
;;
;;      derives A  from B           & B  from *               => derives A, B   from B
;;
;;      derives A  from L3          & B  from L3              => derives A, B   from L3
;;
;;      derives A  from *, B        & B  from *               => derives A, B   from *, B
;;                                    B  from *, A            => derives A, B   from A, B
;;
;;      derives A  from *, L3       & B  from *, L3           => derives A, B   from *, L3
;;                                    B  from A, L3           => derives A, B   from A, L3
;;                                    B  from *, A, L3        => derives A, B   from *, A, L3
;;
;;      derives A  from B, L3       & B  from *, L3           => derives A, B   from B, L3
;;
;;      derives A  from *, B, L3    & B  from *, L3           => derives A, B   from *, B, L3
;;                                    B  from *, A, L3        => derives A, B   from A, B, L3

;; Compare A and L1 (involve A, L1 and L3 with A not in L1, A not in L3 and L1 may be equal to L3)

;; [01] derives A  from *           & L1 from *               => derives A, L1  from *
;; [02]                               L1 from A               => derives A, L1  from A
;; [03]                               L1 from *, A            => derives A, L1  from *, A
;;
;; [04] derives A  from L1          & L1 from L1              => derives A, L1  from L1 (derives A from L3 & L1 from L3 => derives A, L1 from L3 with L1 = L3)
;; [05]                               L1 from *, L1           => derives A, L1  from L1
;;
;; [06] derives A  from L3          & L1 from L3              => derives A, L1  from L3
;;
;; [07] derives A  from *, L1       & L1 from L1              => derives A, L1  from *, L1
;; [08]                               L1 from *, L1           => derives A, L1  from *, L1 (derives A from *, L3 & L1 from *, L3 => derives A, L1 from *, L3 with L1 = L3)
;; [09]                               L1 from A, L1           => derives A, L1  from A, L1 (derives A from *, L3 & L1 from A, L3 => derives A, L1 from A, L3 with L1 = L3)
;; [10]                               L1 from *, A, L1        => derives A, L1  from A, L1
;;
;; [11] derives A  from *, L3       & L1 from *, L3           => derives A, L1  from *, L3
;; [12]                               L1 from A, L3           => derives A, L1  from A, L3
;; [13]                               L1 from *, A, L3        => derives A, L1  from *, A, L3
;;
;; [14] derives A from L1, L3       & L1 from L1, L3          => derives A, L1  from L1, L3 (derives A from L3 & L1 from L3 => derives A, L1 from L3 with L1 = L3)
;; [15]                               L1 from *, L1, L3       => derives A, L1  from L1, L3
;;
;; [16] derives A  from *, L1, L3   & L1 from L1, L3          => derives A, L1  from *, L1, L3
;; [17]                               L1 from *, L1, L3       => derives A, L1  from *, L1, L3 (derives A from *, L3 & L1 from *, L3 => derives A, L1 from *, L3 with L1 = L3)
;; [18]                               L1 from A, L1, L3       => derives A, L1  from A, L1, L3 (derives A from *, L3 & L1 from A, L3 => derives A, L1 from A, L3 with L1 = L3)
;; [19]                               L1 from *, A, L1, L3    => derives A, L1  from A, L1, L3

;; Compare L1 and A (involve A, L1 and L3 with A not in L1, A not in L3 and L1 may be equal to L3)

;; [01] derives L1 from *            & A  from *              => derives A, L1  from *
;;
;; [02] derives L1 from A            & A  from *              => derives A, L1  from A
;;
;; [04] derives L1 from L1           & A  from L1             => derives A, L1  from L1
;; [07]                                A  from *, L1          => derives A, L1  from *, L1
;;
;; [06] derives L1 from L3           & A  from L3             => derives A, L1  from L3
;;
;; [03] derives L1 from *, A         & A  from *              => derives A, L1  from *, A
;;
;; [05] derives L1 from *, L1        & A  from L1             => derives A, L1  from L1
;; [08]                                A  from *, L1          => derives A, L1  from *, L1
;;
;; [11] derives L1 from *, L3        & A  from *, L3          => derives A, L1  from *, L3
;;
;; [09] derives L1 from A, L1        & A  from *, L1          => derives A, L1  from A, L1
;;
;; [12] derives L1 from A, L3        & A  from *, L3          => derives A, L1  from A, L3
;;
;; [14] derives L1 from L1, L3       & A  from L1, L3         => derives A, L1  from L1, L3
;; [16]                                A  from *, L1, L3      => derives A, L1  from *, L1, L3
;;
;; [10] derives L1 from *, A, L1     & A  from *, L1          => derives A, L1  from A, L1
;;
;; [13] derives L1 from *, A, L3     & A  from *, L3          => derives A, L1  from *, A, L3
;;
;; [15] derives L1 from *, L1, L3    & A  from L1, L3         => derives A, L1  from L1, L3
;; [17]                                A  from *, L1, L3      => derives A, L1  from *, L1, L3
;;
;; [18] derives L1 from A, L1, L3    & A  from *, L1, L3      => derives A, L1  from A, L1, L3
;;
;; [19] derives L1 from *, A, L1, L3 & A  from *, L1, L3      => derives A, L1  from A, L1, L3

;; Compare L1 and L2 (involve L1, L2 and L3)

;;      derives L1 from *             & L2 from *             => derives L1, L2 from *
;;
;;      derives L1 from L1            & L2 from L1            => derives L1, L2 from L1 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L1 = L3)
;;                                      L2 from *, L1         => derives L1, L2 from *, L1
;;
;;      derives L1 from L2            & L2 from L2            => derives L1, L2 from L2 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L2 = L3)
;;                                      L2 from *, L2         => derives L1, L2 from L2
;;
;;      derives L1 from L3            & L2 from L3            => derives L1, L2 from L3
;;
;;      derives L1 from *, L1         & L2 from L1            => derives L1, L2 from L1
;;                                      L2 from *, L1         => derives L1, L2 from *, L1 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L1 = L3)
;;
;;      derives L1 from *, L2         & L2 from L2            => derives L1, L2 from *, L2
;;                                      L2 from *, L2         => derives L1, L2 from *, L2 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L2 = L3)
;;
;;      derives L1 from *, L3         & L2 from *, L3         => derives L1, L2 from *, L3
;;
;;      derives L1 from L1, L2        & L2 from L1, L2        => derives L1, L2 from L1, L2
;;                                      L2 from *, L1, L2     => derives L1, L2 from L1, L2
;;
;;      derives L1 from L1, L3        & L2 from L1, L3        => derives L1, L2 from L1, L3 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L1 = L3)
;;                                      L2 from *, L1, L3     => derives L1, L2 from *, L1, L3
;;
;;      derives L1 from L2, L3        & L2 from L2, L3        => derives L1, L2 from L2, L3 (derives L1 from L3 & L2 from L3 => derives L1, L2 from L3 with L2 = L3)
;;                                      L2 from *, L2, L3     => derives L1, L2 from L2, L3
;;
;;      derives L1 from *, L1, L2     & L2 from L1, L2        => derives L1, L2 from L1, L2
;;                                      L2 from *, L1, L2     => derives L1, L2 from L1, L2
;;
;;      derives L1 from *, L1, L3     & L2 from L1, L3        => derives L1, L2 from L1, L3
;;                                      L2 from *, L1, L3     => derives L1, L2 from *, L1, L3 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L1 = L3)
;;
;;      derives L1 from *, L2, L3     & L2 from L2, L3        => derives L1, L2 from *, L2, L3
;;                                      L2 from *, L2, L3     => derives L1, L2 from *, L2, L3 (derives L1 from *, L3 & L2 from *, L3 => derives L1, L2 from *, L3 wih L2 = L3)
;;
;;      derives L1 from L1, L2, L3    & L2 from L1, L2, L3    => derives L1, L2 from L1, L2, L3
;;                                      L2 from *, L1, L2, L3 => derives L1, L2 from L1, L2, L3
;;
;;      derives L1 from *, L1, L2, L3 & L2 from L1, L2, L3    => derives L1, L2 from L1, L2, L3
;;                                      L2 from *, L1, L2, L3 => derives L1, L2 from L1, L2, L3

(defun spark-sort-no-case (el1 el2)
  "Sort SPARK variable list casing is ignored"
  (string< (upcase el1) (upcase el2)))

(defun spark-in-spark-derives-p ()
  "Return t if point is inside a SPARK derives annotation."
  (if (ada-in-spark-annotation-p)
      (let ((orgpoint (point)))
	(save-excursion
	  (ada-goto-stmt-start nil t)
	  (or (re-search-forward "derives\\>[ \t\n]" orgpoint t)
	      (save-excursion
		(backward-word 1)
		(re-search-forward "from\\>[ \t\n]" orgpoint t)))))
    nil))

(defun spark-compare-string-list (list1 list2)
  "Return TRUE if string list LIST1 and string list LIST2 are equal.
The comparison is not case sensitive"
  (let ((i (length list1))
	(j (length list2))
	(is-equal 't))
    (if (not (= i j))
	(setq is-equal nil)
      (while (and (not (zerop i))
		  is-equal)
	(setq i (1- i))
	(unless (eq t (compare-strings (nth i list1) 0 nil (nth i list2) 0 nil t))
	  (setq is-equal nil))))
    is-equal))

(defun spark-string-search (var list)
  "Return TRUE if the string VAR is in the list of string LIST"
  (not (eq (member-ignore-case var list) nil)))

(defun spark-delete-string (var list)
  "Delete all occurences of the string VAR in the list of string LIST"
  (let ((result-list nil)
	(i (length list)))
    (while (not (zerop i))
      (setq i (1- i))
      (if (not (string= (upcase var) (upcase (nth i list))))
	  (setq result-list (append (list (nth i list)) result-list))))
    result-list))

(defun spark-include (varlist1 varlist2)
  "Return TRUE if VARLIST1 is included in VARLIST2"
  (let ((i (length varlist1))
	(include 't))
    (while (and (not (zerop i))
		include)
      (setq i (1- i))
      (if (not (spark-string-search (nth i varlist1) varlist2))
	  (setq include nil)))
    include))

(defun spark-non-canonical-format (varlist from-varlist)
  "Return a list of all the equivalent lists of FROM-VARLIST
Assuming that element A is not in list L
Assuming that list L1 is not part of list L2

derives A from L

derives A from *, A, L <=>
derives A from A, L    <=>
derives A from *, L

derives L1 from L2

derives L1 from L1, L2 <=>
derives L1 from *, L1, L2"
  (if from-varlist
      (if (= (length varlist) 1)
	  (cond
	   ((and (not (spark-string-search (nth 0 varlist) from-varlist))
		 (not (spark-string-search "*" from-varlist)))
	    (list from-varlist))
	   ((and (not (spark-string-search (nth 0 varlist) from-varlist))
		 (spark-string-search "*" from-varlist))
	    (list (sort (copy-sequence (append (list (nth 0 varlist)) from-varlist)) 'spark-sort-no-case)
		  (sort (copy-sequence (append (list (nth 0 varlist)) (spark-delete-string "*" from-varlist))) 'spark-sort-no-case)
		  from-varlist))
	   ((and (spark-string-search (nth 0 varlist) from-varlist)
		 (not (spark-string-search "*" from-varlist)))
	    (list (sort (copy-sequence (append (list "*") from-varlist)) 'spark-sort-no-case)
		  from-varlist
		  (sort (copy-sequence (append (list "*") (spark-delete-string (nth 0 varlist) from-varlist))) 'spark-sort-no-case)))
	   ((and (spark-string-search (nth 0 varlist) from-varlist)
		 (spark-string-search "*" from-varlist))
	    (list from-varlist
		  (sort (copy-sequence (spark-delete-string "*" from-varlist)) 'spark-sort-no-case)
		  (sort (copy-sequence (spark-delete-string (nth 0 varlist) from-varlist)) 'spark-sort-no-case))))
	(if (spark-include varlist from-varlist)
	    (cond
	     ((spark-string-search "*" from-varlist)
	      (list from-varlist
		    (spark-delete-string "*" from-varlist)))
	     ((not (spark-string-search "*" from-varlist))
	      (list (sort (copy-sequence (append (list "*") from-varlist)) 'spark-sort-no-case)
		    from-varlist)))
	  (list from-varlist)))
    nil))

(defun spark-compare-from-varlist (varlist1 varlist2 from-varlist1 from-varlist2)
  "Compare 2 lists of variables to check if they are compatible.
Return 'nil' if they are not compatible or
the new lists of variables if they are compatible"
  (let ((from-varlist1-non-canonical (spark-non-canonical-format varlist1 from-varlist1))
	(from-varlist2-non-canonical (spark-non-canonical-format varlist2 from-varlist2))
	i
	j
	(found nil))
    (setq i (length from-varlist1-non-canonical))
    (while (and (not (zerop i))
		(not found))
      (setq i (1- i))
      (setq j (length from-varlist2-non-canonical))
      (while (and (not (zerop j))
		  (not found))
	(setq j (1- j))
	(if (spark-compare-string-list (nth i from-varlist1-non-canonical) (nth j from-varlist2-non-canonical))
	    (setq found (nth i from-varlist1-non-canonical)))))
    found))

(defun spark-derives-reformat (paramlist)
  "Reorganize and refactorize the lists of variables"
  (let ((i (length paramlist))
	j
	from-varlist
	continue
	(paramlist-reformatted nil))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))

      (setq j (length paramlist-reformatted))
      (setq continue 't)
      ;; loop until last parameter
      (while (and (not (zerop j))
		  continue)
	(setq j (1- j))
	(if (and (not (nth 1 (nth i paramlist)))
		 (not (nth 1 (nth j paramlist-reformatted))))
	    (progn (setq from-varlist nil)
		   (setq continue nil))
	  (setq from-varlist (spark-compare-from-varlist (nth 0 (nth i paramlist))
							 (nth 0 (nth j paramlist-reformatted))
							 (nth 1 (nth i paramlist))
							 (nth 1 (nth j paramlist-reformatted))))
	  (if from-varlist
	      (setq continue nil))))

      ;; add this parameter-declaration to the list
      (setq paramlist-reformatted (append paramlist-reformatted (list
								 (if continue
								     (nth i paramlist)
								   (list (sort (copy-sequence (append (nth 0 (nth i paramlist)) (nth 0 (nth j paramlist-reformatted)))) 'spark-sort-no-case)
									 from-varlist)))))
      (if (not continue)
	  (setq paramlist-reformatted (delete (nth j paramlist-reformatted) paramlist-reformatted))))
    (sort paramlist-reformatted 'spark-sort-derives-varlist)))

(defun spark-scan-derives (begin end)
  "Scan the SPARK derives annotation found in between BEGIN and END.
Return the equivalent internal parameter list."
  (let ((notend t)
	(apos nil)
	(epos nil)
	(fpos nil)
	(semipos nil)
	star
	(paramlist (list))
	derives-varlist
	derives-from-varlist)

    (goto-char begin)

    ;; loop until end of the SPARK derives annotation
    (while notend

      ;; find first character of parameter-declaration
      (ada-goto-next-non-ws nil nil t)
      (setq apos (point))

      ;; find last character of parameter-declaration
      (if (setq match-cons
		(ada-search-ignore-string-comment "from\\>" nil end nil nil t))
	  (setq fpos (car match-cons))
	(setq fpos end))

      (if (setq match-cons
		(ada-search-ignore-string-comment "[ \t\n]*[&;]" nil end nil nil t))
	  (progn
	    (setq epos (car match-cons))
	    (setq semipos (cdr match-cons)))
	(setq epos end))

      (goto-char apos)

      (setq derives-varlist nil)
      (while (progn (ada-goto-next-non-ws nil nil t)
		    (< (point) fpos))

	(looking-at "\\(\\sw\\|[_.]\\)+")
	(setq derives-varlist
	      (append derives-varlist
		      (list (match-string 0))))

	(ada-search-ignore-string-comment ",\\|from\\>" nil epos nil nil t))

      (setq derives-from-varlist nil)
      (setq star nil)
      (while (progn (ada-goto-next-non-ws nil nil t)
		    (< (point) epos))
      
	(looking-at "\\(\\sw\\|[_.\\*]\\)+")
	(if (and (eq (length derives-varlist) 1)
		 (or (string= (upcase (match-string 0)) (upcase (car derives-varlist)))
		     (string= (match-string 0) "*")))
	    (if (not star)
		(progn (setq star 't)
		       (setq derives-from-varlist
			     (append derives-from-varlist
				     (list "*")))))
	  (if (not (spark-string-search (match-string 0) derives-from-varlist))
	      (setq derives-from-varlist
		    (append derives-from-varlist
			    (list (match-string 0))))))

	(ada-search-ignore-string-comment "[ \t\n]*[,&]" nil epos nil nil t))

      (if (and (> (length derives-varlist) 1)
	       (spark-include derives-varlist derives-from-varlist)
	       (spark-string-search "*" derives-from-varlist))
	  (setq derives-from-varlist (spark-delete-string "*" derives-from-varlist)))

      ;; add this parameter-declaration to the list
      (setq paramlist (when derives-varlist
			(append paramlist (list (list (sort derives-varlist 'spark-sort-no-case)
						      (sort derives-from-varlist 'spark-sort-no-case))))))

      ;; check if it was the last parameter
      (if (eq epos end)
	  (setq notend nil)
	(goto-char semipos)))
    paramlist))

(defun spark-insert-derives (paramlist)
  "Insert a formatted PARAMLIST in the buffer."
  (let (i
	j
	derives-col
	fromcol
	derives-from-col
	derives-list-len
	paramlist-len
	(parlen 0))

    (setq i (length paramlist))
    (setq derives-list-len (1- i))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      
      (setq j (length (nth 0 (nth (- derives-list-len i) paramlist))))

      ;; loop until last parameter
      (while (not (zerop j))
	(setq j (1- j))
      
	;; get max length of parameter-name
	(setq parlen (max parlen (length (nth j (nth 0 (nth (- derives-list-len i) paramlist))))))))

    (setq derives-col (current-column))
    (setq fromcol (+ (current-column) parlen))
    (setq derives-from-col (+ fromcol 6))
    (setq i (length paramlist))

    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))

      (setq j (length (nth 0 (nth (- derives-list-len i) paramlist))))
      (setq paramlist-len (1- j))

      ;; loop until last parameter
      (while (not (zerop j))
	(setq j (1- j))
	(indent-to derives-col)
	(insert (nth (- paramlist-len j) (nth 0 (nth (- derives-list-len i) paramlist))))
	(if (not (zerop j))
	    (progn (insert ",")
		   (newline)
		   (insert ada-spark-annotation-start)
		   (backward-char 3)
		   (ada-indent-current)
		   (forward-char 3))))

      (indent-to fromcol)
      (insert " from ")

      (setq j (length (nth 1 (nth (- derives-list-len i) paramlist))))
      (setq paramlist-len (1- j))

      ;; loop until last parameter
      (while (not (zerop j))
	(setq j (1- j))
	(indent-to derives-from-col)
	(insert (nth (- paramlist-len j) (nth 1 (nth (- derives-list-len i) paramlist))))
	(if (not (zerop j))
	    (progn (insert ",")
		   (newline)
		   (insert ada-spark-annotation-start)
		   (backward-char 3)
		   (ada-indent-current)
		   (forward-char 3))))
      (if (zerop i)
	  (insert ";")
	(insert " &")
	(newline)
	(insert ada-spark-annotation-start)
	(backward-char 3)
	(ada-indent-current)
	(forward-char 3)))))

(defun spark-format-derives ()
  "Reformat the SPARK derives annotation point is in."
  (interactive)
  (let ((begin nil)
	(end nil)
	(paramlist nil)
	(paramlist-reformatted nil))

    ;; check if really inside a SPARK derives annotation
    (or (spark-in-spark-derives-p)
	(error "Not in SPARK derives annotation"))

    ;; find start of current SPARK derives annotation
    (ada-search-ignore-string-comment "derives\\>" t nil nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (forward-word 1)
    (setq begin (1+ (point)))
    (ada-goto-next-non-ws nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))

    ;; find end of the SPARK derives annotation
    (ada-search-ignore-string-comment ";" nil nil nil nil t)
    (delete-char -1)
    (insert "\n")
    (setq end (point))

    ;; build a list of all elements of the SPARK derives annotation
    (setq paramlist (spark-scan-derives begin (1- end)))

    ;; delete the original SPARK derives annotation
    (delete-region begin end)

    (when paramlist
      ;; reformat the SPARK derives list
      (setq paramlist-reformatted (spark-derives-reformat paramlist)))

    ;; insert the new SPARK derives list
    (goto-char begin)
    (if paramlist
	(spark-insert-derives paramlist-reformatted)
      (insert ";"))))

(defun spark-in-spark-inherit-p ()
  "Return t if point is inside a SPARK inherit annotation."
  (if (ada-in-spark-annotation-p)
      (let ((orgpoint (point)))
	(save-excursion
	  (ada-goto-stmt-start nil t)
	  (re-search-forward "inherit\\>[ \t\n]" orgpoint t)))
    nil))

(defun spark-format-inherit ()
  "Reformat SPARK inherit annotation point is in."
  (interactive)
  (let (i
	inheritlist-len
	(begin nil)
	(end nil)
	(inheritlist nil)
	(inheritlist-reformatted nil))

    ;; check if really inside a SPARK inherit annotation
    (or (spark-in-spark-inherit-p)
	(error "Not in SPARK inherit annotation"))

    ;; find start of current SPARK inherit annotation
    (ada-search-ignore-string-comment "inherit\\>" t nil nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (forward-word 1)
    (ada-goto-next-non-ws nil nil t)
    (save-excursion (beginning-of-line)
		    (ada-indent-current))
    (setq begin (point))

    ;; find end of the SPARK inherit annotation
    (ada-search-ignore-string-comment ";" nil nil nil nil t)
    (delete-char -1)
    (insert "\n")
    (setq end (point))

    ;; build a list of all elements of the SPARK inherit annotation
    (goto-char begin)
    (while (progn (ada-goto-next-non-ws nil nil t)
		  (< (point) end))
      (looking-at "\\(\\sw\\|[_.\\*]\\)+")
      (setq inheritlist (append inheritlist 
				(list (match-string 0))))
      (ada-search-ignore-string-comment "[ \t\n]*," nil end nil nil t))
    
    ;; delete the original SPARK inherit annotation
    (delete-region begin end)

    ;; reformat the SPARK inherit list
    (setq inheritlist-reformatted (sort inheritlist 'spark-sort-no-case))

    ;; insert the new SPARK inherit list
    (goto-char begin)
    (setq i (length inheritlist-reformatted))
    (setq inheritlist-len (1- i))
    ;; loop until last parameter
    (while (not (zerop i))
      (setq i (1- i))
      (insert (nth (- inheritlist-len i) inheritlist-reformatted))
      (if (zerop i)
	  (insert ";")
	(insert ",")
	(newline)
	(insert ada-spark-annotation-start)
	(backward-char 3)
	(ada-indent-current)
	(forward-char 3)
	(ada-indent-current)))))

(defun spark-current (&optional arg)
  "SPARK the current file."
  (interactive "P")
  (compile (concat ada-spark-examiner-command " -brief=fullpath " (file-name-nondirectory (buffer-file-name)))))

(defun spark-metafile-exists (filename)
  (let ((meta-filename (file-name-sans-extension filename)))
    (while (and (not (file-exists-p (concat meta-filename ".smf")))
		(string-match "^\\(.*\\)[.-][^.-]*" meta-filename))
      (setq meta-filename (match-string 1 meta-filename)))
    (if (file-exists-p (concat meta-filename ".smf"))
	(concat meta-filename ".smf")
      nil)))

(defun spark-current-metafile (&optional arg)
  "SPARK the current metafile."
  (interactive "P")
  (let* ((filename (buffer-file-name))
	 (meta-filename (spark-metafile-exists filename)))
    (when meta-filename
      (compile (concat ada-spark-examiner-command " -brief=fullpath @" (file-name-nondirectory meta-filename))))))

(defun spark-metafile (meta-filename &optional arg)
  "SPARK the metafile."
  (interactive "FSPARK Metafile: \np")
  (compile (concat ada-spark-examiner-command " -brief=fullpath @" (file-name-nondirectory meta-filename))))

(defun spark-pretty-print-reformat (re-pattern process)
  "Local function for spark-pretty-print"
  (let (search-pos)
    (goto-char (point-min))
    (while (re-search-forward re-pattern nil t)
      (forward-char 1)
      (setq search-pos (point))
      (funcall process)
      (goto-char search-pos))))

(defun spark-pretty-print ()
  "Pretty print Ada code and SPARK annotations"
  (interactive)
  (indent-region (point-min) (point-max))
  (spark-pretty-print-reformat "\\<\\(procedure\\|function\\)[ \t\n]+[a-zA-Z0-9_]+[ \t\n]+("
			       (lambda ()
				 (when (or (not (ada-in-string-or-comment-p))
					   (ada-in-spark-annotation-not-in-string-p))
				   (ada-format-paramlist))))
  (spark-pretty-print-reformat "([ \t]*[A-Za-z0-9][^()\"]*=>" 
			       (lambda ()
				 (when (or (not (ada-in-string-or-comment-p))
					   (ada-in-spark-annotation-not-in-string-p))
				   (ada-format-call-paramlist))))
  (when ada-spark-mode
    (spark-pretty-print-reformat (concat "^[ \t]*" ada-spark-annotation-start "[ \t]*inherit[ \t\n]") 'spark-format-inherit)
    (spark-pretty-print-reformat (concat "^[ \t]*" ada-spark-annotation-start "[ \t]*global[ \t\n]") 'spark-format-global)
    (spark-pretty-print-reformat (concat "^[ \t]*" ada-spark-annotation-start "[ \t]*derives[ \t\n]") 'spark-format-derives)))

(provide 'spark-mode)
