/* This Spaceshooter is an small space adventure game
 * Copyright (C) 2006,2007  Steffen Nörtershäuser
 * Copyright (C) 2008 Christoph Egger
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Weapon.hpp"
#include "Singletons.hpp"
#include "Globals.hpp"
#include "MathFuncs.hpp"

#include <cstdio>
#include <cmath>
#include <boost/format.hpp>
#include <libintl.h>
#define _(string) gettext(string)

CWeapon::CWeapon()
		: m_ProjectileSpeedX(0.0f),
		  m_ProjectileSpeedY(0.0f),
		  m_ProjectileAccelX(0.0f),
		  m_ProjectileAccelY(0.0f),
		  m_SpeedLimitX(-99999.0f),
		  m_SpeedLimitY(-99999.0f),
		  m_SpeedLimitXSmaller(false),
		  m_SpeedLimitYSmaller(false),
		  m_ShootSpeed(0),
		  m_LastShot(0),
		  m_Damage(0),
		  m_WeaponGroupID(0),
		  m_ProjectileAutoRotate(false),
		  m_InitiallyEquiped(false),
		  m_Equiped(false),
		  m_Foreground(false),
		  m_X(0),
		  m_Y(0),
		  log_(log4cxx::Logger::getLogger("CWeapon"))
{}

CWeapon::~CWeapon()
{
	for(std::size_t CurProjectil = 0; CurProjectil < m_Projectiles.size(); CurProjectil++)
	{
		delete m_Projectiles[CurProjectil];
	}
	m_Projectiles.clear();
}

void CWeapon::ReadData(const std::string& Name)
{
	FILE* ShipFile = std::fopen( ( std::string(DATADIR "/" PACKAGE_NAME "/") + Name ).c_str(),"r");

	if(ShipFile == NULL)
	{
		LOG4CXX_ERROR(log_, boost::format(_("Could not open Weaponfile \"%1%\"")) % Name);
		return;
	}

	char strBuf[100];
	int HelpInt;
	std::string HelpString;
	std::string OldString;
	std::string DataString;
	int ProjectileFrameCount = 1;
	int FrameWidth = -1;
	int FrameHeight = -1;
	int FrameDelay = 999999;

	while(std::feof(ShipFile) == 0)
	{
		std::fscanf(ShipFile,"%s",strBuf);
		HelpString = strBuf;

		if(HelpString == "=")
		{
			if(OldString == "WeaponName")
			{
				std::fscanf(ShipFile,"%s",strBuf);
				m_Name = strBuf;
			}

			if(OldString == "X")
			{
				std::fscanf(ShipFile,"%f",&m_X);
			}

			if(OldString == "Y")
			{
				std::fscanf(ShipFile,"%f",&m_Y);
			}

			if(OldString == "ProjectileSpeedX")
			{
				std::fscanf(ShipFile,"%f",&m_ProjectileSpeedX);
			}

			if(OldString == "ProjectileSpeedY")
			{
				std::fscanf(ShipFile,"%f",&m_ProjectileSpeedY);
			}

			if(OldString == "ProjectileAccelX")
			{
				std::fscanf(ShipFile,"%f",&m_ProjectileAccelX);
			}

			if(OldString == "ProjectileAccelY")
			{
				std::fscanf(ShipFile,"%f",&m_ProjectileAccelY);
			}
	
			
			if(OldString == "ProjectileSpeedLimitX")
			{
				std::fscanf(ShipFile,"%s",strBuf);
				std::fscanf(ShipFile,"%f",&m_SpeedLimitX);
				DataString = strBuf;

				m_SpeedLimitXSmaller = false;
				if(DataString.find("<") != std::string::npos)
				{
					m_SpeedLimitXSmaller = true;
				}
			}

			if(OldString == "ProjectileSpeedLimitY")
			{
				std::fscanf(ShipFile,"%s",strBuf);
				std::fscanf(ShipFile,"%f",&m_SpeedLimitY);
				DataString = strBuf;

				m_SpeedLimitYSmaller = false;
				if(DataString.find("<") != std::string::npos)
				{
					m_SpeedLimitYSmaller = true;
				}
			}

			if(OldString == "ProjectileImage")
			{
				std::fscanf(ShipFile,"%s",strBuf);
				m_ProjectileImageName = strBuf;
			}
			if(OldString == "ProjectileFrameCount")
			{
				std::fscanf(ShipFile,"%d",&ProjectileFrameCount);
			}
			if(OldString == "ProjectileFrameWidth")
			{
				std::fscanf(ShipFile,"%d",&FrameWidth);
			}
			if(OldString == "ProjectileFrameHeight")
			{
				std::fscanf(ShipFile,"%d",&FrameHeight);
			}
			if(OldString == "ProjectileFrameDelay")
			{
				std::fscanf(ShipFile,"%d",&FrameDelay);
			}
				
			if(OldString == "ShootSpeed")
			{
				std::fscanf(ShipFile,"%d",&m_ShootSpeed);
			}

			if(OldString == "Damage")
			{
				std::fscanf(ShipFile,"%d",&m_Damage);
			}

			if(OldString == "WeaponGroupID")
			{
				std::fscanf(ShipFile,"%d",&m_WeaponGroupID);
			}

			if(OldString == "AutoRotate")
			{
				std::fscanf(ShipFile,"%d",&HelpInt);
				if(HelpInt == 0)
				{
					m_ProjectileAutoRotate = false;
				}
				else
				{
					m_ProjectileAutoRotate = true;
				}
			}

	
			if(OldString == "CollisionSystem")
			{
				std::fscanf(ShipFile,"%s",strBuf);
				m_ProjectileData.CollisionFile = strBuf;
			}


			if(OldString == "ShootSound")
			{
				std::fscanf(ShipFile,"%s",strBuf);
				m_ShootSoundFile = strBuf;
				m_ShootSound.SetBuffer(sgl::get_sndbuffmgr().get_snd(m_ShootSoundFile));
			}

			if(OldString == "OnHitSound")
			{
				std::fscanf(ShipFile,"%s",strBuf);
				m_HitSoundFile = strBuf;
				m_HitSound.SetBuffer(sgl::get_sndbuffmgr().get_snd(m_HitSoundFile));
			}
			

			if(OldString == "InitiallyEquiped")
			{
				std::fscanf(ShipFile,"%d",&HelpInt);
				if(HelpInt == 0)
				{
					m_Equiped = false;
				}
				else
				{
					m_Equiped = true;
				}

				m_InitiallyEquiped = m_Equiped;
			}
	
			if(OldString == "Foreground")
			{
				std::fscanf(ShipFile,"%d",&HelpInt);
				if(HelpInt == 0)
				{
					m_Foreground = false;
				}
				else
				{
					m_Foreground = true;
				}
			}
		}

		OldString = HelpString;
	}

	m_ProjectileData.AccelX = m_ProjectileAccelX;
	m_ProjectileData.AccelY = m_ProjectileAccelY;

	m_ProjectileData.SpeedX = m_ProjectileSpeedX;
	m_ProjectileData.SpeedY = m_ProjectileSpeedY;

	m_ProjectileData.SpeedLimitX = m_SpeedLimitX;
	m_ProjectileData.SpeedLimitXSmaller = m_SpeedLimitXSmaller;

	m_ProjectileData.SpeedLimitY = m_SpeedLimitY;
	m_ProjectileData.SpeedLimitYSmaller = m_SpeedLimitYSmaller;

	m_ProjectileData.AutoRotate = m_ProjectileAutoRotate;
	
	m_ProjectileData.FileName = m_ProjectileImageName;
	m_ProjectileData.FrameCount = ProjectileFrameCount;
	m_ProjectileData.FrameWidth = FrameWidth;
	m_ProjectileData.FrameHeight = FrameHeight;
	m_ProjectileData.FrameDelay = FrameDelay;

	std::fclose(ShipFile);
}

void CWeapon::Shoot(float X, float Y)
{	
	if(m_Equiped == false || sgl::get_clock().GetElapsedTime()*1000 - m_LastShot < m_ShootSpeed)
	{
		return;
	}

	m_ShootSound.Play();

	m_LastShot = sgl::get_clock().GetElapsedTime()*1000;

	m_ProjectileData.X = X + m_X;
	m_ProjectileData.Y = Y + m_Y;

	CProjectile* Projectile = new CProjectile;
	Projectile->Init(m_ProjectileData);

	m_Projectiles.push_back(Projectile);
}

void CWeapon::DrawProjectiles()
{
	for(std::size_t CurProjectil = 0; CurProjectil < m_Projectiles.size(); CurProjectil++)
	{
		m_Projectiles[CurProjectil]->Draw();
		m_Projectiles[CurProjectil]->Update();

		if(m_Projectiles[CurProjectil]->GetXPosition() > (SCREEN_X_SIZE - HUD_SIZE_X) + 50 ||
		   m_Projectiles[CurProjectil]->GetXPosition() < -50 ||
		   m_Projectiles[CurProjectil]->GetYPosition() > SCREEN_Y_SIZE + 50 ||
		   m_Projectiles[CurProjectil]->GetYPosition() < -50)
		{
			delete m_Projectiles[CurProjectil];
			m_Projectiles.erase(m_Projectiles.begin() + CurProjectil);
		}
	}	
}

int CWeapon::CheckCollision(const CCollisionSystem* CounterPart)
{
	const CCollisionSystem* CollisionSystem;
	int Damage = 0;

	for(std::size_t CurProjectil = 0; CurProjectil < m_Projectiles.size(); CurProjectil++)
	{
		CollisionSystem = m_Projectiles[CurProjectil]->GetCollisionSystem();
		if(CollisionSystem->CheckCollision(CounterPart) == true)
		{
			delete m_Projectiles[CurProjectil];
			m_Projectiles.erase(m_Projectiles.begin() + CurProjectil);
			Damage += m_Damage;

			m_HitSound.Play();
		}
	}

	return Damage;
}


void CWeapon::ClearProjectils()
{
	for(std::size_t CurProjectil = 0; CurProjectil < m_Projectiles.size(); CurProjectil++)
	{
		delete m_Projectiles[CurProjectil];
	}
	m_Projectiles.clear();
}


int CWeapon::GetProjectilXRel(int YDistance) const
{
	float Time = 0.0f;
	int TempXAdd = 0;

	if(m_ProjectileSpeedY == 0.0f)
	{
		Time = std::sqrt(2.0f*(YDistance + m_Y) / m_ProjectileAccelY);
	}
	else
	{
		if(sgn(m_ProjectileAccelY) != sgn(m_ProjectileSpeedY))
		{
			float TempTime = std::fabs(m_ProjectileSpeedY / m_ProjectileAccelY);

			TempXAdd = 0.5f*m_ProjectileAccelX*TempTime*TempTime + m_ProjectileSpeedX*TempTime;	
		}
		
		if(m_ProjectileAccelY != 0.0f)
		{
			//0 = m_ProjectileAccelY*t^2 + m_ProjectileSpeedY*t + (m_Y + YDistance)
			float Dis = m_ProjectileSpeedY*m_ProjectileSpeedY - 4.0f*m_ProjectileAccelY*std::fabs(m_Y - YDistance);
	
			Time = std::fabs((m_ProjectileSpeedY + std::sqrt(Dis)) / (2.0f*m_ProjectileAccelY));

			if(m_SpeedLimitY != -99999.0f)
			{
				float TimeVel = std::fabs((m_SpeedLimitY - m_ProjectileSpeedY) / m_ProjectileAccelY);

				if(TimeVel < Time)
				{
					int NewY = 0.5f*m_ProjectileAccelY*TimeVel*TimeVel + m_ProjectileSpeedY*TimeVel;
					Time = TimeVel + std::fabs(((float)NewY) / m_ProjectileSpeedY);
				}
			}
		}
		else
		{
			Time = std::fabs(((float)YDistance + m_Y) / m_ProjectileSpeedY);
		}
	}

	int XReturn = 0.5f*m_ProjectileAccelX*Time*Time + m_ProjectileSpeedX*Time + m_X + TempXAdd;

	if(m_SpeedLimitX != -99999.0f)
	{
		float TimeVel = std::fabs((m_SpeedLimitX - m_ProjectileSpeedX) / m_ProjectileAccelX);

		XReturn = 0.5f*m_ProjectileAccelX*TimeVel*TimeVel + m_ProjectileSpeedX*TimeVel + 
				  m_SpeedLimitX*(Time-TimeVel) + m_X;
	}

	return XReturn;
}

