/* This Spaceshooter is an small space adventure game
 * Copyright (C) 2006,2007,2008  Christoph Egger
 * Copyright (C) 2006,2007 Steffen Nörtershäuser
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Ship.hpp"
#include "Singletons.hpp"
#include "Starfield.hpp"
#include "Player.hpp"
#include "PowerUp.hpp"
#include "EnemyManager.hpp"
#include "Enemy_Help.hpp"
#include "Kampagne.hpp"
#include "Mission.hpp"
#include "Bitmapfont.hpp"
#include "HUD-Manager.hpp"
#include "Menu.hpp"
#include "Briefing.hpp"
#include "CreditsManager.hpp"
#include "Init.hpp"

#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <clocale>
#include <cmath>
#include <vector>
#include <list>

#include <SFML/System.hpp>
#include <SFML/Window.hpp>
#include <SFML/Window/Event.hpp>

#include <log4cxx/logger.h>
#include <log4cxx/propertyconfigurator.h>

#include <libintl.h>
#define _(string) gettext(string)

enum GameStatus
{
	GAMESTATE_MENU,
	GAMESTATE_PLAY,
	GAMESTATE_BRIEF,
	GAMESTATE_CREDITS
};

CStarfield Starfield;
CPlayer Player;
CEnemyManager EnemyManager;
CKampagne Kampagne;
CBitmapFont BitMapFont;
CHUDManager HUD;
CMenu Menu;
CBriefing Briefing;
CCreditsManager g_CreditsManager;
AHUDPoll * LiveHUD;

std::list<SInfo*> g_Infos;
bool g_Info_On;

std::string nowplay;


std::string Weapon1;
std::string Weapon2;

global_data_pointers datptrs;

int g_MouseX;
int g_MouseY;
bool g_MouseDown;


bool g_IsPlaying = false;

bool g_FirstBrief = false;

void ResetAll()
{
	Player.GetShip()->SetCurArmor(Player.GetShip()->GetMaxArmor());
	Player.GetShip()->SetPosition((SCREEN_X_SIZE - HUD_SIZE_X)/2,SCREEN_Y_SIZE - 100);
	Player.GetShip()->ClearProjectils();
	EnemyManager.clearPowerups();
	Starfield.RemovePlanets();
}

bool StartCampaign(std::string KampagneName, GameStatus& state)
{
	static log4cxx::LoggerPtr log_ (log4cxx::Logger::getLogger("main.StartCampaign"));
	
	Kampagne.init(KampagneName);

	ResetAll();
	Player.ResetWeapons();

	Briefing.SetContent(Kampagne.get_Title(), Kampagne.get_Intro(), Kampagne.get_ImgFile(), BRIEF_BUTTON_START, Kampagne.getFramecount(), Kampagne.getFramedelay());
	Briefing.SetCampaign(KampagneName);
	Briefing.SetCurMission(0);

	EnemyManager.reset();

    /// \todo SoundManager.LoadPlaylist(Kampagne.get_Mission()->getNormalPlaylist().c_str());

	if (!EnemyManager.init(Kampagne.get_Mission()->get_ai_wave(), Kampagne.get_Mission()->get_ai_params(), datptrs, &Player, &g_Info_On, &g_Infos))
	{
        LOG4CXX_ERROR(log_, _("Error while initialising the Enemy Manager"));
		return false;
	}

	sgl::get_clock().Reset();

	Player.Init(Kampagne.get_Mission()->getPlayerShip());
	Player.SetLives(Kampagne.getLeben());
	LiveHUD->init(Player.GetShip()->GetMaxArmor(), Player.GetShip()->GetCurrentArmor());

	state = GAMESTATE_BRIEF;

	g_IsPlaying = true;
	g_FirstBrief = true;

	return true;
}

bool LoadSave(std::string SaveName, GameStatus& state)
{
	static log4cxx::LoggerPtr log_ (log4cxx::Logger::getLogger("main.LoadSave"));

	CSaveManager Manager;
	SaveData* Data = Manager.ReadSave(SaveName);

	if(Data == NULL)
	{
		return false;
	}

	StartCampaign(Data->KampagneName, state);
	Kampagne.set_MissionIndex(Data->Mission);

	EnemyManager.reset();
	if (!EnemyManager.init(Kampagne.get_Mission()->get_ai_wave(), Kampagne.get_Mission()->get_ai_params(), datptrs, &Player, &g_Info_On, &g_Infos))
	{
		LOG4CXX_ERROR(log_, _("Error while initialising the Enemy Manager"));
		return false;
	}

	if(Kampagne.get_Mission()->getPlayerShip() != "")
	{
		Player.Init(Kampagne.get_Mission()->getPlayerShip());
		LiveHUD->init(Player.GetShip()->GetMaxArmor(), Player.GetShip()->GetCurrentArmor());
	}
	Player.SetLives(Data->Lives);

	Briefing.SetCurMission(Data->Mission);
	Briefing.SetContent(_("New Mission!"),Kampagne.get_Mission()->get_Beschreibung(),"",BRIEF_BUTTON_START);

	return true;
}




// Temporäre Lösung
void init()
{
	g_Info_On = false;

	std::srand(42); ///< Truely random!

	HUD.init(datptrs, &BitMapFont);

	BitMapFont.Init();

	Starfield.Init(/*DATADIR "/" PACKAGE_NAME*/ "/background/BackgroundData.txt");

	Player.Init(/*DATADIR "/" PACKAGE_NAME*/ "/ships/Bummer.txt");

	HUD.registerHUD(&g_Info_On, &g_Infos, &BitMapFont);

	LiveHUD = (AHUDPoll*)HUD.registerHUD(Player.GetShip()->GetMaxArmor(), Player.GetShip()->GetCurrentArmor(), POLL_GRAFIK_HORZ, 884, 20, 130, 20);

	HUD.registerHUD(&Weapon1, POLL_TEXT, 880, 80, &BitMapFont, 15);

	HUD.registerHUD(&Weapon2, POLL_TEXT, 880, 140, &BitMapFont, 15, 138);

	HUD.registerHUD(Player.GetLiveString(), POLL_TEXT, 880, 200, &BitMapFont, 15);

	HUD.registerHUD(&nowplay, POLL_TEXT, 880, 600, &BitMapFont);

	Menu.Init(&BitMapFont, &g_MouseX, &g_MouseY, &g_MouseDown);
	Briefing.Init(&BitMapFont, &g_MouseX, &g_MouseY, &g_MouseDown, &Player);
	Briefing.SetContent(Kampagne.get_Title(), Kampagne.get_Intro(),Kampagne.get_ImgFile());

	/// \todo SoundManager.LoadPlaylist("Playlists/Test.txt");

	/// \todo SoundManager.setHUDString(&nowplay);
}

void play(GameStatus& state, log4cxx::LoggerPtr& log_)
{
	Starfield.DrawBack();
	Weapon1 = Player.GetCurWeaponNameOfType(0);
	Weapon2 = Player.GetCurWeaponNameOfType(1);
                
	EnemyManager.update();

	Player.Draw();
	Player.Update();
	Starfield.DrawFront();

	if(Player.CanBeRemoved() == true)
	{
		state = GAMESTATE_BRIEF;

		Briefing.SetContent(_("Defeat"),Kampagne.get_Niederlage(),"",BRIEF_BUTTON_CANCEL);
	}

	if (EnemyManager.hasWon())
	{
		if (Kampagne.nextMission())
		{
			EnemyManager.reset();
			if (!EnemyManager.init(Kampagne.get_Mission()->get_ai_wave(), Kampagne.get_Mission()->get_ai_params(), datptrs, &Player, &g_Info_On, &g_Infos))
			{
				LOG4CXX_ERROR(log_, _("Error while initialising the Enemy Manager"));
				std::exit(0);
			}
			sgl::get_clock().Reset();

			if(Kampagne.get_Mission()->getNormalPlaylist() != "")
			{
				/** \todo SoundManager.FadeOutAllBackgroundMusic();
				  * \todo SoundManager.LoadPlaylist(Kampagne.get_Mission()->getNormalPlaylist());
				  */
			}

			if(Kampagne.get_Mission()->getPlayerShip() != "")
			{
				Player.Init(Kampagne.get_Mission()->getPlayerShip());
				LiveHUD->init(Player.GetShip()->GetMaxArmor(), Player.GetShip()->GetCurrentArmor());
			}

			state = GAMESTATE_BRIEF;

			Starfield.RemovePlanets();

			Briefing.IncCurMission();
			Briefing.SetContent(Kampagne.get_Mission()->get_Title(),Kampagne.get_Mission()->get_Beschreibung(),Kampagne.get_Mission()->getImage(),
								BRIEF_BUTTON_START, Kampagne.get_Mission()->getFramecount(), Kampagne.get_Mission()->getFramedelay());
		}
		else
		{
			state = GAMESTATE_BRIEF;

			Briefing.SetContent(_("Victory"),Kampagne.get_Epilog(),"",BRIEF_BUTTON_FINISH);
		}
	}

	HUD.draw();
}

void update_world(GameStatus& state)
{
	static log4cxx::LoggerPtr log_ (log4cxx::Logger::getLogger("main.update_world"));
	
	std::string KampageName = "";

	
	switch (state)
	{
		case GAMESTATE_MENU:
			KampageName = Menu.Update();

			switch (Menu.Close())
			{
				case MENU_CLOSE_START:
					if(StartCampaign(KampageName, state) == false)
					{
						std::exit(0);
					}
					break;
				case MENU_CLOSE_LOAD:
					if(LoadSave(KampageName, state) == false)
					{
						std::exit(0);
					}
					break;
				case MENU_CLOSE_CREDITS:
					g_CreditsManager.Init(&BitMapFont);
					Starfield.SetSpawnPlanets(false);
					Starfield.SetWidth(SCREEN_X_SIZE);
					state = GAMESTATE_CREDITS;
					break;
			}
			break;
		case GAMESTATE_BRIEF:
			Briefing.Update();

			if(Briefing.Close() == true)
			{
				if(Briefing.GetButtonType() == BRIEF_BUTTON_CANCEL || Briefing.GetButtonType() == BRIEF_BUTTON_FINISH)
				{
					state = GAMESTATE_MENU;
					g_IsPlaying = false;
				}
				else
				{
					if(g_FirstBrief == true)
					{
						CMission* mis = Kampagne.get_Mission();
						Briefing.SetContent(mis->get_Title(),
											mis->get_Beschreibung(),
											mis->getImage(),
											BRIEF_BUTTON_START,
											mis->getFramecount(),
											mis->getFramedelay());
						
						state = GAMESTATE_BRIEF;
						g_FirstBrief = false;
					}
					else
					{
						state = GAMESTATE_PLAY;
						ResetAll();
					}
				}
				sgl::get_clock().Reset();
			}
			break;
		case GAMESTATE_PLAY:
			play(state, log_);
			break;
		case GAMESTATE_CREDITS:
			Starfield.DrawBack();
			g_CreditsManager.Draw();
			Starfield.DrawFront();

			if(g_CreditsManager.IsFinished())
			{
				Menu.SetSubMenu(MENU_SUB_START);
				state = GAMESTATE_MENU;
				Starfield.SetWidth(SCREEN_X_SIZE - HUD_SIZE_X);
				Starfield.SetSpawnPlanets(true);
			}
			break;
	}
}
// Temporäre Lösung ende

int main(int argc, char** argv)
{
	/* GNU Gettext */
	std::setlocale(LC_ALL, "");
	std::setlocale(LC_NUMERIC, "C");
	textdomain ("sfml_space");
	bindtextdomain ("sfml_space", "l10n");
	
	log4cxx::PropertyConfigurator::configure(DATADIR "/" PACKAGE_NAME "/logging.conf");
	
	log4cxx::LoggerPtr log_ (log4cxx::Logger::getLogger("main"));
	
	init();
	
	GameStatus cur_game_state(GAMESTATE_MENU);
	
	while (true)
	{
		sf::Event event;
		while (sgl::get_window().GetEvent(event))
		{
			switch (event.Type)
			{
				case sf::Event::Closed:
					std::exit(0);
					break;
				case sf::Event::KeyPressed:
					if (event.Key.Code == sf::Key::Escape)
					{
						if(cur_game_state == GAMESTATE_MENU)
						{
							sgl::get_clock().Reset();
							cur_game_state = GAMESTATE_PLAY;
						}
						else
						{
							Menu.SetSubMenu(MENU_SUB_START);
							cur_game_state = GAMESTATE_MENU;
							Starfield.SetWidth(SCREEN_X_SIZE - HUD_SIZE_X);
							Starfield.SetSpawnPlanets(true);
						}
					}
					break;
				case sf::Event::MouseMoved:
					g_MouseX = event.MouseMove.X;
					g_MouseY = event.MouseMove.Y;
					break;
				case sf::Event::MouseButtonPressed:
					if (event.MouseButton.Button == sf::Mouse::Left) g_MouseDown = true;
					break;
				case sf::Event::MouseButtonReleased:
					if (event.MouseButton.Button == sf::Mouse::Left) g_MouseDown = false;
					break;
			}
		}
		sgl::get_window().Clear();
		
		update_world(cur_game_state);
		
		sgl::get_window().Display();
	}
}
