/* This Spaceshooter is an small space adventure game
 * Copyright (C) 2006,2007  Steffen Nörtershäuser
 * Copyright (C) 2008 Christoph Egger
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Starfield.hpp"
#include "MathFuncs.hpp"
#include "Singletons.hpp"

#include <cstdio>
#include <cassert>

CStarfield::CStarfield()
		: m_SpawnPlanets(true),
		  m_PlanetSpawnTime(0),
		  m_Width(SCREEN_X_SIZE - HUD_SIZE_X)
{}

CStarfield::~CStarfield()
{
	CleanUp();
}

void CStarfield::CleanUp()
{
	std::list<CStar*>::iterator ListIterator;

	ListIterator = m_StarsBack.begin();
	while(ListIterator != m_StarsBack.end())
	{
		delete (*ListIterator);

		++ListIterator;
	}
	m_StarsBack.clear();

	ListIterator = m_StarsFront.begin();
	while(ListIterator != m_StarsFront.end())
	{
		delete (*ListIterator);

		++ListIterator;
	}
	m_StarsFront.clear();

	m_AsteroidData.clear();
}


void CStarfield::Init(const std::string& FileName)
{
	CleanUp();

	int Layer = 0;

	int StarCount = 200;
	std::string StarName = "data/ackground/Star.png";
	ReadData(FileName, &StarName, &StarCount);

	for(unsigned CurStar = 0; CurStar < StarCount; CurStar++)
	{
		Layer = std::rand() % 128;

		CStar* Star = new CStar;
		Star->Init(StarName, Layer, false, m_Width);

		if(Layer < STARFIELD_STAR_FRONT)
		{
			InsertStarBack(Star);
		}
		else
		{
			InsertStarFront(Star);
		}
	}
}

void CStarfield::ReadData(std::string FileName, std::string* StarFile, int* StarCount)
{
	//Diese Funktion liest die Sternendaten aus der angeben Datei aus

	if(StarFile == NULL || StarCount == NULL)
	{
		return;
	}

	FILE* BackFile = std::fopen(FileName.c_str(),"r");
	if(BackFile == NULL)
	{
		return;
	}

	char strBuf[100];
	std::string HelpString;
	std::string OldString;
	std::string DataString;

	int AsteroidFrameX = 0;
	int AsteroidFrameY = 0;
	int AsteroidFrameCount = 0;

	//Alle Daten auslesen
	while(std::feof(BackFile) == 0)
	{
		std::fscanf(BackFile,"%s",strBuf);
		HelpString = strBuf;

		if(HelpString == "=")
		{
			if(OldString == "Star")
			{
				std::fscanf(BackFile,"%s",strBuf);
				(*StarFile) = strBuf;
			}
			else if(OldString == "StarCount")
			{
				std::fscanf(BackFile,"%d",StarCount);
			}	

			else if(OldString == "Planet")
			{
				std::fscanf(BackFile,"%s",strBuf);
				DataString = strBuf;
				m_PlanetFiles.push_back(DataString);
			}	

			else if(OldString == "AsteroidFrameX")
			{
				std::fscanf(BackFile,"%d",&AsteroidFrameX);
			}
			else if(OldString == "AsteroidFrameY")
			{
				std::fscanf(BackFile,"%d",&AsteroidFrameY);
			}
			else if(OldString == "AsteroidFrameCount")
			{
				std::fscanf(BackFile,"%d",&AsteroidFrameCount);
			}

			else if(OldString == "Asteroid")
			{
				AsteroidData* Data = new AsteroidData;
				Data->FrameX = AsteroidFrameX;
				Data->FrameY = AsteroidFrameY;
				Data->FrameCount = AsteroidFrameCount;

				std::fscanf(BackFile,"%s",strBuf);
				DataString = strBuf;
				Data->FileName = DataString;

				m_AsteroidData.push_back(Data);
			}
		}

		OldString = HelpString;
	}

	std::fclose(BackFile);
}

void CStarfield::DrawBack()
{
	std::list<CStar*>::iterator ListIterator;

	ListIterator = m_StarsBack.begin();
	while(ListIterator != m_StarsBack.end())
	{
		assert (*ListIterator);
		(*ListIterator)->Draw();
	
		if((*ListIterator)->CanBeRemoved() == true)
		{
			delete (*ListIterator);
			m_StarsBack.erase(ListIterator++);
		}
		else
		{
			++ListIterator;
		}
	}
	if(m_SpawnPlanets == true)
	{
		CheckForNewPlanet();
		CheckForNewAsteroids();
	}
}

void CStarfield::DrawFront()
{
	std::list<CStar*>::iterator ListIterator;

	ListIterator = m_StarsFront.begin();
	while(ListIterator != m_StarsFront.end())
	{
		(*ListIterator)->Draw();
	
		if((*ListIterator)->CanBeRemoved() == true)
		{
			delete (*ListIterator);
			m_StarsFront.erase(ListIterator++);
		}
		else
		{
			++ListIterator;
		}
	}

}

void CStarfield::InsertStarBack(CStar* Star)
{
	std::list<CStar*>::iterator ListIterator;

	int Index = -1;
	int CurIndex = 0;
	ListIterator = m_StarsBack.begin();

	while(ListIterator != m_StarsBack.end())
	{
		if((*ListIterator)->GetLayer() > Star->GetLayer())
		{
			Index = CurIndex;
			break;
		}

		ListIterator++;
		CurIndex++;
	}

	if(Index == -1)
	{
		m_StarsBack.push_back(Star);
	}
	else if(Index == 0)
	{
		m_StarsBack.push_front(Star);
	}
	else
	{
		m_StarsBack.insert(ListIterator, Star);
	}
}

void CStarfield::InsertStarFront(CStar* Star)
{
	std::list<CStar*>::iterator ListIterator;

	int Index = -1;
	int CurIndex = 0;
	ListIterator = m_StarsFront.begin();

	while(ListIterator != m_StarsFront.end())
	{
		if((*ListIterator)->GetLayer() > Star->GetLayer())
		{
			Index = CurIndex;
			break;
		}

		++ListIterator;
		++CurIndex;
	}

	if(Index == -1)
	{
		m_StarsFront.push_back(Star);
	}
	else if(Index == 0)
	{
		m_StarsFront.push_front(Star);
	}
	else
	{
		m_StarsFront.insert(ListIterator, Star);
	}	
}


void CStarfield::RemovePlanets()
{
	std::list<CStar*>::iterator ListIterator;

	ListIterator = m_StarsBack.begin();
	while(ListIterator != m_StarsBack.end())
	{
		if((*ListIterator)->IsPlanet() == true)
		{
			delete (*ListIterator);
			m_StarsBack.erase(ListIterator++);
		}
		else
		{
			++ListIterator;
		}
	}
}

void CStarfield::CheckForNewPlanet()
{
	if(m_PlanetFiles.size() == 0 || STARFIELD_PLANET_MAXCHANCE == 0 || 
			sgl::get_clock().GetElapsedTime()*1000 - m_PlanetSpawnTime < STARFIELD_PLANET_DELAY)
	{
		return;
	}

	int CurChance = (std::rand() % STARFIELD_PLANET_MAXCHANCE);

	if(CurChance <= STARFIELD_PLANET_CHANCE)
	{
		int FileIndex = std::rand() % m_PlanetFiles.size();
		int Layer = std::rand() % 20;

		if(m_PlanetFiles.size() > 1)
		{
			while(FileIndex == m_LastPlanet)
			{
				FileIndex = std::rand() % m_PlanetFiles.size();
			}
		}

		m_LastPlanet = FileIndex;

		m_PlanetSpawnTime = sgl::get_clock().GetElapsedTime()*1000;

		CStar* Star = new CStar;
		Star->Init(m_PlanetFiles[FileIndex], Layer, true, m_Width);

		InsertStarBack(Star);
	}
}


void CStarfield::CheckForNewAsteroids()
{
	if(m_AsteroidData.size() == 0 || STARFIELD_ASTEROID_MAXCHANCE == 0)
	{
		return;
	}

	int CurChance = (std::rand() % STARFIELD_ASTEROID_MAXCHANCE);

	if(CurChance <= STARFIELD_ASTEROID_CHANCE)
	{

		int Dif = STARFIELD_ASTEROID_MAXSIZE - STARFIELD_ASTEROID_MINSIZE;
		int Count = STARFIELD_ASTEROID_MAXSIZE;
		if(Dif != 0)
		{	
			Count = STARFIELD_ASTEROID_MINSIZE + (rand() % (Dif + 1));
		}

		int SgnX = -1;
		if(std::rand() % 100 < 50)
		{
			SgnX = 1;
		}

		int SgnY = 1;

		float XSpeed = frnd(ASTEROID_MIN_SPEED * SgnX,ASTEROID_MAX_SPEED * SgnX);
		float YSpeed = frnd(ASTEROID_MIN_SPEED * SgnY,ASTEROID_MAX_SPEED * SgnY);

		float X = frnd(0.0f, SCREEN_X_SIZE);
		X -= STAR_ASTEROID_RADIUS * sgn(XSpeed);
		float Y = frnd(0.0f, -STAR_ASTEROID_RADIUS -150.0f * sgn(YSpeed));
		Y -= STAR_ASTEROID_RADIUS * sgn(YSpeed);

		if(sgn(Y) > 0)
		{
			Y += SCREEN_Y_SIZE;
		}

		int CurAsteroid = 0;
		for(CurAsteroid = 0; CurAsteroid < Count; CurAsteroid++)
		{
			int FileIndex = std::rand() % m_AsteroidData.size();
			int Layer = rand() % 20 + 21;

			CStar* Star = new CStar;
			Star->InitAsteroid(m_AsteroidData[FileIndex], Layer, XSpeed, YSpeed, X, Y);

			InsertStarBack(Star);
		}
	}
}


void CStarfield::SetWidth(int Width)
{
	std::list<CStar*>::iterator ListIterator;

	ListIterator = m_StarsBack.begin();
	while(ListIterator != m_StarsBack.end())
	{
		(*ListIterator)->SetFieldWidth(Width);
		(*ListIterator)->SetPosition(rand() % Width, (rand() % (SCREEN_Y_SIZE + 100) - 100));

		++ListIterator;
	}
}

