/* This Spaceshooter is an small space adventure game
 * Copyright (C) 2006,2007  Steffen Nörtershäuser
 * Copyright (C) 2008 Christoph Egger
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

//////////////////////////////////////////////////////////////////////////////////////////////////
//Ship.cpp - Defintion der Klasse CShip
//CShip Klasse die für das Verwalten der Raumschiffe benuzt
//Steffen Nörtershäuser
//////////////////////////////////////////////////////////////////////////////////////////////////


////////////////////////////////Makros und Includes///////////////////////////////////////////////
//Schutz vor Mehrfach Deklaration
#ifndef SHIP_CPP
#define SHIP_CPP

#include "Ship.hpp"
#include "UpdateFrame.hpp"

#include <boost/format.hpp>

#include <libintl.h>
#define _(string) gettext(string)

//////////////////////////////////////////////////////////////////////////////////////////////////

CShip::CShip()
		: m_CurArmor(0)
		, m_Armor(0)
		, m_Speed(0.0f)
		, m_AniDelay(9999)
		, m_DeadEmiterTime(0)
		, m_DeadTime(0)
		, m_Invulnerable(0)
		, log_(log4cxx::Logger::getLogger("CShip"))
		, frame_count(1)
{}

CShip::~CShip()
{
	CleanUp();
}

/// Diese Funktion gibt alles frei
void CShip::CleanUp()
{
	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		delete m_Weapons[CurWeapon];
	}
	m_Weapons.clear();
}

void CShip::Init(const std::string& FileName)
{
	CleanUp();

	char strBuf[100];
	std::string HelpString;
	std::string OldString;
	std::string DataString;
	std::string ImageFile;
	int FrameWidth  = -1;
	int FrameHeight = -1;

	FILE* ShipFile = fopen(FileName.c_str(),"r");
	if(ShipFile == NULL)
	{
		LOG4CXX_ERROR(log_, boost::format(_("Could not open Shipfile \"%1%\"")) % FileName);
		return;
	}

	//Alle Daten auslesen
	while(feof(ShipFile) == 0)
	{
		fscanf(ShipFile,"%s",strBuf);
		HelpString = strBuf;

		if(HelpString == "=")
		{

			if(OldString == "Image")
			{
				fscanf(ShipFile,"%s",strBuf);
				ImageFile = strBuf;
			}	

			if(OldString == "FrameWidth")
			{
				fscanf(ShipFile,"%d",&FrameWidth);
			}	
			else if(OldString == "FrameHeight")
			{
				fscanf(ShipFile,"%d",&FrameHeight);
			}	
			else if(OldString == "FrameCount")
			{
				fscanf(ShipFile,"%d",&frame_count);
			}	
			else if(OldString == "FrameDelay")
			{
				fscanf(ShipFile,"%d",&m_AniDelay);
			}
			
			else if(OldString == "Armor")
			{
				fscanf(ShipFile,"%d",&m_Armor);
				m_CurArmor = m_Armor;
			}
			else if(OldString == "Speed")
			{
				fscanf(ShipFile,"%f",&m_Speed);
			}

			if(OldString == "EngineEmiter")
			{			
				fscanf(ShipFile,"%s",strBuf);
				HelpString = strBuf;
				m_EngineEmiter.InitFromFile(HelpString);
			}
			else if(OldString == "EngineX")
			{
				fscanf(ShipFile,"%f",&m_EngineX);
			}
			else if(OldString == "EngineY")
			{
				fscanf(ShipFile,"%f",&m_EngineY);
			}	

			if(OldString == "OnDeadSound")
			{
				fscanf(ShipFile,"%s",strBuf);
				m_DeadSoundFile = strBuf;
				m_DeadSound.SetBuffer(sgl::get_sndbuffmgr().get_snd(m_DeadSoundFile));
				m_DeadSoundFile = strBuf;
			}

			if(OldString == "OnDeadEmiter")
			{
				fscanf(ShipFile,"%s",strBuf);
				HelpString = strBuf;
				m_DeadEmiter.InitFromFile(HelpString);
			}
	
			if(OldString == "DeadEmiterTime")
			{
				fscanf(ShipFile,"%d",&m_DeadEmiterTime);
			}
			
			if(OldString == "CollisionSystem")
			{
				fscanf(ShipFile,"%s",strBuf);
				DataString = strBuf;
				m_CollisionSystem.Init(DataString);
			}

			if(OldString == "WeaponFile")
			{
				fscanf(ShipFile,"%s",strBuf);
				HelpString = strBuf;

				CWeapon* Weapon = new CWeapon;
				Weapon->Init(HelpString);

				m_Weapons.push_back(Weapon);
			}

		}

		OldString = HelpString;
	}

	m_Sprite.SetImage(sgl::get_imagemgr().get_img(ImageFile));
	m_Sprite.SetSubRect(sf::IntRect(0, 0, FrameWidth, FrameHeight));

	fclose(ShipFile);
}



void CShip::Draw()
{
	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(m_Weapons[CurWeapon]->IsForeground() == false)
		{
			m_Weapons[CurWeapon]->DrawProjectiles();
		}
	}

	if(sgl::get_clock().GetElapsedTime()*1000 - m_LastAniUpdate > m_AniDelay)
	{
		update_frame(m_Sprite, frame_count);
		m_LastAniUpdate = sgl::get_clock().GetElapsedTime()*1000;
	}

	if(IsDead() == false)
	{
		sgl::get_window().Draw(m_Sprite);
		m_CollisionSystem.Draw();
		m_CollisionSystem.SetPosition(m_Sprite.GetPosition().x + m_Sprite.GetSize().x/2, m_Sprite.GetPosition().y + m_Sprite.GetSize().y/2);
	}
	else
	{
		m_EngineEmiter.Activate(false);

		if(sgl::get_clock().GetElapsedTime()*1000 - m_DeadTime > m_DeadEmiterTime)
		{
			m_DeadEmiter.Activate(false);
		}

		m_DeadEmiter.SetPosition(m_Sprite.GetPosition().x + m_Sprite.GetSize().x/2, m_Sprite.GetPosition().y + m_Sprite.GetSize().y/2);
		m_DeadEmiter.Update();
		m_DeadEmiter.Draw();
	}

	m_EngineEmiter.SetPosition(m_Sprite.GetPosition().x + m_Sprite.GetSize().x/2 + m_EngineX, m_Sprite.GetPosition().y + m_Sprite.GetSize().y/2 + m_EngineY);
	m_EngineEmiter.Update();
	m_EngineEmiter.Draw();

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(m_Weapons[CurWeapon]->IsForeground() == true)
		{
			m_Weapons[CurWeapon]->DrawProjectiles();
		}
	}
}

void CShip::Shoot()
{
	if(IsDead())
	{
		return;
	}

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		m_Weapons[CurWeapon]->Shoot(m_Sprite.GetPosition().x + m_Sprite.GetSize().x/2,m_Sprite.GetPosition().y+m_Sprite.GetSize().y/2);
	}
}

void CShip::EquipWeapon(std::string Name)
{
	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(m_Weapons[CurWeapon]->GetName() == Name)
		{
			m_Weapons[CurWeapon]->Equip();

			for(std::size_t CurOldWeapon = 0; CurOldWeapon < m_Weapons.size(); CurOldWeapon++)
			{
				if(m_Weapons[CurOldWeapon]->GetGroupID() == m_Weapons[CurWeapon]->GetGroupID() &&
				   m_Weapons[CurOldWeapon]->GetName() != Name)
				{
					m_Weapons[CurOldWeapon]->Equip(false);
				}
			}
		}
	}
}

void CShip::EquipWeapon(int GroupID)
{
	int CurID = -1;
	std::size_t CurWeapon = 0;
	std::string OldName = "";

	for(CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(m_Weapons[CurWeapon]->GetGroupID() == GroupID && m_Weapons[CurWeapon]->IsEquiped() == true)
		{
			CurID = CurWeapon;
			OldName = m_Weapons[CurWeapon]->GetName();
			break;
		}
	}

	for(CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		/// \todo genauer überfliegen, evtl umschreiben
		if(m_Weapons[CurWeapon]->GetGroupID() == GroupID && CurID < CurWeapon && OldName != m_Weapons[CurWeapon]->GetName())
		{
			EquipWeapon(m_Weapons[CurWeapon]->GetName());
			return;
		}
	}
}

void CShip::ResetWeapons()
{
	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		m_Weapons[CurWeapon]->Reset();
	}
}

void CShip::MoveX(int Direction)
{
	if(IsDead() == true)
	{
		return;
	}

	m_Sprite.Move(m_Speed*Direction,0.0f);
}

void CShip::MoveY(int Direction)
{
	if(IsDead() == true)
	{
		return;
	}

	m_Sprite.Move(0.0f,	m_Speed*Direction);
}

bool CShip::CheckCollision(CPowerUp* CounterPart)
{
	if(IsDead() == true)
	{
		return false;
	}

	if(m_CollisionSystem.CheckCollision(CounterPart->GetCollisionSystem()))
	{
		EquipWeapon(CounterPart->GetEquipWeapon());
		EquipWeapon(CounterPart->GetWeaponGroupID());
		Repair(CounterPart->GetRepair());

		return true;
	}

	return false;
}

bool CShip::CheckCollision(CShip* CounterPart)
{
	int Damage = 0;
		
	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{	
		Damage = m_Weapons[CurWeapon]->CheckCollision(CounterPart->GetCollisionSystem());

		if(CounterPart->Damage(Damage) == true)
		{
			return true;
		}
	}

	return false;
}

void CShip::Repair(int Repair)
{
	if(IsDead() == true)
	{
		return;
	}

	m_CurArmor += Repair;

	if(m_CurArmor > m_Armor)
	{
		m_CurArmor = m_Armor;
	}
}

bool CShip::Damage(int Damage)
{
	if(IsDead() == true)
	{
		return true;
	}

	if(m_Invulnerable == true)
	{
		return false;
	}

	m_CurArmor -= Damage;

	if(m_CurArmor <= 0)
	{
		m_DeadSound.Play();

		m_DeadEmiter.Activate(true);
		m_DeadTime = sgl::get_clock().GetElapsedTime()*1000;

		return true;
	}
	return false;
}


int CShip::GetProjectilCount() const
{
	int Count = 0;
	
	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		Count += m_Weapons[CurWeapon]->GetProjectilCount();
	}

	return Count;
}

float CShip::GetProjectilXPosition(int Projectil) const
{
	int Add = 0;

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(Add + m_Weapons[CurWeapon]->GetProjectilCount() > Projectil)
		{
			return m_Weapons[CurWeapon]->GetProjectilX(Projectil - Add);
		}

		Add += m_Weapons[CurWeapon]->GetProjectilCount();
	}

	return 0.0f;
}

float CShip::GetProjectilYPosition(int Projectil) const
{
	int Add = 0;

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(Add + m_Weapons[CurWeapon]->GetProjectilCount() > Projectil)
		{
			return m_Weapons[CurWeapon]->GetProjectilY(Projectil - Add);
		}

		Add += m_Weapons[CurWeapon]->GetProjectilCount();
	}

	return 0.0f;
}

int CShip::GetProjectilDamage(int Projectil) const
{
	int Add = 0;

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(Add + m_Weapons[CurWeapon]->GetProjectilCount() > Projectil)
		{
			return m_Weapons[CurWeapon]->GetDamage();
		}

		Add += m_Weapons[CurWeapon]->GetProjectilCount();
	}

	return 0;
}


float CShip::GetProjectilWidth(int Projectil) const
{
	int Add = 0;

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(Add + m_Weapons[CurWeapon]->GetProjectilCount() > Projectil)
		{
			return m_Weapons[CurWeapon]->GetProjectilWidth(Projectil - Add);
		}

		Add += m_Weapons[CurWeapon]->GetProjectilCount();
	}

	return 0.0f;
}


int CShip::GetWeaponDamage(std::size_t Weapon) const
{
	if(Weapon > m_Weapons.size())
	{
		return 0;
	}

	return m_Weapons[Weapon]->GetDamage();
}

std::string CShip::GetWeaponName(std::size_t Weapon) const
{
	if(Weapon > m_Weapons.size())
	{
		return 0;
	}

	return m_Weapons[Weapon]->GetName();
}


int CShip::GetWeaponTypeCount() const
{
	std::vector<int> UsedTypes;
	bool Used;
	int Count = 0;

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		Used = false;
		for(std::size_t CurType = 0; CurType < UsedTypes.size(); CurType++)
		{
			if(UsedTypes[CurType] == m_Weapons[CurWeapon]->GetGroupID())
			{
				Used = true;
			}
		}

		if(Used == false)
		{
			Count++;
			UsedTypes.push_back(m_Weapons[CurWeapon]->GetGroupID());
		}
	}

	UsedTypes.clear();

	return Count;
}

std::string CShip::GetCurWeaponNameOfType(std::size_t GroupID) const
{
	const CWeapon* Weapon = GetCurWeaponOfType(GroupID);

	if(Weapon != NULL)
	{
		return Weapon->GetName();
	}

	return "";
}

const CWeapon* CShip::GetCurWeaponOfType(std::size_t GroupID) const
{
	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(m_Weapons[CurWeapon]->GetGroupID() == GroupID && m_Weapons[CurWeapon]->IsEquiped())
		{
			return m_Weapons[CurWeapon];
		}
	}

	return NULL;
}

int CShip::GetProjectileTarget(int YDistance) const
{
	int Count = 0;;
	int XSum = 0;

	for(std::size_t CurWeapon = 0; CurWeapon < m_Weapons.size(); CurWeapon++)
	{
		if(m_Weapons[CurWeapon]->IsEquiped())
		{
			++Count;
			XSum += m_Weapons[CurWeapon]->GetProjectilXRel(YDistance);
		}
	}

	return (Count==0) ? 0 : (XSum / Count);
}


#endif
