# svs_simulation.world.processes

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
hanlders for processing the simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.utilities.constants import sim_const
from svs_simulation.events.eventlib import SimEventCentre

#############################
# EXCEPTIONS
#############################
class ProcessException(Exception):pass

#############################
# CLASSES
#############################
class ProcessHandler:
	"""
	Provides generic support for handling update and event messaging
	processed within the simulation.
	"""
	def __init__(self):
		self.name = None
		self.world = None
		self.components = []
		self.eventCentre = SimEventCentre()

	def addComponent(self, component):
		"""
		Adds new component to process making sure
		that it is not already present.
		"""
		if not component in self.components:
			component.processHandler = self
			self.components.append(component)

	def addToWorld(self, world):
		"""
		Called when added to simulation world.

		This should be overridden by extending classes to setup
		components for processing.
		"""
		self.world = world

	#############################
	# PROCESS FUNCTIONS
	#############################
	def updateWorld(self, simTime):
		"""
		Forwards C{update} call from world.
		"""
		for component in self.components:component.updateWorld(simTime)

	def startWorld(self, simTime):
		"""
		Forwards C{start} call from world.
		"""
		print "startWorld"
		for component in self.components:component.startWorld(simTime)

	def stopWorld(self, simTime):
		"""
		Forwards C{stop} call from world.
		"""
		print "stopWorld"
		for component in self.components:component.stopWorld(simTime)

	#############################
	# PROCESS HANDLING
	#############################
	def postEvent(self, event):
		"""
		Receives change event from component.
		"""
		self.eventCentre.postEvent(event)

	def getChanges(self):
		"""
		Return changes that have occurred during last update.
		"""
		return self.eventCentre.getEvents(sim_const.LABEL_UPDATE, flush=True)


	#############################
	# EVENT FUNCTIONS
	#############################
	def notifyOfEvents(self, events):
		"""
		Forwards event messages.
		"""
		for component in self.components:component.handleEvents(events)


	#############################
	# ARCHIVING
	#############################
	def setup(self, name=None):
		self.name = name

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		data = {}
		if self.name:data[sim_const.LABEL_NAME] = self.name
		return data

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		name = data.get(sim_const.LABEL_NAME, None)
		self.setup(name=name)

		
#############################
# STANDALONE
#############################
class StandaloneProcessHandler(ProcessHandler):
	"""
	Does all processing as for world running as standalone process.
	"""
	def __init__(self):
		ProcessHandler.__init__(self)

	def addToWorld(self, world):
		"""
		Called when added to simulation world.

		This should be overridden by extending classes to setup
		components for processing.
		"""
		self.world = world
		for agentGroup in self.world.agentGroups.groups.values():
			for agent in agentGroup.getMembers():self.addComponent(agent)
		for objectGroup in self.world.objectGroups.groups.values():
			for simObject in objectGroup.getMembers():self.addComponent(simObject)




#############################
# DISTRIBUTED
#############################
class ProcessHandlerProxy(ProcessHandler):
	"""
	Represents remote process handler to world.
	"""
	def __init__(self, localClient):
		ProcessHandler.__init__(self, world)
		self.localClient = localClient
		self.remoteClient = None

	#############################
	# NETWORK FUNCTIONS
	#############################
	def connectToClient(self, remoteClient):
		"""
		Called when client connects to world, forwards necessary data.
		"""
		self.remoteClient = remoteClient

	def disconnectFromClient(self):
		"""
		Called when client discconnects from world.
		"""
		self.remoteClient = None
	
	#############################
	# PROCESS FUNCTIONS
	#############################
	def updateWorld(self, simTime):
		"""
		Forwards C{update} call from world.
		"""
		if not self.remoteClient:return
		self.localClient.sendUpdateWorld(self.remoteClient, simTime)

	def startWorld(self, simTime):
		"""
		Forwards C{start} call from world.
		"""
		if not self.remoteClient:return
		self.localClient.sendStartWorld(self.remoteClient, simTime)

	def stopWorld(self, simTime):
		"""
		Forwards C{stop} call from world.
		"""
		if not self.remoteClient:return
		self.localClient.sendStopWorld(self.remoteClient, simTime)

	#############################
	# EVENT FUNCTIONS
	#############################
	def notifyOfEvents(self, events):
		"""
		Forwards event messages.
		"""
		if not self.remoteClient:return
		self.localClient.sendNotifyOfEvents(self.remoteClient, events)



class EntityGroupProcessHandlerProxy(ProcessHandlerProxy):
	"""
	Represents remote process handler that manages one or more groups of entities.
	"""
	def __init__(self, localClient, remoteClient, groupNames):
		ProcessHandlerProxy.__init__(self, localClient, remoteClient)
		self.groupNames = groupNames


class AgentGroupProcessHandlerProxy(EntityGroupProcessHandlerProxy):
	"""
	Represents remote process handler that manages a group of agents.
	"""
	def __init__(self, localClient, remoteClient, groupNames):
		EntityGroupProcessHandlerProxy.__init__(self, localClient, remoteClient, groupNames)

	def addToWorld(self, world):
		"""
		Called when added to simulation world.

		This should be overridden by extending classes to setup
		components for processing.
		"""
		self.world = world
		# send data to client


class ObjectGroupProcessHandlerProxy(EntityGroupProcessHandlerProxy):
	"""
	Represents remote process handler that manages a group of objects.
	"""
	def __init__(self, localClient, remoteClient, groupNames):
		EntityGroupProcessHandlerProxy.__init__(self, localClient, remoteClient, groupNames)

	def addToWorld(self, world):
		"""
		Called when added to simulation world.

		This should be overridden by extending classes to setup
		components for processing.
		"""
		self.world = world
		# send data to client




#############################
# FUNCTIONS
#############################
def createProcessHandlerFromSVSFile(filename, processClass=ProcessHandler):
	"""
	Reads in data file for process handler and creates it.
	
	@rtype: L{ProcessHandler}
	"""
	data = PersistentObject()
	data.read(filename)
	procHandler = processClass()
	procHandler.decode(data.process)
	return procHandler
