# svs_simulation.utilities.svgutils

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Utility functions and classes for handling SVG file data.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from string import upper, lower, split, strip

# internal imports
from svs_core.filehandling.loaders import DocumentLoadingException, DocumentLoaderXML, GenericXMLContentHandler, malformedInputForXMLNodeException
from svs_simulation.utilities.constants import sim_const, sim_xml
from svs_core.utilities.lib import Constants

#############################
# CONSTANTS
#############################
svgutils_const = Constants()
# parser states
svgutils_const.PARSE_START = 1
svgutils_const.PARSE_M = 10
svgutils_const.PARSE_L = 11
svgutils_const.PARSE_Z = 12

#############################
# FUNCTIONS
#############################
def convertVerticesFromString(pathString):
	"""
	Converts the list of vertices provided as a string
	into an array in which each vertice is a tuple.
	"""
	vertarray = []
	tmparray = None
	tmpList = pathString.split()
	parserState = svgutils_const.PARSE_START
	for item in tmpList:
		#print item
		if item.lower() == 'm':
			#print '__m'
			parserState = svgutils_const.PARSE_M # start path
			tmparray = []
		if item.lower() == 'l':
			#print '__l'
			parserState = svgutils_const.PARSE_L # link on path
		if item.lower() == 'z':
			#print '__z:', tmparray
			parserState = svgutils_const.PARSE_Z # end path
			if tmparray:
				vertarray.append(tmparray)
				tmparray = None
		else:
			points = __convertPointsFromString(item)
			#print "points:", points
			if points:tmparray.append(points)
	if tmparray:vertarray.append(tmparray)
	#print "convertVerticesFromString:", vertarray
	return vertarray


def __convertPointsFromString(pointString):
	"""
	Converts comma separated points to tuple of numbers.
	"""
	pointParts = pointString.split(',')
	pointTuple = None
	if len(pointParts) < 2:return None
	try:pointTuple = (float(pointParts[0]), float(pointParts[1]))
	except:return None
	return pointTuple

def parseSVSDataValues(dataString):
	"""
	Parses string provided from C{SVS_DATA} attribute
	and returns as a dict.
	"""
	data = {}
	tmpList = dataString.split(';')
	for item in tmpList:
		itemstrs = item.split('=')
		if len(itemstrs) < 2:continue
		itemstrs[1] = itemstrs[1].strip()
		if itemstrs[1].isalpha():data[itemstrs[0].strip()] = itemstrs[1]
		else:
			try:data[itemstrs[0].strip()] = float(itemstrs[1])
			except ValueError:data[itemstrs[0].strip()] = itemstrs[1]
	return data
	
#############################
# LOADER AND PARSER
#############################
class SVGLoader(DocumentLoaderXML):
	"""
	Class for reading and writing structures in xml (SVG) format.
	""" 
	def __init__(self, target, contentHandlerClass=None):
		DocumentLoaderXML.__init__(self, target, contentHandlerClass)



class SVGContentHandler(GenericXMLContentHandler):
	"""
	Handler for parsing terrain SVG document.
	"""
	
	def __init__(self, target):
		GenericXMLContentHandler.__init__(self, target)
		self.chardata = ""
		self.tmpdata = None
		self.parentNode = None
	
	##########################
	# GENERAL XML HANDLING
	##########################
	def startElement(self, name, attrs):
		name = lower(name)
		if name == sim_xml.RECT:self.handleRectTag(attrs)
		elif name == sim_xml.PATH:self.handlePathTag(attrs)

		
	def endElement(self, name):
		name = lower(name)
		self.chardata = strip(self.chardata)
		
	
	##########################
	# SPECIFIC TAG HANDLING
	##########################	
	def handleRectTag(self, attrs):
		"""
		Parses C{RECT} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = sim_xml.RECT
		try:
			x = attrs.get(sim_xml.X)
			y = attrs.get(sim_xml.Y)
			width = attrs.get(sim_xml.WIDTH)
			height = attrs.get(sim_xml.HEIGHT)
			svs_data = attrs.get(sim_xml.DATA)
			#svgid = attrs.get('id')
			#print "id:", svgid
		except ValueError:raise DocumentLoadingException("Essential data missing from input document.")

		if x:
			try:x = float(x.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		
		if y:
			try:y = float(y.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)

		if width:
			try:width = float(width.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		
		if height:
			try:height = float(height.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		verts = [(x, y), (x + width, y), (x + width, y+height), (x, y+height), (x, y)]
		
		if not svs_data:raise DocumentLoadingException("Essential data missing from input document.")
		if svs_data:
			svs_data = parseSVSDataValues(svs_data.encode("utf-8"))
			self.createSimComponent(verts, svs_data)
		else:raise DocumentLoadingException("No svs_data defined for node.")
		
	
	def handlePathTag(self, attrs):
		"""
		Parses C{PATH} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = sim_xml.PATH
		try:
			pathPoints = attrs.get(sim_xml.D)
			svs_data = attrs.get(sim_xml.DATA)
			#svgid = attrs.get('id')
			#print "id:", svgid
		except ValueError:raise DocumentLoadingException("Essential data missing from input document.")
		if pathPoints:verts = convertVerticesFromString(pathPoints.encode("utf-8"))
		else:raise DocumentLoadingException("Vertices not defined.")
		if not verts:raise DocumentLoadingException("Vertices not defined.")
		if svs_data:
			svs_data = parseSVSDataValues(svs_data.encode("utf-8"))
			self.createSimComponent(verts, svs_data)
		else:raise DocumentLoadingException("No svs_data defined for node.")

	
	def createSimComponent(self, vertices, svs_data):
		"""
		Creates simulation object from data.

		This should be overidden by implementing classes.
		"""
		print "createSimComponent: PASS"

