# svs_simulation.terrain.special_areas

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Classes for representing specialised forms of terrain-based information..

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.utilities.constants import sim_const
from svs_simulation.entities.base_entities import SpatialEntity


#############################
# CLASSES
#############################
class GenericArea(SpatialEntity):
	"""
	Base class for all specialised areas.
	"""
	def __init__(self):
		SpatialEntity.__init__(self)
		self.floorplane = None
		self.name = None

	def setup(self, idtag=None, name=None):
		SpatialEntity.setup(self, idtag)
		self.name = name

	def getName(self):
		"""
		Returns name for structure.  If the structure
		has no name retuens its idtag.
		"""
		if not self.name:return self.idtag
		return self.name

	def containsPoint(self, x, y):
		"""
		Tests if coordinates are contaiend within structure bounds.
		"""
		return self.bounds.containsPointInWorld(x, y)

	def __str__(self):
		"""
		Returns string representation of structure.
		"""
		return "structure [%s]" % self.idtag

	def setFloorplane(self, floorplane):
		"""
		Sets floorplane and adjust bounds to contain it.
		"""
		self.floorplane = floorplane
		for vertice in floorplane.getVertices():
			self.bounds.addPoint(vertice[0], vertice[1])

	def encode(self):
		"""
		Returns encoded representation of self.

		@rtype: dict
		"""
		data = SpatialEntity.encode(self)
		if self.floorplane:data[sim_const.LABEL_FLOORPLANE] = self.floorplane.encode()
		if self.name:data[sim_const.LABEL_NAME] = self.name
		return data

	def decode(self, data):
		"""
		Decodes data and applies it to self.
		"""
		SpatialEntity.decode(self, data)
		if data.has_key(sim_const.LABEL_FLOORPLANE):
			floorplane = FloorPlane()
			floorplane.decode(data[sim_const.LABEL_FLOORPLANE])
			self.setFloorplane(floorplane)
		if data.has_key(sim_const.LABEL_NAME):self.name = data[sim_const.LABEL_NAME]


	def enterPartition(self, partition):
		"""
		Called when structure is added to a partition.
		"""
		self.partition = partition

	def exitPartition(self, partition):
		"""
		Called when structure is removed from a partition.
		"""
		self.partition = None

	def enter(self, agent):
		"""
		Responds to agent entering area.
		"""
		pass

	def exit(self, agent):
		"""
		Responds to agent leaving area.
		"""
		pass


class SimDataArea(GenericArea):
	"""
	Area that contains simulation data.
	"""
	def __init__(self):
		GenericArea.__init__(self)
		self.simdata = None

	def setup(self, idtag=None, name=None, simdata=None):
		GenericArea.setup(self, idtag=idtag, name=name)
		self.simdata = simdata

	def encode(self):
		"""
		Returns encoded representation of self.

		@rtype: dict
		"""
		data = GenericArea.encode(self)
		if self.simdata:data[sim_const.LABEL_SIMDATA] = self.simdata.encode()
		return data

	def decode(self, data):
		"""
		Decodes data and applies it to self.
		"""
		GenericArea.decode(self, data)
		if data.has_key(sim_const.LABEL_SIMDATA):self.simdata = data[sim_const.LABEL_SIMDATA].decode()

	def inspect(self, request=None, agent=None):
		"""
		Responds to agent requesting data from area.
		"""
		pass

	def deposit(self, data, agent=None):
		"""
		Responds to agent placing data in area.
		"""
		pass

	def collect(self, request=None, agent=None):
		"""
		Responds to agent collecting data from area.
		"""
		pass


class AudioArea(GenericArea):
	"""
	Area that controls audio within simulation.
	"""
	def __init__(self):
		GenericArea.__init__(self)
