# svs_simulation.terrain.base_classes

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Manager classes for simulation terrains.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities.objectmanager import PersistentObject
from svs_simulation.numdata.geomlib import geom_const, AxialBounds2D
from svs_simulation.utilities.constants import sim_const
from svs_simulation.terrain.util_classes import GenericTerrainObject
from svs_simulation.terrain.structures import Structure
from svs_simulation.terrain.navgraphs import NavGraph
from svs_simulation.simdata.simdatalib import SimData
from svs_simulation.terrain.paths import Path, PathStore
from svs_simulation.terrain.linkpaths import LinkPathGraph


#############################
# EXCEPTIONS
#############################
class PartitionException(Exception):pass


#############################
# FUNCTIONS
#############################
def createTerrainFromSVSFile(filename):
	"""
	Reads in data file for terrain and creates it.
	
	@rtype: L{Terrain}
	"""
	tData = PersistentObject()
	tData.read(filename)
	terrain = Terrain()
	terrain.decode(tData.terrain)
	return terrain

def createTerrainFromSVGFile(filename):
	"""
	Converts data in SVG file to Terrain object.

	@rtype: L{Structure}
	"""
	from svs_simulation.terrain.svgloaders import createTerrainSVGLoader
	terrain = Terrain()
	terrain.setup()
	loader = createTerrainSVGLoader(terrain)
	loader.read(filename)
	terrain.resolveBounds()
	return terrain

def saveTerrainToFile(terrain, filename):
	"""
	Writes terrain data to file.
	"""
	from svs_core.utilities.lib import validateFilePath
	filename = validateFilePath(filename)
	data = terrain.encode()
	dataStr = "terrain = %s" % data
	f = open(filename, 'w')
	f.write(dataStr)
	f.close()
		
	
#############################
# CLASSES
#############################
class Terrain(GenericTerrainObject):
	"""
	Provides overall management of terrain partitions.

	Every terrain should contain at least one partition.
	"""
	def __init__(self):
		GenericTerrainObject.__init__(self)
		self.name = None
		self.world = None
		self.bounds = None
		self.partitions = {}
		self.pathStore = PathStore()
		self.navGraph = NavGraph()
		self.nodesByName = {}
		# linkpaths
		self.linkPathGraph = LinkPathGraph('%s link paths' % self.name)

	def setup(self, name=None, bounds=None, spaceX=None, spaceY=None, margin=None):
		#self.world = world
		self.name = name
		if not bounds:self.bounds = AxialBounds2D()
		else:self.bounds = bounds
		self.spaceX = spaceX
		self.spaceY = spaceY
		self.margin = margin
		self.linkPathGraph = LinkPathGraph('%s link paths' % self.name)

	def prepareToRun(self):
		"""
		Ensures that all necessary data is setup for terrain,
		makes sure linkpaths are resilved, etc.
		"""
		#pass
		self.linkPathGraph.resolveLinkPaths(self)

	#############################
	# BOUNDS
	#############################
	def getBoundsMinX(self):
		"""
		Returns minimum x coordinate from bounds.
		"""
		return self.bounds.minX

	def getBoundsMinY(self):
		"""
		Returns minimum y coordinate from bounds.
		"""
		return self.bounds.minY

	def getBoundsMaxX(self):
		"""
		Returns maximum x coordinate from bounds.
		"""
		return self.bounds.maxX

	def getBoundsMaxY(self):
		"""
		Returns maximum y coordinate from bounds.
		"""
		return self.bounds.maxY

	def setBoundsMinX(self, newvalue):
		"""
		Sets minimum x coordinate from bounds.
		"""
		self.bounds.minX = newvalue

	def setBoundsMinY(self, newvalue):
		"""
		Returns minimum y coordinate from bounds.
		"""
		self.bounds.minY = newvalue

	def setBoundsMaxX(self, newvalue):
		"""
		Returns maximum x coordinate from bounds.
		"""
		self.bounds.maxX = newvalue

	def setBoundsMaxY(self, newvalue):
		"""
		Returns maximum y coordinate from bounds.
		"""
		self.bounds.maxY = newvalue

	def resolveBounds(self):
		"""
		Checks the bounds of the terrain match its contents.
		"""
		for partition in self.partitions.values():
			partition.resolveBounds()
			self.bounds.addPoint(partition.bounds.minX, partition.bounds.minY)
			self.bounds.addPoint(partition.bounds.maxX, partition.bounds.maxY)	

	#############################
	# PARTITIONS
	#############################
	def addPartition(self, partition):
		"""
		Adds a new partition to the terrain.
		"""
		partition.terrain = self
		self.partitions[partition.name] = partition

	def createPartitionFromName(self, partitionName, bounds=None):
		"""
		Create a new partition to the terrain with the speficied name.

		If a partition with the same name already exists, this is
		returned in its place. 
		"""
		if self.partitions.has_key(partitionName):return self.partitions[partitionName]
		partition = Partition()
		partition.setup(name=partitionName, 
				terrain=self, 
				bounds=bounds,
				spaceX=self.spaceX, 
				spaceY=self.spaceY, 
				margin=self.margin)
		self.addPartition(partition)
		return partition

	def getPartitionAtPoint(self, x, y):
		"""
		Returns partition containing specified coordinates.

		Returns C{None} if not found.
		"""
		for partition in self.partitions.values():
			if partition.containsPoint(x, y):return partition
		return None

	def getPartition(self, partitionName):
		"""
		Returns partition mathcing specified name.

		Returns C{None} if not found.
		"""
		return self.partitions.get(partitionName, None)

	#############################
	# STRUCTURES
	#############################
	def addStructure(self, structure, partitionName):
		"""
		Adds structure to partition with specified name.

		If the partition does not exist it is created.
		If no partition is specified, it is ignored.
		"""
		if not partitionName:return
		partition = self.createPartitionFromName(partitionName)
		partition.addStructure(structure)
		self.bounds.addPoint(partition.bounds.minX, partition.bounds.minY)
		self.bounds.addPoint(partition.bounds.maxX, partition.bounds.maxY)

	def createStructureFromName(self, structureName, partitionName):
		"""
		Creates structure in specified partition with specified name and
		either wall or floorplane components.

		If the partition does not exist it is created.
		If no partition is specified, it is ignored.
		"""
		# check if it already exists first
		partition = self.createPartitionFromName(partitionName)
		structure = partition.createStructureFromName(structureName)
		return structure

	def getStructureContainingPoint(self, x, y):
		"""
		Returns the structure within which the
		specified coordinates are located.
		"""
		partition = self.getPartitionAtPoint(x, y)
		if not partition:return None
		return partition.getStructureAtPoint(x, y)


	#############################
	# SIM OBJECTS
	#############################
	def addSimObjectToStructure(self, simObject, structureName):
		"""
		Adds sim object to partition with specified name.

		If the partition does not exist it is created.
		If no partition is specified, it is ignored.
		"""
		pass
		#if not partitionName:return
		#partition = self.createPartitionFromName(partitionName)
		#partition.addStructure(structure)
		#self.bounds.addPoint(partition.bounds.minX, partition.bounds.minY)
		#self.bounds.addPoint(partition.bounds.maxX, partition.bounds.maxY)

	def createSimObjectFromName(self, objectName, objectClass, structureName, vertices):
		"""
		Creates structure to partition with specified name and
		eother wall or floorplane components.

		If the partition does not exist it is created.
		If no partition is specified, it is ignored.
		"""
		pass
		# check if it already exists first
		#partition = self.createPartitionFromName(partitionName)
		#structure = partition.createStructureFromName(structureName)
		#return structure

	#############################
	# PATHS
	#############################
	def addPath(self, path):
		"""
		Adds a path to the terrain.
		"""
		self.pathStore.addPath(path)

	def getPath(self, name=None, start=None, destination=None):
		"""
		Retrieves path from path store.
		"""
		return self.pathStore.getPath(name=name, start=start, destination=destination)

	def getAllPaths(self):
		"""
		Returns a list of all paths for terrain.
		"""
		return self.pathStore.getAllPaths()

	#############################
	# LINKPATHS
	#############################
	def addLinkPath(self, linkPath):
		"""
		Adds linkpath to terrain, if the path has not been mapped to
		specific structures it attempts to do this.
		"""
		self.linkPathGraph.addLinkPath(linkPath)

	def getAllLinkPaths(self):
		"""
		Returns a list of all resolved linkpaths for terrain.
		"""
		return self.linkPathGraph.getAllPaths()
 

	#############################
	# NAV GRAPH
	#############################
	def addTerrainNavigationNode(self, x, y):
		"""
		Adds and returns new node to high-level navigation graph.

		This also sets the partition and structure that the node is within.
		"""
		partition = self.getPartitionAtPoint(x, y)
		if not partition:return None
		structure = partition.getStructureAtPoint(x, y)
		if not structure:return None
		simdata = SimData()
		simdata.content = {sim_const.LABEL_PARTITION:partition, sim_const.LABEL_STRUCTURE:structure}
		node = self.navGraph.addNode(x, y, simdata=simdata)
		self.nodesByName[structure.getName()] = node
		return node

	def removeTerrainNavigationNode(self, node):
		"""
		Removes node from high-level navigation graph.
		"""
		try:
			self.navGraph.removeNode(node)
			self.nodesByName.pop(node)
		except:pass

	def linkTerrainNavigationNodes(self, nodeList):
		"""
		Creates edges between list of navigation nodes.
		"""
		for node in nodeList:
			for otherNode in nodeList:
				if node != otherNode:self.navGraph.addEdge(node, otherNode)
		

	#############################
	# ARCHIVING
	#############################
	def encodeForProxy(self):
		"""
		Returns simplified encoding of self, suitable for use by
		proxies.

		This includes the basic terrain data and the names, but 
		none of the data of its partitions.
		"""
		encoded = self.encode(partitions=False, structures=False, objects=False, agents=False)
		encoded[sim_const.LABEL_PARTITIONS] = self.partitions.keys()
		return encoded

	def encode(self, partitions=True, structures=True, objects=True, agents=False):
		"""
		Returns encoded model of self.

		@rtype: dict
		"""
		encoded = {}
		encoded[sim_const.LABEL_NAME] = self.name
		encoded[sim_const.LABEL_SPACEX] = self.spaceX
		encoded[sim_const.LABEL_SPACEY] = self.spaceY
		encoded[sim_const.LABEL_MARGIN] = self.margin
		encoded[sim_const.LABEL_BOUNDS] = self.bounds.encode()
		encoded[sim_const.LABEL_PATHS] = self.encodePaths()
		encoded[sim_const.LABEL_LINKPATHS] = self.linkPathGraph.encode()
		if partitions:encoded[sim_const.LABEL_PARTITIONS] = self.encodePartitions(structures, objects, agents)
		return encoded

	def encodePartitions(self, structures=True, objects=True, agents=False):
		"""
		Returns encoded model of partitions.

		@rtype: dict
		"""
		encoded = []
		for partition in self.partitions.values():encoded.append(partition.encode(structures, objects, agents))
		return encoded

	def encodePaths(self):
		"""
		Returns encoded model of paths.

		@rtype: dict
		"""
		encoded = []
		for path in self.pathStore.getAllPaths():encoded.append(path.encode())
		return encoded


	def decode(self, data):
		"""
		Creates terrain from encoded data.
		
		@type data:dict
		"""
		bounds = AxialBounds2D()
		bounds.decode(data[sim_const.LABEL_BOUNDS])
		self.setup(data.get(sim_const.LABEL_NAME, 'terrain'),
			bounds,
			data[sim_const.LABEL_SPACEX], 
			data[sim_const.LABEL_SPACEY],
			data[sim_const.LABEL_MARGIN])
		if data.has_key(sim_const.LABEL_PATHS):
			self.decodePaths(data[sim_const.LABEL_PATHS])
		if data.has_key(sim_const.LABEL_LINKPATHS):
			self.linkPathGraph.decode(data[sim_const.LABEL_LINKPATHS])
		if data.has_key(sim_const.LABEL_PARTITIONS):
			self.decodePartitions(data[sim_const.LABEL_PARTITIONS])

	def decodePartitions(self, data):
		"""
		Creates partitions from encoded data.

		@type data:dict
		"""
		for entry in data:
			partition = Partition()
			partition.decode(entry, self)
			self.addPartition(partition)

	def decodePaths(self, data):
		"""
		Creates paths from encoded data.

		@type data:dict
		"""
		for entry in data:
			path = Path()
			path.decode(entry)
			self.addPath(path)
		



class Partition(GenericTerrainObject):
	"""
	Represents a section of a terrain.

	The partition handles objects and structures within the terrain.
	"""
	def setup(self, name, terrain=None, bounds=None, spaceX=None, spaceY=None, margin=None):
		self.name = name
		if not bounds:self.bounds = AxialBounds2D()
		else:self.bounds = bounds
		self.terrain = terrain
		if not terrain:self.spaceX = spaceX
		else:self.spaceX = self.terrain.spaceX
		if not terrain:self.spaceY = spaceY
		else:self.spaceY = self.terrain.spaceY
		if not terrain:self.margin = margin
		else:self.margin = self.terrain.margin
		self.structures = {}
		self.objects = {}
		self.agents = {}
		#self.createTiles(tileX, tileY)

	def __str__(self):
		"""
		Returns string representation of partition.
		"""
		return "partition [%s]" % self.name

	#############################
	# BOUNDS
	#############################
	def resolveBounds(self):
		"""
		Checks the bounds of the partition match its contents.
		"""
		self.bounds.reset()
		for structure in self.structures.values():
			structBounds = structure.getWorldbounds()
			self.bounds.addPoint(structBounds.minX, structBounds.minY)
			self.bounds.addPoint(structBounds.maxX, structBounds.maxY)

	#############################
	# TILES
	#############################
	def createTiles(self, tilesX, tilesY):
		"""
		Creates empty tile set for partition.
		"""
		self.tilesX = tilesX
		self.tilesY = tilesY
		self.tiles = []
		for i in range(tileX * tileY):self.tiles.append((None, None, None))

	def mapStructureToTiles(self, structure):
		"""
		Creates tiles representation for structure.
		"""
		minX = structure.bounds.minX - self.rect.origin.x
		minY = structure.bounds.minY - self.rect.origin.y
		maxX = structure.bounds.maxX - self.rect.origin.x
		maxY = structure.bounds.maxY - self.rect.origin.y
		structure.mapToTile(self.tiles, minX, minY, maxX, maxY)

	def clearStructureFromTiles(self, structure):
		"""
		Clears tiles representation for structure.
		"""
		minX = structure.bounds.minX - self.rect.origin.x
		minY = structure.bounds.minY - self.rect.origin.y
		maxX = structure.bounds.maxX - self.rect.origin.x
		maxY = structure.bounds.maxY - self.rect.origin.y
		width = maxX - minX
		height = maxY - minY
		for y in range(height):
			horiz = y * width
			for x in range(width):
				idx = horiz + x
				self.tiles[idx][0] = None
				self.tiles[idx][1] = None

	def mapObjectToTiles(self, obj):
		"""
		Creates tiles representation for object.
		"""
		minX = obj.bounds.minX - self.rect.origin.x
		minY = obj.bounds.minY - self.rect.origin.y
		maxX = obj.bounds.maxX - self.rect.origin.x
		maxY = obj.bounds.maxY - self.rect.origin.y
		obj.mapToTile(self.tiles, minX, minY, maxX, maxY)

	def clearObjectFromTiles(self, obj):
		"""
		Clears tiles representation for structure.
		"""
		minX = obj.bounds.minX - self.rect.origin.x
		minY = obj.bounds.minY - self.rect.origin.y
		maxX = obj.bounds.maxX - self.rect.origin.x
		maxY = obj.bounds.maxY - self.rect.origin.y
		width = maxX - minX
		height = maxY - minY
		for y in range(height):
			horiz = y * width
			for x in range(width):
				self.tiles[horiz + x][2] = None

	#############################
	# STRUCTURES
	#############################
	def addStructure(self, structure):
		"""
		Adds a new structure to the partition.

		Every structure must have a unique name.  If a structure 
		is added with an existing name it is rejected.
		"""
		if self.structures.has_key(structure.name):return
		#print "addStructure:", structure.name
		self.structures[structure.name] = structure
		structure.enterPartition(self)
		structBounds = structure.getWorldbounds()
		self.bounds.addPoint(structBounds.minX, structBounds.minY)
		self.bounds.addPoint(structBounds.maxX, structBounds.maxY)

	def createStructureFromName(self, structureName):
		"""
		Checks of structure with given name alreday exists, and
		returns this, iof not creates a new structure.
		"""
		if self.structures.has_key(structureName):return self.structures[structureName]
		structure = Structure()
		structure.setup(name=structureName)
		self.addStructure(structure)
		return structure
		

	def removeStructure(self, structure):
		"""
		Removes an existing structure from the partition.
		"""
		if not self.structures.has_key(structure._uid):
			raise PartitionException("structure <%s> is not present in partition <%s>" % (structure._uid, self._uid))
		structure.exitPartition(self)
		return self.structures.popitem(structure._uid)


	#############################
	# OBJECTS
	#############################
	def addObject(self, obj):
		"""
		Adds a new object to the partition.
		"""
		obj.enterPartition(self)
		self.objects[obj._uid] = obj

	def removeObject(self, obj):
		"""
		Removes an existing object from the partition.
		"""
		if not self.objects.has_key(obj._uid):
			raise PartitionException("object <%s> is not present in partition <%s>" % (obj._uid, self._uid))
		obj.exitPartition(self)
		return self.objects.popitem(obj._uid)

	#############################
	# AGENTS
	#############################
	def addAgent(self, agent):
		"""
		Adds a new agent to the partition.
		"""
		agent.enterPartition(self)
		self.agents[agent._uid] = agent

	def removeAgent(self, agent):
		"""
		Removes an existing agent from the partition.
		"""
		if not self.agents.has_key(agent._uid):
			raise PartitionException("agent <%s> is not present in partition <%s>" % (agent._uid, self._uid))
		agent.exitPartition(self)
		return self.agents.popitem(agent._uid)

	#############################
	# UPDATES
	#############################
	def update(self, timeInterval):
		"""
		Forwards and update message to components in the partition.
		"""
		for agent in self.agents.values():agent.update(timeInterval)
		
	#############################
	# ENVIRONMENT INFO
	#############################
	def containsPoint(self, x, y):
		"""
		Checks if point is within bounds of partition.
		"""
		return self.bounds.containsPoint(x, y)

	def intersects(self, minX, minY, maxX, maxY):
		"""
		Checks if specified area intersects with partition.
		"""
		return self.bounds.intersects(minX, minY, maxX, maxY)

	def getAgentsInArea(self, minX, minY, maxX, maxY, returnTagged=False):
		"""
		Returns a list of agents that lie within the
		specified bounds.

		@rtype: array
		"""
		agentsInArea = []
		for agent in self.agents.values():
			if agent.containedWithin(minX, minY, maxX, maxY):
				if returnTagged and agent.tagged:agentsInArea.append(agent)
		return agentsInArea

	def getObjectsInArea(self, minX, minY, maxX, maxY, returnTagged=False):
		"""
		Returns a list of objects that lie within the
		specified bounds.

		@rtype: array
		"""
		objectsInArea = []
		for obj in self.objects.values():
			if obj.containedWithin(minX, minY, maxX, maxY):
				if returnTagged and obj.tagged:objectsInArea.append(object)
		return objectsInArea

	def getStructuresInArea(self, minX, minY, maxX, maxY, returnTagged=False):
		"""
		Returns a list of structures that lie within the
		specified bounds.

		@rtype: array
		"""
		structuresInArea = []
		for structure in self.structures.values():
			if structure.containedWithin(minX, minY, maxX, maxY):
				if returnTagged and structure.tagged:structuresInArea.append(structure)
		return structuresInArea

	def getStructuresInAreaWithDensity(self, minX, minY, maxX, maxY, density=0.0):
		"""
		Returns a list of structures that lie within the
		specified bounds and whose density is greater then or equal
		to the specified density.

		@rtype: array
		"""
		structuresInArea = []
		for structure in self.structures.values():
			if structure.containedWithin(minX, minY, maxX, maxY):
				if structure.density >= density:structuresInArea.append(structure)
		return structuresInArea

	def getStructureAtPoint(self, x, y):
		"""
		Returns structure containing specified coordinates.

		Returns C{None} if not found.
		"""
		for structure in self.structures.values():
			if structure.containsPoint(x, y):return structure
		return None

	def getTaggedEntitiesInArea(self, minX, minY, maxX, maxY):
		"""
		Returns a list of entities that lie within the
		specified bounds and are tagged.

		This returns a list including agents, objects and structures.

		@rtype: array
		"""
		tagged = {}
		tagged[sim_const.LABEL_AGENT] = self.getAgentsInArea(minX, minY, maxX, maxY, True)
		tagged[sim_const.LABEL_OBJECT] = self.getObjectsInArea(minX, minY, maxX, maxY, True)
		tagged[sim_const.LABEL_STRUCTURE] = self.getStructuresInArea(minX, minY, maxX, maxY, True)
		return tagged

	def hasObstaclesInArea(self, minX, minY, maxX, maxY):
		"""
		Tests if any obstacles exist in specified area.

		Returns C{True} if there are, C{False} otherwise.
		"""
		if self.getStructuresInAreaWithDensity(minX, minY, maxX, maxY, density=1.0):return True # needs to use navmesh here
		if self.getAgentsInArea(minX, minY, maxX, maxY):return True
		if self.getObjectsInArea(minX, minY, maxX, maxY):return True
		return False


	#############################
	# ARCHIVING
	#############################
	def encode(self, structures=True, objects=True, agents=False):
		"""
		Returns encoded model of data within partition.

		@rtype: dict
		"""
		encoded = {sim_const.LABEL_NAME:self.name,
			sim_const.LABEL_BOUNDS:self.bounds.encode(),
			sim_const.LABEL_SPACEX:self.spaceX,
			sim_const.LABEL_SPACEY:self.spaceY,
			sim_const.LABEL_MARGIN:self.margin}
		if structures:encoded[sim_const.LABEL_STRUCTURES] = self.encodeStructures()
		if objects:encoded[sim_const.LABEL_OBJECTS] = self.encodeObjects()
		if agents:encoded[sim_const.LABEL_AGENTS] = self.encodeAgents()
		return encoded

	def decode(self, data, terrain=None):
		"""
		Applies encoded data to self.
		"""
		bounds = AxialBounds2D()
		bounds.decode(data[sim_const.LABEL_BOUNDS])
		self.setup(name=data[sim_const.LABEL_NAME], 
			terrain=terrain,
			bounds=bounds,
			spaceX=data.get(sim_const.LABEL_SPACEX, None), 
			spaceY=data.get(sim_const.LABEL_SPACEY, None),
			margin=data.get(sim_const.LABEL_MARGIN, None))
		if data.has_key(sim_const.LABEL_STRUCTURES):
			self.decodeStructures(data[sim_const.LABEL_STRUCTURES])
		if data.has_key(sim_const.LABEL_OBJECTS):
			self.decodeObjects(data[sim_const.LABEL_OBJECTS])
		if data.has_key(sim_const.LABEL_AGENTS):
			self.decodeAgents(data[sim_const.LABEL_AGENTS])

	def encodeAgents(self):
		"""
		Returns encoded model of agents within partition.

		@rtype: dict
		"""
		encoded = []
		for agent in self.agents.values():encoded.append(agent.encode())
		return encoded

	def decodeAgents(self, data):
		"""
		Constructs classes from encoded data of agents within partition.

		@type data: dict
		"""
		for entry in data:
			agent = Agent()
			agent.decode(entry)
			self.addAgent(agent)

	def encodeObjects(self):
		"""
		Returns encoded model of objects within partition.

		@rtype: dict
		"""
		encoded = []
		for obj in self.objects.values():encoded.append(obj.encode())
		return encoded

	def decodeObjects(self, data):
		"""
		Constructs classes from encoded data of objects within partition.

		@type data: dict
		"""
		pass

	def encodeStructures(self):
		"""
		Returns encoded model of structures within partition.

		@rtype: dict
		"""
		encoded = []
		for structure in self.structures.values():encoded.append(structure.encode())
		return encoded

	def decodeStructures(self, data):
		"""
		Constructs classes from encoded data of structures within partition.

		@type data: dict
		"""
		for entry in data:
			structure = Structure()
			structure.decode(entry, self.margin)
			self.addStructure(structure)
