# svs_simulation.simobjects.basic_objects

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Objects within simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# internal imports
from svs_simulation.entities.base_entities import SpatialEntity
from svs_simulation.ai_lib.statemachine import StateManager
from svs_simulation.behaviours.behaviourlib import BehaviourHolder
from svs_simulation.numdata.geomlib import SphereBounds2D
from svs_simulation.utilities.constants import sim_const


#############################
# CLASSES
#############################
class SimObject(SpatialEntity):
	"""
	Basic class for objects with statemachine.
	"""
	def __init__(self):
		SpatialEntity.__init__(self)
		self.stateManager = StateManager(self)
		self.behaviourHolder = BehaviourHolder()

	def handleSimEvent(self, event):
		"""
		Handles event form simulation.
		"""
		print event

	#############################
	# BASIC INFO
	#############################
	def getName(self):
		"""
		Returns name for structure.  If the structure
		has no name retuens its _uid.
		"""
		if not self.name:return self._uid
		return self.name

	def getBoundsMinX(self):
		"""
		Returns minimum x coordinate from bounds.
		"""
		return self.bounds.minX

	def getBoundsMinY(self):
		"""
		Returns minimum y coordinate from bounds.
		"""
		return self.bounds.minY

	def getBoundsMaxX(self):
		"""
		Returns maximum x coordinate from bounds.
		"""
		return self.bounds.maxX

	def getBoundsMaxY(self):
		"""
		Returns maximum y coordinate from bounds.
		"""
		return self.bounds.maxY

	def containsPoint(self, x, y):
		"""
		Tests if coordinates are contaiend within structure bounds.
		"""
		return self.bounds.containsPointInWorld(x, y)

	def __str__(self):
		"""
		Returns string representation of object.
		"""
		if not self.name:return "sim object [%s]" % self._uid
		return "sim object [%s]" % self.name

	def getWorldbounds(self):
		"""
		Returns the bounds of the structure as mapped to the world coordinates.
		"""
		return self.bounds.worldbounds

	#############################
	# ARCHIVING
	#############################
	def setup(self, name=None):
		SpatialEntity.setup(self)
		self.name = name

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		data = {sim_const.LABEL_BOUNDS:self.bounds.encode()}
		if self.name:data[sim_const.LABEL_NAME] = self.name
		if self._uid:data[sim_const.LABEL_IDTAG] = self._uid
		data[sim_const.LABEL_FACING] = self.facing
		return data

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		if data.has_key(sim_const.LABEL_BOUNDS):
			self.bounds = SphereBounds2D()
			self.bounds.decode(data[sim_const.LABEL_BOUNDS])
			self.location.x = self.bounds.worldbounds.x
			self.location.y = self.bounds.worldbounds.y
		if data.has_key(sim_const.LABEL_LOCATION):self.setLocation(data[sim_const.LABEL_LOCATION][0],data[sim_const.LABEL_LOCATION][1])
		if data.has_key(sim_const.LABEL_FACING):self.facing = data[sim_const.LABEL_FACING]
		name = data.get(sim_const.LABEL_NAME, None)
		self.setup(name=name)

	#############################
	# PROCESS FUNCTIONS
	#############################
	def updateWorld(self, simTime):
		"""
		Forwards C{update} call from world.
		"""
		#print "SimObject updateWorld"
		pass

	def startWorld(self, simTime):
		"""
		Forwards C{start} call from world.
		"""
		#print "SimObject startWorld"
		pass

	def stopWorld(self, simTime):
		"""
		Forwards C{stop} call from world.
		"""
		#print "SimObject stopWorld"
		pass

class SimObjectProxy(SimObject):
	"""
	Lightweight client-side representation of object.
	"""
	def __init__(self):
		SimObject.__init__(self)
		self.sourceClass = None

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		#print "SimObjectProxy decode:", data
		SimObject.decode(self, data)
		self._uid = data.get(sim_const.LABEL_IDTAG, None)
		properties = self.__dict__.keys()
		for key, value in data.items():
			if not key in properties:setattr(self,key,value)

	def decodeChangeData(self, data):
		"""
		Applies data to self.
		"""
		newLoc = data.get(sim_const.LABEL_LOCATION, None)
		if newLoc:self.bounds.setLocation(newLoc[0], newLoc[1])
