# svs_simulation.numdata.matrices

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
2 dimensional matrix calculations.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
import math

class Matrix2D:
	"""
	Basic matrix class for 2D calculations.
	"""
	def __init__(self):
		self._11 = 0.0
		self._12 = 0.0
		self._13 = 0.0
		self._21 = 0.0
		self._22 = 0.0
		self._23 = 0.0
		self._31 = 0.0
		self._32 = 0.0
		self._33 = 0.0


	def identity(self):
		"""
		Create an identity matrix.
		"""
		self._11 = 1.0
		self._12 = 0.0
		self._13 = 0.0
		self._21 = 0.0
		self._22 = 1.0
		self._23 = 0.0
		self._31 = 0.0
		self._32 = 0.0
		self._33 = 1.0

	def matrixMultiply(self, otherMatrix):
		"""
		Multiply another matrix with this one.
		"""
		mat_temp = Matrix2D()
		# first row
		mat_temp._11 = (self._11*otherMatrix._11) + (self._12*otherMatrix._21) + (self._13*otherMatrix._31)
		mat_temp._12 = (self._11*otherMatrix._12) + (self._12*otherMatrix._22) + (self._13*otherMatrix._32)
		mat_temp._13 = (self._11*otherMatrix._13) + (self._12*otherMatrix._23) + (self._13*otherMatrix._33)
		# second
		mat_temp._21 = (self._21*otherMatrix._11) + (self._22*otherMatrix._21) + (self._23*otherMatrix._31)
		mat_temp._22 = (self._21*otherMatrix._12) + (self._22*otherMatrix._22) + (self._23*otherMatrix._32)
		mat_temp._23 = (self._21*otherMatrix._13) + (self._22*otherMatrix._23) + (self._23*otherMatrix._33)
		# third
		mat_temp._31 = (self._31*otherMatrix._11) + (self._32*otherMatrix._21) + (self._33*otherMatrix._31)
		mat_temp._32 = (self._31*otherMatrix._12) + (self._32*otherMatrix._22) + (self._33*otherMatrix._32)
		mat_temp._33 = (self._31*otherMatrix._13) + (self._32*otherMatrix._23) + (self._33*otherMatrix._33)
		self = mat_temp
  
	def translate(self, x, y):
		"""
		Transform matrix on x and y axes.

		@type x : float
		@type y : float
		"""
		mat = Matrix2D()
		mat._11 = 1.0
		mat._12 = 0.0
		mat._13 = 0.0
		mat._21 = 0.0
		mat._22 = 1.0
		mat._23 = 0.0
		mat._31 = x
		mat._32 = y
		mat._33 = 1.0
  		self.matrixMultiply(mat)

	def angleRotation(self, angle):
		"""
		Rotate matrix by specified angle.

		@type angle: float
		"""
		sin = math.sin(angle)
		cos = math.cos(angle)
		mat = Matrix2D()
		mat._11 = cos
		mat._12 = sin
		mat._13 = 0.0
		mat._21 = -sin
		mat._22 = cos
		mat._23 = 0.0
		mat._31 = x
		mat._32 = y
		mat._33 = 1.0
  		self.matrixMultiply(mat)

	def vectorRotation(self, fwdVector, sideVector):
		"""
		Rotate matrix by specified 2 vectors.

		@type fwdVector: L{svs_demogame.numdata.vectors.Vector2D}
		@type sideVector: L{svs_demogame.numdata.vectors.Vector2D}
		"""
		mat = Matrix2D()
		mat._11 = fwdVector.x
		mat._12 = fwdVector.y
		mat._13 = 0.0
		mat._21 = sideVector.x
		mat._22 = sideVector.y
		mat._23 = 0.0
		mat._31 = 0.0
		mat._32 = 0.0
		mat._33 = 1.0
  		self.matrixMultiply(mat)

	def scale(self, xScale, yScale):
		"""
		Scale matrix by specified dimensions.

		@type xScale: float
		@type yScale: float
		"""
		mat = Matrix2D()
		mat._11 = xScale
		mat._12 = 0.0
		mat._13 = 0.0
		mat._21 = 0.0
		mat._22 = yScale
		mat._23 = 0.0
		mat._31 = 0.0
		mat._32 = 0.0
		mat._33 = 1.0
  		self.matrixMultiply(mat)


	def transformVector2DList(self, vectors):
		"""
		Applies a 2D transformation matrix to an array of L{svs_demogame.numdata.vectors.Vector2D}s.
		"""
		for vector in vectors:
			tempX =(self._11*vector.x) + (self._21*vector.y) + (self._31)
			tempY = (self._12*vector.x) + (self._22*vector.y) + (self._32)
			vector.x = tempX;
			vector.y = tempY;

	def transformVector2D(self, vector):
		"""
		Applies a 2D transformation matrix to a single L{svs_demogame.numdata.vectors.Vector2D}.
		"""
		tempX =(self._11*vector.x) + (self._21*vector.y) + (self._31);
		tempY = (self._12*vector.x) + (self._22*vector.y) + (self._32);
		vector.x = tempX;
		vector.y = tempY;
