# svs_simulation.network.worldclients

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Specialised clients for running simulation world.


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from twisted.internet import reactor

# internal imports
from svs_simulation.network.simclients import SimulationClient
from svs_simulation.utilities.constants import sim_const
from svs_core.commands.scriptcommands import *
from svs_core.network.packets import *
from svs_simulation.world.worlds import createWorldFromSVSFile


class SimWorldClient(SimulationClient):
	"""
	Basic client for running simulation world.
	"""
	def __init__(self, name, passwd, tickDuration=1000):
		SimulationClient.__init__(self, name, passwd, tickDuration)
		self.world = None
		self.partitionClients = []
		self.running = False

	def loadWorld(self, filename):
		"""
		Load world info from local file.
		"""
		self.world = createWorldFromSVSFile(filename)
		#print "loadWorld:", self.world

	#############################
	# SIMULATION LOOP
	#############################
	def startWorld(self):
		"""
		Starts simulation loop in world.
		"""
		reactor.callLater(self.tickDelay, self.tick)
		self.world.startWorld()
		self.running = True


	def stopWorld(self):
		"""
		Stops simulation loop in world.
		"""
		self.running = False
		self.world.stopWorld()

	def tick(self):
		"""
		Responds to tick of internal timer.
		"""
		if not self.running:return
		self.world.updateWorld()
		# broadcast changes to clients
		changes = self.world.getChanges()
		changeData = []
		for change in changes:changeData.append(change.encode())
		if changes:self.notifyListeners(makeDataPacket(self.profile.name, content=changeData, label=sim_const.LABEL_UPDATE))
		# reset timer
		reactor.callLater(self.tickDelay, self.tick)

	#############################
	# PROCESS FUNCTIONS
	#############################
	def sendUpdateWorld(self, remoteClient, simTime):
		"""
		Sends C{update} to C{remoteClient}.
		"""
		self.sendData(makeDataPacket(sender=self.profile.name, recipient=remoteClient, content=simTime.encode(), label=sim_const.LABEL_UPDATEWORLD))
		
	def sendStartWorld(self, remoteClient, simTime):
		"""
		Sends C{startWorld} to C{remoteClient}.
		"""
		self.sendData(makeDataPacket(sender=self.profile.name, recipient=remoteClient, content=simTime.encode(), label=sim_const.LABEL_STARTWORLD))
		
	def sendStopWorld(self, remoteClient, simTime):
		"""
		Sends C{stopWorld} to C{remoteClient}.
		"""
		self.sendData(makeDataPacket(sender=self.profile.name, recipient=remoteClient, content=simTime.encode(), label=sim_const.LABEL_STOPWORLD))

	#############################
	# EVENT FUNCTIONS
	#############################
	def sendNotifyOfEvents(self, remoteClient, events):
		"""
		Sends events to C{remoteClient}.
		"""
		content = []
		for event in events:content.append(event.encode())
		self.sendData(makeDataPacket(sender=self.profile.name, recipient=remoteClient, content=content, label=sim_const.LABEL_NOTIFYOFEVENTS))
		

	#############################
	# WORLD DATA
	#############################
	def getWorldData(self):
		"""
		Returns basic data for world.
		"""
		return makeDataPacket(self.profile.name, content=self.world.encode(), label=sim_const.LABEL_WORLD)

	def getPartitionModel(self, partitionName):
		"""
		Returns partition with specified name.
		"""
		print "getPartitionModel:", partitionName
		#if partitionName in self.partitionClients: return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_PARTITION) # should be error
		partition = self.world.terrain.getPartition(partitionName)
		if not partition:return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_PARTITION) # should be error
		#self.partitionClients.append(partitionName) # remember to release on disconnection
		return makeDataPacket(self.profile.name, content=partition.encode(), label=sim_const.LABEL_PARTITION)

	def getTerrainModel(self, lod=0):
		"""
		Returns model of terrain, this can be tailored in detail 
		according to the C{LOD} argument.
		"""
		if not lod:return makeDataPacket(self.profile.name, content=self.world.terrain.encodeForProxy(), label=sim_const.LABEL_TERRAIN)
		else:return makeDataPacket(self.profile.name, content=self.world.terrain.encode(), label=sim_const.LABEL_TERRAIN)


	def getAgentGroups(self):
		"""
		Returns a list of agent groups in world.
		"""
		content = []
		agentGroups = self.world.getAgentGroups()
		for agentGroup in agentGroups.values():
			content.append(agentGroup.name)
		return makeDataPacket(self.profile.name, content=content, label=sim_const.LABEL_AGENTGROUPS)


	def getAgentGroup(self, groupName):
		"""
		Returns agent group with specified name.
		"""
		agentGroup = self.world.getAgentGroup(groupName)
		if not agentGroup:return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_AGENTGROUP) # should be error
		return makeDataPacket(self.profile.name, content=agentGroup.encode(), label=sim_const.LABEL_AGENTGROUP)


	def getObjectGroups(self):
		"""
		Returns a list of object groups in world.
		"""
		content = []
		objectGroups = self.world.getObjectGroups()
		for objectGroup in objectGroups.values():
			content.append(objectGroup.name)
		return makeDataPacket(self.profile.name, content=content, label=sim_const.LABEL_OBJECTGROUPS)


	def getObjectGroup(self, groupName):
		"""
		Returns object group with specified name.
		"""
		objectGroup = self.world.getObjectGroup(groupName)
		if not objectGroup:return makeDataPacket(self.profile.name, content=None, label=sim_const.LABEL_OBJECTGROUP) # should be error
		return makeDataPacket(self.profile.name, content=objectGroup.encode(), label=sim_const.LABEL_OBJECTGROUP)
			
	#############################
	# DATA HANDLING
	#############################
	def getDataForLabel(self, dataRequest):
		"""
		Provides custom handling of data requests.

		This should be overidden by extending classes.
		"""
		if not dataRequest.label:return ScriptableClient.getDataForLabel(dataRequest)
		# get world data
		if dataRequest.label == sim_const.LABEL_WORLD:
			return self.getWorldData()
		# get terrain data
		if dataRequest.label == sim_const.LABEL_TERRAIN:
			return self.getTerrainModel(dataRequest.args[sim_const.LABEL_LOD])
		# get partition data
		if dataRequest.label == sim_const.LABEL_PARTITION:
			return self.getPartitionModel(dataRequest.args[sim_const.LABEL_NAME])
		# get agent groups list
		if dataRequest.label == sim_const.LABEL_AGENTGROUPS:
			return self.getAgentGroups()
		# get agent group
		if dataRequest.label == sim_const.LABEL_AGENTGROUP:
			return self.getAgentGroup(dataRequest.args[sim_const.LABEL_NAME])
		# get object groups list
		if dataRequest.label == sim_const.LABEL_OBJECTGROUPS:
			return self.getObjectGroups()
		# get object group
		if dataRequest.label == sim_const.LABEL_OBJECTGROUP:
			return self.getObjectGroup(dataRequest.args[sim_const.LABEL_NAME])
		return SimulationClient.getDataForLabel(self, dataRequest)


	#############################
	# CHAT
	#############################
	def remote_receiveChatMessage(self, msgPacket):
		"""
		Receives chat message from other clients.

		Chat messages are used as a way of handling remote commands from other clients.

		NOTE: command messages should be passed to command handler in future versions.
		"""
		msgText = "message from <%s>:\n%s\n" % (msgPacket.sender, msgPacket.content)
		self.logMessage(msgText)
		if len(msgPacket.recipient) != 1:return
		if msgPacket.recipient[0] == self.profile.name:
			msg = msgPacket.content.lower()
			if msg == 'start':
				self.startWorld()
			elif msg == 'stop':
				self.stopWorld()
