# svs_simulation.network.playerclients

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Player clients that can interact with a simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.network.packets import *
from svs_core.commands.scriptcommands import *
from svs_core.utilities.constants import svs_const
from svs_simulation.network.simclients import GUISimulationClient
from svs_simulation.utilities.constants import sim_const
from svs_simulation.world.proxyworlds import SimWorldProxy


#############################
# VISITOR CLIENT
#############################
class VisitorClient:
	"""
	Client that can retrieve data from the terrain and partitions.
	"""
	def __init__(self, tickDuration=1000):
		self.tickDuration = tickDuration
		self.lod = 0
		self.worldClientName = None
		self.world = SimWorldProxy()


	#######################
	# WORLD SETUP
	#######################
	def setupWorld(self, worldData):
		"""
		Handles basic info about world.
		"""
		self.world.setupWorld(worldData)
		self.displayWorldInfo()

	def setupTerrain(self, model):
		"""
		Creates terrain from received data.
		"""
		self.world.setupTerrain(model, self.lod)
		self.displayPartitionInfo()

	def setupPartition(self, model):
		"""
		Creates partition from received data.
		"""
		self.world.setupPartition(model)
		self.displayPartitionInfo()

	def setupAgentGroups(self, groups):
		"""
		Handles received list of agent groups.
		"""
		self.world.setupAgentGroups(groups)
		self.displayAgentGroupsInfo()

	def setupAgentGroup(self, groupData):
		"""
		Handles received data for agent group.
		"""
		self.world.setupAgentGroup(groupData)
		self.displayAgentGroupInfo(groupData[sim_const.LABEL_NAME])

	def setupObjectGroups(self, groups):
		"""
		Handles received list of object groups.
		"""
		self.world.setupObjectGroups(groups)
		self.displayObjectGroupsInfo()

	def setupObjectGroup(self, groupData):
		"""
		Handles received data for object group.
		"""
		self.world.setupObjectGroup(groupData)
		self.displayObjectGroupsInfo()



	#######################
	# INFO DISPLAY
	#######################
	def displayWorldInfo(self):
		"""
		Displays info about world.
		"""
		worldStr = "World:\n"
		if not self.world.name:worldStr += "  world not loaded."
		else:
			worldStr += "  name: %s" % self.world.name
		self.statusMessage(worldStr)
		

	def displayPartitionInfo(self):
		"""
		Displays info about partitions.
		"""
		if not self.world.terrain:return "No partitions available."
		partitionStr = "Terrain partitions:\n"
		for partitionName, data in self.world.terrain.partitions.items():
			if data:partitionStr += "  %s - loaded\n" % partitionName
			else:partitionStr += "  %s - not loaded\n" % partitionName
		self.statusMessage(partitionStr)

	def displayAgentGroupsInfo(self):
		"""
		Displays info about agent groups.
		"""
		agentGroups = self.world.getAgentGroups()
		if not agentGroups:
			self.statusMessage("No agent groups.")
			return
		groupStr = "Agent groups:\n"
		for group in agentGroups.keys():groupStr += "  %s\n" % group
		self.statusMessage(groupStr)


	def displayAgentGroupInfo(self, groupName):
		"""
		Displays info about specific agent group.
		"""
		agentGroup = self.world.getAgentGroup(groupName)
		if not agentGroup:
			self.statusMessage("No agent group with name '%s'." % groupName)
			return
		groupStr = "Agent group <%s>:\n" % agentGroup.name
		agents = agentGroup.getMembers()
		for agent in agents:groupStr += "  %s\n" % agent.getName()
		self.statusMessage(groupStr)

	def displayObjectGroupsInfo(self):
		"""
		Displays info about object groups.
		"""
		objectGroups = self.world.getObjectGroups()
		if not objectGroups:
			self.statusMessage("No object groups.")
			return
		groupStr = "Object groups:\n"
		for group in objectGroups.keys():groupStr += "  %s\n" % group
		self.statusMessage(groupStr)

	#######################
	# NETWORK
	#######################
	def avatarResult_joinClusterGroup(self, group):
		"""
		Handles result of L{joinClusterGroup} action.
		"""
		self.clusterGroup = group
		self.logMessage("joined cluster group <%s>" % self.clusterGroupName)
		self.getWorldData()
		self.getTerrainModel(self.lod)

	#######################
	# WORLD DATA
	#######################
	def getWorldData(self):
		"""
		Retrieves absic info about world.
		"""
		if not self.worldClientName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldClientName, label=sim_const.LABEL_WORLD)
		self.getData(dataRequest)


	def getTerrainModel(self, lod):
		"""
		Retrieves model of partition from terrain client.
		"""
		if not self.worldClientName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldClientName, label=sim_const.LABEL_TERRAIN, args={sim_const.LABEL_LOD:lod})
		self.getData(dataRequest)
		self.startListeningTo(self.worldClientName, listenFor=sim_const.LABEL_UPDATE)

	def getPartitionModel(self, partitionName):
		"""
		Retrieves model of partition from terrain client.
		"""
		if not self.worldClientName:return
		#dataRequest = makeDataRequest(self.profile.name, recipient=partitionName, label=sim_const.LABEL_PARTITION, args={sim_const.LABEL_LOD:self.lod})
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldClientName, label=sim_const.LABEL_PARTITION, args={sim_const.LABEL_NAME:partitionName})
		self.getData(dataRequest)
		#self.startListeningTo(partitionName, listenFor=sim_const.LABEL_UPDATE)

	def getAgentGroups(self):
		"""
		Retrieves list of agent groups from world.
		"""
		if not self.worldClientName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldClientName, label=sim_const.LABEL_AGENTGROUPS)
		self.getData(dataRequest)

	def getAgentsInGroup(self, groupName):
		"""
		Retrieves agents in specified group.
		"""
		if not self.worldClientName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldClientName, label=sim_const.LABEL_AGENTGROUP, args={sim_const.LABEL_NAME:groupName})
		self.getData(dataRequest)

	def getObjectGroups(self):
		"""
		Retrieves list of object groups from world.
		"""
		if not self.worldClientName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldClientName, label=sim_const.LABEL_OBJECTGROUPS)
		self.getData(dataRequest)

	def getObjectsInGroup(self, groupName):
		"""
		Retrieves objects in specified group.
		"""
		if not self.worldClientName:return
		dataRequest = makeDataRequest(self.profile.name, recipient=self.worldClientName, label=sim_const.LABEL_OBJECTGROUP, args={sim_const.LABEL_NAME:groupName})
		self.getData(dataRequest)

	#############################
	# PROCESS HANDLERS
	#############################
	def handleUpdateWorld(self, data):
		"""
		Handles update data from world.
		"""
		self.world.handleUpdateWorld(data)


	#######################
	# DATA HANDLING
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.

		This should be overidden by extending classes.
		""" 
		if not dataPacket:return
		dataLabel = dataPacket.label
		if not dataLabel:return
		# update
		if dataLabel == sim_const.LABEL_UPDATE:
			self.handleUpdateWorld(dataPacket.content)
		# world data
		elif dataLabel == sim_const.LABEL_WORLD:
			self.setupWorld(dataPacket.content)
		# terrain model
		elif dataLabel == sim_const.LABEL_TERRAIN:
			self.setupTerrain(dataPacket.content)
		# partition model
		elif dataLabel == sim_const.LABEL_PARTITION:
			self.setupPartition(dataPacket.content)
		# agent groups
		elif dataLabel == sim_const.LABEL_AGENTGROUPS:
			self.setupAgentGroups(dataPacket.content)
		# agent group
		elif dataLabel == sim_const.LABEL_AGENTGROUP:
			self.setupAgentGroup(dataPacket.content)
		# object groups
		elif dataLabel == sim_const.LABEL_OBJECTGROUPS:
			self.setupObjectGroups(dataPacket.content)
		# object group
		elif dataLabel == sim_const.LABEL_OBJECTGROUP:
			self.setupObjectGroup(dataPacket.content)
		else:self.logMessage("data packet received: <%s>" % dataPacket.content)

	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_partition(self, cmd):
		"""
		Get model for named partition.
		"""
		try:
			partitionName = cmd.args[0]
		except IndexError:
			return makeCommandResult(cmd, message="partition <partitionName>", status=svs_const.ERROR)

		self.getPartitionModel(partitionName)
		return makeCommandResult(cmd, message="getting partition '%s' ..." % partitionName, status=svs_const.OK)

	def cmdprivate_agentgroups(self, cmd):
		"""
		Get list of agent groups.
		"""
		self.getAgentGroups()
		return makeCommandResult(cmd, message="getting agent groups ...", status=svs_const.OK)

	def cmdprivate_agents(self, cmd):
		"""
		Get agents for named group.
		"""
		try:
			groupName = cmd.args[0]
		except IndexError:
			return makeCommandResult(cmd, message="agents <groupName>", status=svs_const.ERROR)

		self.getAgentsInGroup(groupName)
		return makeCommandResult(cmd, message="getting agents in group '%s' ..." % groupName, status=svs_const.OK)

	def cmdprivate_objectgroups(self, cmd):
		"""
		Get list of object groups.
		"""
		self.getObjectGroups()
		return makeCommandResult(cmd, message="getting object groups ...", status=svs_const.OK)

	def cmdprivate_objects(self, cmd):
		"""
		Get objects for named group.
		"""
		try:
			groupName = cmd.args[0]
		except IndexError:
			return makeCommandResult(cmd, message="objects <groupName>", status=svs_const.ERROR)

		self.getObjectsInGroup(groupName)
		return makeCommandResult(cmd, message="getting objects in group '%s' ..." % groupName, status=svs_const.OK)


#############################
# GUI VISITOR CLIENT
#############################
class GUIVisitorClient(VisitorClient, GUISimulationClient):
	"""
	Client with basic SVS GUI that can retrieve data from the terrain and partitions.
	"""
	def __init__(self, name, passwd, tickDuration=1000,):
		VisitorClient.__init__(self, tickDuration=tickDuration)
		GUISimulationClient.__init__(self, name, passwd)


#############################
# PLAYER CLIENT
#############################
class PlayerClient(VisitorClient):
	"""
	Basic client that can interact with simulation world.
	"""
	def __init__(self, tickDuration=1000):
		VisitorClient.__init__(self, tickDuration=tickDuration)
