# svs_simulation.events.eventlib

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Main classes representing and handling simulation events.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from time import time

# internal imports
from svs_core.utilities.notification import Listenable
from svs_simulation.utilities.constants import sim_const


#######################
# EXCEPTIONS
#######################
class SimEventException(Exception):pass


#######################
# CLASSES
#######################
class SimEvent:
	"""
	Base class for events within simulation.
	"""
	def __init__(self, label=None, source=None, data=None):
		self.realtimestamp = time()
		self.simtimestamp = None
		self.source = source
		self.data = data
		self.label = label

	def __str__(self):
		"""
		Returns string representation of self.
		"""
		return "%s <%s>" % ("SimEvent", self.label)

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype: dict
		"""
		data = {sim_const.LABEL_SOURCE:self.source,
			sim_const.LABEL_DATA:self.data,
			sim_const.LABEL_EVENTTAG:self.label}
		if self.simtimestamp:data[sim_const.LABEL_TIME] = self.simtimestamp.encode()
		return data

	def decode(self, data):
		"""
		Applies encoded representation to self.
		"""
		if data.has_key(sim_const.LABEL_TIME):self.timestamp = data[sim_const.LABEL_TIME]
		if data.has_key(sim_const.LABEL_SOURCE):self.source = data[sim_const.LABEL_SOURCE]
		if data.has_key(sim_const.LABEL_DATA):self.data = data[sim_const.LABEL_DATA]
		if data.has_key(sim_const.LABEL_EVENTTAG):self.label = data[sim_const.LABEL_EVENTTAG]




class SimEventCentre(Listenable): # will need proxy for networking
	"""
	This provides coordination between agents and events within the simulation.
	"""
	def __init__(self):
		Listenable.__init__(self)
		self.eventQueue = {}

	def postEvent(self, event):
		"""
		Receives event to be sent on to listeners.
		"""
		if event == None:return
		if not self.eventQueue.has_key(event.label):self.eventQueue[event.label] = []
		self.eventQueue[event.label].append(event)

	def getEvents(self, eventLabel, flush=False):
		"""
		Returns events from queue that match specified label.
		"""
		events = self.eventQueue.get(eventLabel, None)
		if flush:self.eventQueue[eventLabel] = []
		return events

	def clearEvents(self, eventLabel):
		"""
		Clears and returns events from queue that match specified label.
		"""
		events = self.eventQueue.get(eventLabel, None)
		if events:self.eventQueue[eventLabel] = []
		return events

	def updateWorld(self, simtime=None):
		"""
		Updates event centre, passing on all recently received events to listeners.
		"""
		for label in self.eventQueue.keys():
			events = self.clearEvents(label)
			if events:self.notifyListeners(label, events)


class SimEventCentreProxy: 
	"""
	Proxy class that provides local interface to networked event centre.
	"""
	def __init__(self, client):
		self.client = client

	def postEvent(self, event):
		"""
		Receives event to be sent on to listeners.
		"""
		if event == None:return
		self.client.sendData() # make event packet
