# svs_simulation.entities.base_entities

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Base classes for simulation.

This module is partly based on the exmaples in
Mat Buckland, 2005, Programming Game AI by Example, Wordware:Plano, 
see U{http://www.wordware.com/files/ai}, and U{http://www.ai-junkie.com}.

This version does not support space partitioning.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.numdata.vectors import Vector2D
from svs_simulation.numdata.geomlib import geom_const, AxialBounds2D, CentredAxialBounds2D, SphereBounds2D
from svs_simulation.utilities.constants import sim_const
from svs_simulation.events.eventlib import SimEvent



#############################
# BASE CLASS
#############################
class Entity:
	"""
	Generic entity for simulation.  All simulation entities
	are derived from this.
	"""
	def __init__(self):
		self._uid = None
		self.tag = False
		self.name = None
		self.processHandler = None
		self.partition = None

	def getName(self):
		"""
		If the entity has a name, return this,
		otherwise returns a name created from
		its class name and unique id.
		"""
		if not self.name:self.name = "%s_%d" % (self.__class__.__name__, self._uid)
		return self.name

	def isTagged(self):
		"""
		Returns tag state.

		@rtype: boolean
		"""
		return self.tag

	#############################
	# PROCESS FUNCTIONS
	#############################
	def startWorld(self, simTime):
		"""
		Called when the simulation world starts running,
		or if an entity has been added to a world after it
		has started.
		"""
		pass

	def stopWorld(self, simTime):
		"""
		Called when the simulation world stops running,
		or if an entity has been removed from a world after it
		has started.
		"""
		pass

	def updateWorld(self, simTime):
		"""
		Called when the simulation world updates.
		"""
		pass

	def handleSimEvent(self, event):
		"""
		Handles event form simulation.

		This should be overridden by extending classes.
		"""
		pass

	#############################
	# PROCESS HANDLING
	#############################
	def reportChange(self, data):
		"""
		Forwards data representing change in entity to process handler.

		Changes are handled as events.
		"""
		if not self.processHandler:return
		self.processHandler.postEvent(SimEvent(label=sim_const.LABEL_UPDATE, source=self._uid, data=data))

	############################
	# WORLD
	############################
	def enterWorld(self, world):
		"""
		Called when entity is added to world.
		"""
		self.world = world
		self.terrain = self.world.terrain
		self.placeInWorld()
		if self.world.running:self.startWorld(self.world.getCurrentTime())

	def exitWorld(self, simTime):
		"""
		Called when structure is removed world.
		"""
		self.stopWorld(simTime)
		self.world = None

	def placeInWorld(self):
		"""
		Sets the structure and partition that the entity is within.
		"""
		structure = self.terrain.getStructureContainingPoint(self.location.x, self.location.y)
		partition = structure.partition
		if partition != self.partition:
			self.exitPartition()
			self.enterPartition(partition)
		if structure != self.structure:
			self.exitStructure()
			self.enterStructure(structure)

	############################
	# PARTITIONS
	############################
	def enterPartition(self, partition):
		"""
		Called when entity enters a partition.
		"""
		self.partition = partition
		self.partition.addEntity(self)

	def exitPartition(self, partition=None):
		"""
		Called when entity is leaves a partition.
		"""
		if not self.partition:return
		self.partition.removeEntity(self)
		self.partition = None

	############################
	# STRUCTURES
	############################
	def enterStructure(self, structure):
		"""
		Called when entity enters a structure.
		"""
		self.structure = structure
		self.structure.addEntity(self)

	def exitPartition(self, partition=None):
		"""
		Called when entity is leaves a structure.
		"""
		if not self.structure:return
		self.structure.removeEntity(self)
		self.structure = None

	
	#############################
	# ARCHIVING
	#############################
	def setup(self):
		self.tag = False

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		return {}

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		self.setup()
		

class BasicSpatialEntity(Entity):
	"""
	A spatial entity is one which has spatial location and size.
	"""
	def __init__(self, _uid=None):
		Entity.__init__(self)
		self.bounds = AxialBounds2D()

	def setup(self, _uid=None):
		Entity.setup(self)

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		return {sim_const.LABEL_IDTAG:self._uid, 
			sim_const.LABEL_BOUNDS:self.bounds.encode()}

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		self.bounds = AxialBounds2D()
		self.bounds.decode(data[sim_const.LABEL_BOUNDS])
		self.setup(data[sim_const.LABEL_IDTAG])


class SpatialEntity(BasicSpatialEntity):
	"""
	A spatial entity is one which has spatial location and size.
	"""
	def __init__(self, _uid=None):
		Entity.__init__(self) # NOTE: calls Entity constructor 
		#self.bounds = CentredAxialBounds2D()
		self.bounds = SphereBounds2D()
		self.location = Vector2D(0.0, 0.0)
		self.terrain = None
		self.partition = None
		self.structure = None
		self.facing = 0
		self.name = None

	def setDimensions(self, dimX, dimY):
		self.bounds.setDimensions(dimX, dimY)

	def setLocation(self, x, y):
		self.location.x = x
		self.location.y = y
		self.bounds.setLocation(x, y)

	############################
	# WORLD
	############################
	def enterWorld(self, world):
		"""
		Called when entity is added to world.
		"""
		self.world = world
		self.locateInWorld()
		if self.world.running:self.startSim(self.world.getCurrentTime())

	def locateInWorld(self):# !!! this maybe needs changed
		if not self.world:return
		structure = self.world.terrain.getStructureContainingPoint(self.bounds.worldbounds.x, self.bounds.worldbounds.y)
		if self.structure != structure:
			self.exitStructure()
			self.enterStructure(structure)

	############################
	# STRUCTURES
	############################
	def enterStructure(self, structure):
		"""
		Called when entity enters a new structure.
		"""
		self.structure = structure
		#print "enterStructure:", self.structure #### FIX ERROR !!!!
		#return
		if not self.structure:return
		if self.partition != self.structure.partition:
			#self.exitPartition()
			self.enterPartition(self.structure.partition)
		#print "self.partition:", self.partition

	def exitStructure(self, structure=None):
		"""
		Called when entity leaves an existing structure.
		"""
		self.structure = None

	############################
	# LOCATION
	############################
	def isOverPoint(self, x, y):
		"""
		Tests if the given coordinates lie within the
		world bounds of the entity.
		"""
		return self.bounds.containsPointInWorld(x,y)

		
#############################
# MOVING ENTITY
#############################
class MovingEntity_OLD(Entity): # NOTE: can probably drop this
	"""
	Entity with basic movement capabilities.

	@ivar self.velocity:velocity of entity.
	@type self.velocity: L{svs_core.geometry.vectors.Vector2D}
	@ivar self.heading:a normalized vector pointing in the direction the entity is heading.
	@type self.heading: L{svs_core.geometry.vectors.Vector2D}
	@ivar self.side:a vector perpendicular to the heading vector.
	@type self.side: L{svs_core.geometry.vectors.Vector2D}
	@ivar self.mass:mass of entity.
	@type self.mass: float
	@ivar self.maxSpeed:the maximum speed this entity may travel at.
	@type self.maxSpeed: float
	@ivar self.maxForce:the maximum force this entity can produce to power itself.
	@type self.maxForce: float
	@ivar self.maxTurnRate:the maximum rate (radians per second)this vehicle can rotate.
	@type self.maxTurnRate: float
	@ivar self.threatRange:distance within which another entity must be for evasion behaviour.
	@type self.threatRange: float
	@ivar self.targetPoint:target point towards which the vehicle is moving.
	@type self.targetPoint: L{svs_core.geometry.vectors.Vector2D}
	"""
	def __init__(self):
		Entity.__init__(self)
		self.velocity = Vector2D(0.0, 0.0)
		self.heading = Vector2D(0.0, 0.0)
		self.side = Vector2D(0.0, 0.0)
		self.mass = 1.0
		self.maxSpeed = 0.0
		self.maxForce = 0.0
		self.maxTurnRate = 0.0
		self.threatRange = 100.0
		self.targetPoint = None
  
	def isSpeedMaxedOut(self):
		"""
		Checks if speed is at maximum.
		
		@rtpe: boolean
		"""
		return self.maxSpeed * self.maxSpeed >= self.velocity.lengthSq()

	def speed(self):
		"""
		Return current speed.
		
		@rtpe: float
		"""
		return self.velocity.length()

	def speedSq(self):
		"""
		Return current speed squared.
		
		@rtpe: float
		"""
		return self.velocity.lengthSq()

	def rotateHeadingToFacePosition(self, target):
		"""
		Given a target position, this method rotates the entity's heading and
		side vectors by an amount not greater than L{MovingEntity.maxTurnRate} until it
		directly faces the target.

		@type target: L{svs_core.geometry.vectors.Vector2D}
		@return: true when the heading is facing in the desired direction
		@rtype: boolean
		"""
		toTarget = target - self.location
		toTarget.normalize()
		# first determine the angle between the heading vector and the target
		angle = math.acos(self.heading.dot(toTarget))
		# return true if the player is facing the target
		if angle < 0.00001: return True
		# clamp the amount to turn to the max turn rate
		if angle > self.maxTurnRate: angle = self.maxTurnRate
		# The next few lines use a rotation matrix to rotate the player's heading
		# vector accordingly
		rotationMatrix = Matrix2D()
		# notice how the direction of rotation has to be determined when creating
		# the rotation matrix
		rotationMatrix.angleRotation(angle * self.heading.sign(toTarget))	
		rotationMatrix.transformVector2D(self.heading)
		rotationMatrix.transformVector2D(self.velocity)
		# finally recreate m_vSide
		self.side = self.heading.perp()
		return False


	def setHeading(self, newHeading):
		"""
		First checks that the given heading is not a vector of zero length. If the
		new heading is valid this function sets the entity's heading and side 
		vectors accordingly.

		@type newHeading: L{svs_core.geometry.vectors.Vector2D}
		"""
		try:assert( (newHeading.lengthSq() - 1.0) < 0.00001)
		except AssertionError:return
		self.heading = newHeading
		# the side vector must always be perpendicular to the heading
		self.side = self.heading.perp()


#############################
# ENTITY GROUP
#############################
class EntityGroup(Entity):
	"""
	Basic entity group.  Used to gather entities into a single group that can be treated as
	an entity in itself.
	"""
	def __init__(self, name=None):
		Entity.__init__(self)
		self.members = {}
		self.name = name

	def __str__(self):
		"""
		Returns string representation of object.
		"""
		if not self.name:return "entity group [%s]" % self._uid
		return "entity group [%s]" % self.name

	def addMember(self, member):
		"""
		Adds member to group, making sure it is unique.
		"""
		name = member.getName()
		#print "addMember: %s, %s" % (member, name)
		if self.members.has_key(name):return
		self.members[name] = member

	def removeMember(self, member):
		"""
		Removes member from group.
		"""
		if self.members.has_key(member._uid):self.members.remove(member._uid)

	def getMembers(self):
		"""
		Returns all memebrs as a list.
		"""
		return self.members.values()

	#############################
	# ARCHIVING
	#############################
	def setup(self, name=None):
		EntityGroup.setup(self, name)
		self.name = name

	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		data = {sim_const.LABEL_NAME:self.name}
		memberList = []
		for member in self.members.values():memberList.append(member.encode())
		data[sim_const.LABEL_MEMBERS] = memberList
		return data

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		name = data.get(sim_const.LABEL_NAME, None)
		self.setup(name=name)
