# svs_simulation.behaviours.behaviourlib

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Base classes for behaviours within a simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from time import time

# internal imports
from svs_simulation.utilities.constants import sim_const


#######################
# BEHAVIOUR CLASSES
#######################
class Behaviour:
	"""
	Basic behaviour class.
	"""
	def __init__(self, name=None, simtime=None):
		self.name = name
		self.version = 1
		self.creationDate = time()
		self.simCreationDate = simtime
		self.usage = 0
		self.triggers = []
		self.actions = []
		self.outputData = None
		self.agent = None

	def trigger(self, inputEvent):
		"""
		Triggers behaviour in response to input.
		"""
		if not inputEvent.label in self.triggers:return
		if self.meetsConditions(inputEvent):return self.doActions(inputEvent)
		return None

	def meetsConditions(self, inputEvent):
		"""
		Determines if input matches behaviour conditions.
		"""
		return False

	def doActions(self, inputEvent):
		"""
		Performs actions for behaviour.
		"""
		return SimEvent(label=self.label, source=self.agent, data=self.outputData)

		
class BehaviourHandler:
	"""
	Maintains a set of behaviours and manages access to them.
	"""
	def __init__(self, agent, eventCentre): 
		from svs_simulation.time.schedules import ScheduleManager
		self.agent = agent
		self.eventCentre = eventCentre
		self.behaviours = []
		self.triggers = {}

	def addBehaviour(self, behaviour):
		"""
		Adds behaviour for handling making sure that it is not already present.
		"""
		if behaviour in self.behaviours:return
		behaviour.agent = self.agent
		self.behaviours.append(behaviour)
		for trigger in behaviour.triggers:self.__addToTriggerGroup(trigger, behaviour)
			

	def __addToTriggerGroup(self, trigger, behaviour):
		"""
		Adds behaviour to trigger group.
		
		This is a private method called by L{BehaviourHandler.addBehaviour}.
		"""
		if not self.triggers.has_key(trigger):self.triggers[trigger] = []
		self.triggers[trigger].append(behaviour)

	def handleInputEvent(self, inputEvent):
		"""
		Receives input event and triggers corresponding behaviours.
		"""
		behaviours = self.triggers.get(inputEvent.label, None)
		if not behaviours:return
		for behaviour in behaviours:self.eventCentre.postEvent(behaviour.trigger(inputEvent))
		

#######################
# STORAGE CLASSES
#######################
class BehaviourHolder:
	"""
	Holds a single behaviour and controls who can access it.

	NOTE: this should evolve into wiki integration
	"""
	def __init__(self):	
		self.behaviourHistory = []
		self.accessRules = None

	def requestBehaviour(self, requester, version=-1):
		if not self.accessRules:return self.behaviour
		for rule in self.accessRules:
			if rule.accessAllowed(requester):return self.getBehaviourVersion(version)
		return None
		
	def getBehaviourVersion(self, version):
		"""
		Returns specific version of behaviour.
		"""
		if version == -1:return self.behaviourHistory[:-1]
		if version > len(self.behaviourHistory):return self.behaviourHistory[:-1]
		if version < 0:return self.behaviourHistory[0]
		return self.behaviourHistory[version]

	def getBehaviourHistory(self):
		"""
		Returns history of behaviour.
		"""
		return self.behaviourHistory


class BehaviourWiki(BehaviourHolder):
	"""
	Extends L{BehaviourHolder} to provide wiki integration.
	"""
	def __init__(self, wikiURL):
		BehaviourHolder.__init__(self)
		self.wikiURL = wikiURL
	

#######################
# ACCESS RULES
#######################
class BehaviourAccessRule:
	"""
	Determines which agents can access a behaviour.

	This should be extended to provide more complex access rules.
	"""
	def __init__(self):
		self.allowedNames = []

	def accessAllowed(self, requester):
		if requester.name in self.allowedNames:return True
		return False


#############################
# FUNCTIONS
#############################
def createBehaviourFromSVSFile(filename, behaviourClass=Behaviour):
	"""
	Reads in data file for behaviour and creates it.
	
	@rtype: L{Behaviour}
	"""
	data = PersistentObject()
	data.read(filename)
	behaviour = behaviourClass()
	behaviour.decode(data.behaviour)
	return behaviour
