# svs_simulation.ai_lib.statemachine

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Operational code and management classes for parallel state machines.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

class StateException(Exception):
    """
    Handles exceptions created by state machine operations.
    """
    pass
    
    
class StateManager:
    """
    Provides management handlers for parallel state machine.
    
    Based on Sean Riley's example in I{Game Programming with Python}, Charles River Media, 2004.
    
    @type	layers: dict
    @cvar	layers: list of layers for parallel state machine.
    @type	agent: object
    @ivar	agent: agent object to whom state manager applies.
    @type	currentStates: dict
    @ivar	currentStates: list of current states.
    """
    
    #layers = {}
    
    def __init__(self, agent):
        """
        Initialises state manager.
        
        @type 	agent: object
        @param 	agent: agent object to whom state manager applies.
        """
        self.agent = agent
        self.currentStates = {}
        self.stateProperties = {}
        self.layers = {}
        
    
    def setInitialState(self, stateName, layerName):
        """
        Sets initial state for agent.
        
        @type 	stateName: string
        @param 	stateName: name of state
        @type 	layerName: string
        @param 	layerName: name of layer that state belongs to
        """
        layer = self.layers.get(layerName, None)
        
        if layer:
            state = layer.get(stateName)
            if state:
                self.currentStates[layerName] = state
                #print "______________INITIAL STATE: %s" % state.name
                return 1
        raise StateException("No state %s or layer %s" % (stateName, layerName))
        
    def setInput(self, input, layerName = None):
        """
        Sends an input event.  Can be sent directly to a specified layer.  
        If layer is not specified, seraches for appropriate layer.
        
        @type 	input: object
        @param 	input: input to respond to
        @type 	layerName: string
        @param 	layerName: name of layer
        """
        if layerName == None:
            # try each of the current states
            for key in self.currentStates.keys():
                state = self.currentStates[key]
                newStateName = state.inputs.get(input)
                if newStateName:
                    return self.gotoState(newStateName, key)
            return
	else:
		key = layerName
            
        currentState = self.currentStates.get(layerName)
        if not currentState:
            raise StateException("No layer %s" % layerName)
        
        newStateName = currentState.inputs.get(input)
        if newStateName:
            return self.gotoState(newStateName, key)
            
    def gotoState(self, newStateName, layerName):
        """
        Set specified layer to new state.
        
        @type 	newStateName: string
        @param 	newStateName: name of new state
        @type 	layerName: string
        @param 	layerName: name of layer
        """
        layer = self.layers.get(layerName)
        
        if not layer:
            raise StateException("No layer %s" % layerName)
            
        currentState = self.currentStates.get(layerName)
        
        if not currentState:
            raise StateException("No current state layer %s" % layerName)
            
        newState = layer.get(newStateName, None)
        
        if not newState:
            raise StateException("New state %s doesn't exist" % newStateName)
            
        currentState.leave(self.agent)
        self.currentStates[layerName] = newState
        newState.enter(self.agent)
        return newState.name
        
    def getNameForState(self, layerName):
        """
        Return the name of the current state for specified layer.
        
        @type 	layerName: string
        @param 	layerName: name of layer
        """
        state = self.currentStates.get(layerName)
        return state.name
        
    def getState(self, layerName):
        """
        Return the current state for specified layer.
        
        @type 	layerName: string
        @param 	layerName: name of layer
        """
        return self.currentStates.get(layerName)
        
    def getCurrentStates(self):
        """
        Returns a dictionary representing the
        current state in each of the state
        layers.
        
        """
        stateDict = {}
        for key, value in self.currentStates.items():
            stateDict[key] = value. name
            
        return stateDict
        
    #######################
    # PROPERTIES
    #######################
    def setStateProperty(self, stateName, propertyName, propertyValue):
        """
        Stores a property value for a state.
        
        @type 	stateName: string
        @param 	stateName: name of state
        @type 	propertyName: string
        @param 	propertyName: name of property
        @type 	propertyValue: object
        @param 	propertyValue: value of property
        """
        if not self.stateProperties.has_key(stateName):
            self.stateProperties[stateName] = {}
        self.stateProperties[stateName][propertyName] = propertyValue
        
    def getStateProperty(self, stateName, propertyName):
        """
        Returns a property value for a state.
        
        @type 	stateName: string
        @param 	stateName: name of state
        @type 	propertyName: string
        @param 	propertyName: name of property
        """
        propList = self.stateProperties.get(stateName, None)
        if propList:return propList.get(propertyName, None)
        return None
        
    def clearPropertiesForState(self, stateName):
        """
        Removes all properties for specified state.
        
        @type 	stateName: string
        @param 	stateName: name of state
        """
        try:del self.stateProperties[stateName]
        except KeyError:pass
    
