# svs_simulation.ai_lib.goals

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Generic classes for goal driven behaviours.

This module is based on the goal driven behaviours outlined in
Mat Buckland, 2005, Programming Game AI by Example, Wordware:Plano, 
see U{http://www.wordware.com/files/ai}, and U{http://www.ai-junkie.com}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_core.utilities.lib import Enumeration


#############################
# CONSTANTS
#############################
goal_state = Enumeration('goal_state',['ACTIVE', 'INACTIVE', 'COMPLETED', 'FAILED'])


#############################
# EXCEPTIONS
#############################
class GoalException(Exception):pass


#############################
# CLASSES
#############################
class Goal:
	"""
	Generic goal.
	"""
	def __init__(self, goalType, owner):
		self.goalType = goalType
		self.owner = owner
		self.status = goal_state.INACTIVE
		
	def activate(self):
		"""
		Logic to run when the goal is activated.
		"""
		pass

	def process(self):
		"""
		Logic to run each update-step.
		"""
		pass

	def terminate(self):
		"""
		Logic to run prior to the goal's destruction.
		"""
		pass

	def handleEvent(self, event):
		"""
		Handles events from game system.
		"""
		return False

	def addSubgoal(self, subgoal):
		"""
		Adds new subgoal.
		"""
		raise GoalException("This class does not accept subgoals.")

	def isComplete(self):
		return self.status == goal_state.COMPLETED

	def isActive(self):
		return self.status == goal_state.ACTIVE

	def isInactive(self):
		return self.status == goal_state.INACTIVE

	def hasFailed(self):
		return self.status == goal_state.FAILED

	def reactivateIfFailed(self):
		"""
		If L{Goal.status} is L{goal_state.FAILED} this method 
		sets it to inactive so that the goal will be reactivated 
		(replanned) on the next update-step.
		"""
 		if self.hasFailed():self.status = goal_state.INACTIVE

	def activateIfInactive(self):
		"""
		Reactivates the goal if L{Goal.status} is L{goal_state.INACTIVE}.
		"""
		if self.isInactive():self.activate()   



class CompositeGoal:
	"""
	A compiste goal has subgoals which it processes as part of its
	own behaviour.
	"""
	def __init__(self):
		Goal.__init__(self)
		self.subgoals = []

	def activate(self):
		"""
		Logic to run when the goal is activated.
		"""
		pass

	def process(self):
		"""
		Logic to run each update-step.
		"""
		pass

	def terminate(self):
		"""
		Logic to run prior to the goal's destruction.
		"""
		pass

	def handleEvent(self, event):
		"""
		Handles event from game system.
		"""
		return self.forwardEventToFrontSubgoal(event)

	def processSubgoals():
		"""
		Processes subgoals.
		"""
		# remove all completed and failed goals from the front of the subgoal list
		for goal in self.subgoals:
			if goal.isComplete() or goal.hasFailed():
				goal.terminate()
				self.subgoals.pop(0)
			else:break
		#if any subgoals remain, process the one at the front of the list
		if self.subgoals:
			# grab the status of the front-most subgoal
			statusOfSubGoals = self.subgoal[0].process()
			# we have to test for the special case where the front-most 
			# subgoal reports 'completed' *and* the subgoal list contains
			# additional goals.When this is the case, to ensure the parent
			# keeps processing its subgoal list we must return the 'active' status.
			if statusOfSubGoals == goal_state.COMPLETED and len(self.subgoals) > 1:
				returngoal_state.ACTIVE
		else:return goal_state.COMPLETED

	def addSubgoal(self, subgoal):
		"""
		Adds new subgoal.
		"""
		if not self.subgoals:self.subgoals = []
		self.subgoals.insert(0, subgoal)

	def removeAllSubgoals(self):
		"""
		Clears all subgoals.
		"""
		self.subgoals = []

	def forwardEventToFrontSubgoal(self, event):
		"""
		Passes the message to the goal at the front of the queue
		"""
		if self.subgoals:return self.subgoals[0].handleEvent(event)
		else:return False
