# svs_simulation.ai_lib.fusm

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Generic classes for Fuzzy State Machines.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_simulation.ai_lib.fsm import FSMState, fsm_state_types


class FuSMState(FSMState):
	"""
	Single state for fuzzy state machine.
	"""
	def __init__(self, parent, fsmType=fsm_state_types.FSM_STATE_NONE):
		self.parent = parent
		self.fsmType = fsmType
		self.activationLevel = 0.0

	def exit(self):
		pass

	def update(self, dt):
		pass

	def initState(self):
		self.activationLevel = 0.0

	def checkTransitions(self):
		return FSM_STATE_NONE

	def calculateActivation(self):
		return self.activationLevel

	def checkLowerBound(self, lbound = 0.0):
		if self.activationLevel < lbound:self.activationLevel = lbound

	def checkUpperBound(self, ubound = 1.0):
		if self.activationLevel > ubound:self.activationLevel = ubound

	def checkBounds(self, lb = 0.0, ub = 1.0):
		self.checkLowerBound(lb)
		self.checkUpperBound(ub)


class FuSMMachine(FSMMachine):
	def __init__(self, parent, fsmType=fsm_state_types.FSM_STATE_NONE):
		FSMMachine.__init__(self, parent, fsmType=fsmType)
		self.activatedStates = []

	def isActive(self, state):
		if self.activatedStates:
			if state in self.activatedStates:return True
		return False

	def reset(self):
		"""
		Exit and init all the states.
		"""
		for state in self.states:
			state.exit()
			state.init()
	
	def updateMachine(self, dt):
		"""
		Updates state machine.
		"""
		# don't do anything if you have no states
		if not self.states:return
		# check for activations, and then update
		self.activatedStates = []
		nonActiveStates = []
		for state in self.states:
			if state.calculateActivation() > 0:self.activatedStates.append(state)
			else:nonActiveStates.append(state)
		# exit all non active states for cleanup
		for state in nonActiveStates:state.exit()
		# update all activated states
		for state in self.activatedStates:state.update(dt)
