# svs_simulation.actions.actionlib

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Main classes and functions for implementing actions.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# internal imports
from svs_core.utilities.lib import encodeValue
from svs_simulation.utilities.constants import sim_const


#############################
# EXCEPTIONS
#############################
class SimActionException(Exception):pass


#############################
# CLASSES
#############################
class SimAction:
	"""
	A SimAction encapsulates a function call that may be sent from one
	part, or component, of a simulation to another and applied to an entity.
	"""
	def __init__(self, target=None, methodName=None, *args, **kwargs):
		self.target = target
		self.methodName = methodName
		self.args = args
		self.kwargs = kwargs

	def execute(self, target=None):
		"""
		Checks of target is able to perform action, and if so, executes it.
		"""
		if not target:target = self.target
		if not target:return
		try:func = getattr(target, self.methodName)
		except:return # bail out with no error
		if not callable(func):return
		try:func(*self.args, **self.kwargs)
		except:raise SimActionException("Unable to execute function: '%s'" % self.methodName)
		print "action executed"

	#############################
	# ARCHIVING
	#############################
	def encode(self):
		"""
		Returns encoded representation of self.
		
		@rtype:dict
		"""
		data = {sim_const.LABEL_METHODNAME:self.methodName,
			sim_const.LABEL_ARGS:self.encodeArgs(),
			sim_const.LABEL_KWARGS:self.encodeKwargs()}
		if self.target:data[sim_const.LABEL_TARGET] = self.target
		return data

	def encodeArgs(self):
		"""
		Encodes arguments.
		"""
		encArgs = []
		for arg in self.args:encArgs.append(encodeValue(arg))
		return encArgs

	def encodeKwargs(self):
		"""
		Encodes keyword arguments.
		"""
		encKwargs = []
		for key, value in self.kwargs.items():encKwargs[key] = encodeValue(value)
		return encKwargs

	def decode(self, data):
		"""
		Applies encoded data to self.
		"""
		self.target = data.get(sim_const.LABEL_TARGET, None)
		self.methodName = data[sim_const.LABEL_METHODNAME]
		self.args = data[sim_const.LABEL_ARGS]
		self.kwargs = data[sim_const.LABEL_KWARGS]
		
