# svs_demogame.views

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Generic base classes for view components.

The game is displayed using the Tkinter library:
U{http://www.pythonware.com/library/tkinter/introduction/}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Tkinter import *

# internal imports
from svs_demogame.utils import demo_const
from svs_core.utilities.constants import svs_const
from svs_core.utilities.lib import Bunch



#############################
# SELECTION
#############################
def encodeSelectedItem(itemName, item, selectionSource):
	"""
	Creates an encoded representation of a selected item
	that can be shared between different view classes.
	"""
	return Bunch(itemName=itemName, item=item, selectionSource=selectionSource)


#############################
# CLASSES
#############################
class GenericComponentView:
	"""
	Generic class for displayed components.
	"""
	def __init__(self, context):
		self.context = context
		self.canvas = self.context.canvas
		self.sprite = None
		self.xMin = 0
		self.yMin = 0
		self.xMax = 0
		self.yMax = 0
		self.selected = False

	def containsPoint(self, x, y):
		"""
		If point is within bounds of component, returns C{True}, otherwise C{False}.
		"""
		if x < self.xMin or x > self.xMax: return False
		if y < self.yMin or y > self.yMax: return False
		return True

	def select(self):
		"""
		Selects sprite.
		"""
		self.selected = True
		self.context.canvas.itemconfig(self.sprite, fill=demo_const.SPRITE_SELECTED_COLOUR)

	def deselect(self):
		"""
		Deselects sprite.
		"""
		if not self.selected:return
		self.selected = False
		self.context.canvas.itemconfig(self.sprite, fill=self.colour)


class GenericDataView:
	"""
	Base class for view components which display values from data source classes.

	These should pair with classes derived from L{svs_demogame.analysis.ViewableDataSource}.
	"""
	def __init__(self):
		self.dataSource = None

	def setDataSource(self, dataSource):
		"""
		Sets data source for view.

		This automatically links the view to the source as a listener.
		"""
		if self.dataSource:self.dataSource.removeView(self)
		self.dataSource = dataSource
		self.dataSource.addView(self)

	def clearDataSource(self):
		"""
		Disconnects view from current data source.
		"""
		if self.dataSource:self.dataSource.removeView(self)
		self.dataSource = None

	def notify(self, dataPacket):
		"""
		Receives notification from data source.

		NOTE: this should be overridden if other types of 
		notification are used by an extending class.
		"""
		if dataPacket.label == svs_const.DATA_UPDATED:
			self.updateView(dataPacket.content)

	
	def updateView(self, data):
		"""
		Performs update of view in response to change in data source.

		This must be overridden by the extending class.
		"""
		pass
