# svs_demogame.tracking_clients

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Clients for tracking and analysing demo game.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# internal imports
from svs_demogame.clients import GUIPlayerClient
from svs_demogame.tracking_gui import TrackingGUI
from svs_demogame.analysis import ScriptHistory, MessageHistory, ScriptNetwork, MessageNetwork
from svs_demogame.utils import demo_const
from svs_core.commands.scriptcommands import *
from svs_core.network.packets import *
from svs_core.utilities.constants import svs_const


#############################
# TRACKING CLIENT
#############################
class TrackingClient(GUIPlayerClient):
	"""
	Client that tracks changes in the script code of a game.
	"""
	def __init__(self, name, passwd):
		GUIPlayerClient.__init__(self, name, passwd)
		self.scriptHistory = ScriptHistory()
		self.messageHistory = MessageHistory()
		self.scriptNetwork = ScriptNetwork()
		self.messageNetwork = MessageNetwork()
	
	def openGUI(self):
		self.gui = TrackingGUI(self)
		self.gui.build()

	def getGameWorldModel(self):
		"""
		Retrieves model of game from gameworld client.
		"""
		if not self.gameWorld:return
		dataRequest = makeDataRequest(self, recipient=self.gameWorld, label=demo_const.WORLD_MODEL_LABEL)
		self.getData(dataRequest)
		self.startListeningTo(self.gameWorld, listenFor=demo_const.WORLD_UPDATE_LABEL)
		self.startListeningTo(self.gameWorld, listenFor=demo_const.SCRIPT_REVISION_LABEL)
		self.startListeningToProcess(svs_const.CHAT_MESSAGE)

	#########################
	# DATA
	#########################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.
		""" 
		if dataPacket.label == svs_const.CLIENT_JOINED:
			self.handleClientJoined(dataPacket.content)
		elif dataPacket.label == svs_const.CLIENT_DEPARTED:
			self.handleClientDeparted(dataPacket.content)
		elif dataPacket.label == demo_const.SCRIPT_REVISION_LABEL:
			self.handleScriptRevision(dataPacket.content)
		elif dataPacket.label == svs_const.CHAT_MESSAGE:
			self.handleChatMessage(dataPacket.content)
		else:GUIPlayerClient.handleDataPacket(self, dataPacket)

	
	#########################
	# CLIENTS
	#########################
	def handleClientJoined(self, client):
		"""
		Responds to new client joining network.
		"""
		#print "handleClientJoined:", client
		pass

	def handleClientDeparted(self, client):
		"""
		Responds to client leaving network.
		"""
		#print "handleClientDeparted:", client
		pass

	#########################
	# SCRIPT TRACKING
	#########################
	def handleScriptRevision(self, script):
		"""
		Responds to new script revisions recieved over network.
		"""
		self.scriptHistory.addScript(script)
		self.scriptNetwork.addScript(script)

	#########################
	# MESSAGE TRACKING
	#########################
	def handleChatMessage(self, msgPacket):
		"""
		Handles message packet received through tracking.

		This is treated as analysis data rather than as a message to the client.
		"""
		self.messageHistory.addMessage(msgPacket)
		self.messageNetwork.addMessage(msgPacket)

	#########################
	# SELECTION METHODS
	#########################
	def areaSelected(self, areaIdNum):
		"""
		Called by area view components when selected.
		"""
		pass

	def nodeSelected(self, node):
		"""
		Called by node view components when selected.
		"""
		pass
		
	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_openview(self, cmd):
		"""
		Open visualisation in window.
		"""
		self.gui.openView(False)
		return makeCommandResult(cmd, status=svs_const.OK)

	def cmdprivate_hideview(self, cmd):
		"""
		Hide visualisation in window.
		"""
		self.gui.hideView()
		return makeCommandResult(cmd, status=svs_const.OK)
	
	def cmdprivate_fullscreen(self, cmd):
		"""
		Open fullscreen visualisation.
		"""
		self.gui.openView(True)
		return makeCommandResult(cmd, status=svs_const.OK)
	
	
#############################
# RECORDING CLIENT
#############################
class RecordingClient(GUIPlayerClient):
	"""
	Client that records events in a game.
	"""
	def __init__(self, name, passwd):
		GUIPlayerClient.__init__(self, name, passwd)
		
#############################
# VERSIONING CLIENT
#############################
class VersioningClient(GUIPlayerClient):
	"""
	Client that maintains versioning system for game scripts.
	"""
	def __init__(self, name, passwd):
		GUIPlayerClient.__init__(self, name, passwd)
