# svs_demogame.scripts

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Class for handling scripts in the demo game.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from twisted.spread import pb
from time import time

# internal imports
from svs_core.utilities.lib import Constants

#############################
# CONSTANTS
#############################
script_const = Constants()
# script environment
script_const.SCRIPT_ENV = {}
script_const.GAME_CLIENT_PROPERTY = 'game_client'
# script code
script_const.TEMPLATE_SCRIPT_CODE_DEFAULT = \
"""
def update():
	pass
"""

#############################
# EXCEPTIONS
#############################
class ScriptException(Exception):pass

#############################
# FUNCTIONS
#############################
def setScriptEnvironmentProperty(propertyName, propertyValue):
	"""
	Sets a value used as part of the environemnt in which
	game scripts are run.
	"""
	script_const.SCRIPT_ENV[propertyName] = propertyValue

def getScriptEnvironmentProperty(propertyName):
	"""
	Returns value used as part of the environemnt in which
	game scripts are run.
	"""
	return script_const.SCRIPT_ENV[propertyName]
	
def makeEmptyScript():
	"""
	Creates L{Script} object with default code entry.
	"""
	script = Script()
	script.code = script_const.TEMPLATE_SCRIPT_CODE
	script.revisionTime = time()
	return script

def makeScriptFromCode(code, author=None):
	"""
	Creates L{Script} object with specified code and author.
	"""
	if not code:return None
	script = Script()
	### NOTE: should add in some kind of code verification
	script.code = code
	script.author = author
	script.revisionTime = time()
	return script

def loadGameCode(path=None):
	"""
	Loads the text file for the code functions available to
	game players.

	If path is not defined, loads the file C{svs_demogame/gamecode.py}.
	"""
	from imp import find_module
	if not path: 
		basepath = find_module('svs_demogame')
		path = '%s/gamecode.py' % basepath[1]
	try:
		gamecode = open(path)
	except:raise ScriptException("unable to load game code: '%s'" % path)
	script_const.DEFAULT_SCRIPT_GAMECODE = gamecode.read()

def loadTemplateCode(path=None):
	"""
	Loads code to be used as template code for
	scriptable game entities.
	"""
	if not path: 
		script_const.TEMPLATE_SCRIPT_CODE = script_const.TEMPLATE_SCRIPT_CODE_DEFAULT
		return
	try:
		templateCode = open(path)
	except:raise ScriptException("unable to load template code: '%s'" % path)
	script_const.TEMPLATE_SCRIPT_CODE = templateCode.read()
	


def importCode(code,name,add_to_sys_modules=0):
	"""
	Import dynamically generated code as a module. C{code} is the
	object containing the code (a string, a file handle or an
	actual compiled code object, same types as accepted by an
	exec statement). The C{name} is the name to give to the module,
	and the final argument says whether to add it to sys.modules
	or not. If it is added, a subsequent import statement using
	name will return this module. If it is not added to sys.modules
	import will try to load it in the normal fashion.

	import foo

	is equivalent to

	foofile = open("/path/to/foo.py")
	foo = importCode(foofile,"foo",1)

	Returns a newly generated module.

	Written by Anders Hammarquist, U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/82234}
	"""
	import sys,imp

	module = imp.new_module(name)
	
	code = script_const.DEFAULT_SCRIPT_GAMECODE + code

	exec code in module.__dict__
	if add_to_sys_modules:
		sys.modules[name] = module

	return module


#############################
# CLASSES
#############################
class ScriptHandler:
	def __init__(self, hostObject):
		self.hostObject = hostObject
		self.scriptModuleName = self.hostObject.getScriptIdentifier()
		self.scriptHistory = []
		self.scriptStore = ScriptStore()
		self.currentScript = None
		self.scriptModule = None
		self.lastRevisionLoaded = -1

	def getName(self):
		"""
		Returns name for script, this is the automatically 
		generated L{ScriptHandler.self.scriptModuleName} property.
		"""
		return self.scriptModuleName

	def getScriptRevision(self, revNum=-1):
		"""
		Returns specified version of script.
		"""
		if revNum < 0:return self.getCurrentScript()
		if revNum < len(self.scriptHistory):return self.scriptHistory[revNum]
	
	def getCurrentScript(self):
		"""
	`	Returns most recent version of script.
		"""
		if len(self.scriptHistory) == 0:
			self.currentScript = makeEmptyScript()
			self.scriptHistory.append(self.currentScript)
		return self.scriptHistory[-1]

	def getRevisionLog(self):
		"""
		Returns log of script revisions, without code.
		"""
		log = {'script_name':self.scriptModuleName}
		revs = []
		for script in self.scriptHistory:
			revData = {'author':script.author, 'revision':script.revisionNumber}
			revData['date'] = script.revisionTime
			revs.append(revData)
		log['revisions'] = revs
		return log
	
	def appendScript(self, newScript):
		"""
		Appends script to end of script history, making it the 
		current script for execution.

		NOTE: should perform some kind of verification of C{newScript} object.
		"""
		if not newScript:return
		script = Script()
		script.scriptID = self.hostObject.getScriptIdentifier()
		script.revisionTime = time()
		script.revisionNumber = len(self.scriptHistory)
		script.executionLog = {}
		script.author = newScript.author
		script.code = newScript.code
		self.currentScript = script
		self.scriptHistory.append(self.currentScript)
		return script

	def copyScript(self, newScript):
		"""
		Copies script.

		NOTE: should perform some kind of verification of C{newScript} object.
		"""
		return
		if not newScript:return
		script = Script()
		script.scriptID = self.hostObject.getScriptIdentifier()
		script.revisionTime = time()
		script.revisionNumber = len(self.scriptHistory)
		script.executionLog = {}
		script.author = newScript.author
		#script.hostObject = self.hostObject
		script.code = newScript.code
		self.currentScript = script
		self.scriptHistory.append(self.currentScript)

	def clearScripts(self):
		"""
		Clears scripts.
		"""
		self.scriptHistory = []
		self.currentScript = None
		self.scriptModule = None
		self.lastRevisionLoaded = -1

	def loadCurrentScript(self):
		"""
		Loads current script and sets up necessary environment
		variables.
		"""
		if not self.currentScript:return
		# make sure code is not imported every time
		if not (self.currentScript.revisionNumber > self.lastRevisionLoaded):return
		# load script code
		try:self.scriptModule = importCode(self.currentScript.code, self.scriptModuleName)
		except:
			self.scriptModule = None
			return
		self.lastRevisionLoaded = self.currentScript.revisionNumber
		# set up environment stuff
		self.scriptModule.CALLING_CLIENT = self.currentScript.author
		self.scriptModule.GAME_CLIENT = getScriptEnvironmentProperty(script_const.GAME_CLIENT_PROPERTY)
		self.scriptModule.STORE = self.scriptStore
		self.scriptModule.me = self.hostObject

	def executeCurrentScriptMethod(self, methodName, methodArgs=None):
		"""
		Executes specified method from current script.

		If the script does not contain this method it is ignored.
		"""
		if not self.scriptModule:return
		try:scriptMethod = getattr(self.scriptModule, methodName)
		except AttributeError:return
		if not callable(scriptMethod):return
		if methodArgs:
			try:scriptMethod(methodArgs)
			except:pass # catch errors but don't bail
			else:self.__updateExecutionLog(self.currentScript, scriptMethod)
		else:
			try:scriptMethod()
			except:pass # catch errors but don't bail
			else:self.__updateExecutionLog(self.currentScript, scriptMethod)

	def executeCurrentScript(self):
		"""
		Executes C{update} method of script.
		"""
		if not self.scriptModule:return
		try:scriptMethod = getattr(self.scriptModule, 'update')
		except AttributeError:return
		if not callable(scriptMethod):return
		try:loopLimit = self.scriptModule.LOOP
		except AttributeError: loopLimit = 1
		if self.isWithinExecutionLimit(self.currentScript, scriptMethod, loopLimit):
			try:scriptMethod()
			except:pass # catch errors but don't bail
			else:self.__updateExecutionLog(self.currentScript, scriptMethod)

	
	def __updateExecutionLog(self, script, method):
		if not script.executionLog.has_key(method.__name__):
			script.executionLog[method.__name__] = 0
		script.executionLog[method.__name__] += 1

	def isWithinExecutionLimit(self, script, method, limitValue):
		"""
		Checks that script method has not been performed more times than
		execution limit allows.  Return C{True} if ok to execute, C{False}
		otherwise.
		"""
		if not script.executionLog.has_key(method.__name__):return True
		if script.executionLog[method.__name__] < limitValue or limitValue < 0: return True
		return False


class Script(pb.Copyable, pb.RemoteCopy):
	"""
	Maintains code for a script, the code itself is treated as a module
	upon which functions are called when executed.
	"""
	def __init__(self):
		self.scriptID = None
		self.author = None
		self.hostObject = None
		self.revisionTime = None
		self.code = None
		self.revisionNumber = 0
		self.executionLog = {}
	
	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		# note: don't copy 'self.executionLog'
		d = self.__dict__.copy()
		return d
	
# register with perspective broker
pb.setUnjellyableForClass(Script, Script)

		

class ScriptStore:
	"""
	This class is used by the L{ScriptHandler} to store data for scripts
	between calls in the game loop.

	Normally any variables set in a game script are lost after the script
	has exited.

	NOTE: in its current form this is just a wrapper around a dictionary,
	future implementatiosn may provide some security handling, or allow
	custom methods to be called on the store.
	"""
	def __init__(self):
		self.datastore = {}

	def set(self, name, value):
		"""
		Stores some data.
		"""
		self.datastore[name] = value
		return value

	def get(self, name, returnValue=None):
		"""
		Retrieves some data.

		If the data is not present, the return value
		can be used as a default.
		"""
		return self.datastore.get(name, returnValue)

	def delete(self, name):
		"""
		Removes the named data from the store.
		"""
		self.datastore.pop(name)
