# svs_demogame.scripthistoryviews

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Graphical representation of changes in script code.

Uses the Tkinter library:
U{http://www.pythonware.com/library/tkinter/introduction/}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Tkinter import *
from time import time
import math

# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.views import GenericDataView,GenericComponentView, encodeSelectedItem
from svs_core.utilities.lib import Constants
from svs_core.utilities.constants import svs_const
from svs_core.geometry.geomlib import geom_const



#############################
# CONSTANTS
#############################
scripthistoryviews_const = Constants()
# tkinter tags
scripthistoryviews_const.FILE_TAG = 'file'
scripthistoryviews_const.FILE_LABEL_TAG = 'file_label'
scripthistoryviews_const.FILE_LABEL_BG_TAG = 'file_label_bg'
scripthistoryviews_const.FILE_SELECTOR_TAG = 'file_selector'
scripthistoryviews_const.FILE_BG_TAG = 'file_bg'
scripthistoryviews_const.TIME_CURSOR_TAG = 'time_cursor'


#############################
# SCRIPT HISTORY
#############################
class ScriptHistoryView(GenericDataView):
	"""
	Displays changes to game scripts over time.
	"""
	def __init__(self, parent, context):
		GenericDataView.__init__(self)
		self.context = context
		self.canvas = Canvas(parent, bg=demo_const.BG_COLOUR, borderwidth=0, highlightthickness=0)
		self.canvas.bind('<Configure>', self.canvasAdjusted)
		self.canvas.bind('<Button-3>', self.onMouseDown_3)
		self.canvas.bind('<ButtonRelease-3>', self.onMouseUp_3)
		self.canvas.bind('<B3-Motion>', self.onMouseDrag_3)
		self.width = self.canvas.winfo_width()
		self.height = self.canvas.winfo_height()
		self.timeSpaceUnit = 1.0
		self.scriptWidth = 5.0
		self.scriptHeight = 10.0
		self.scriptSpacing = 1.0
		self.horizontalMargin = 10
		self.verticalMargin = 10
		self.labelAreaWidth = 80
		self.scripts = {}
		self.orderedScriptNames = []
		self.selectedScript = None
		self.scriptLabels = ScriptHistoryLabels(self)
		# time display
		self.startTime = time()
		self.timeCursorSprite = None
		self.pauseTimeCursor()
		# scrolling
		self.prevX, self.prevY = 0,0
		self.scrollLeft = self.horizontalMargin + self.labelAreaWidth
		self.scrollRight = self.canvas.winfo_width()
		self.scrollTop = self.verticalMargin
		self.scrollBottom = self.canvas.winfo_height()
		self.scrollOffsetX = 0.0 # amount moved by scroll operation
		self.scrollOffsetY = 0.0
		self.scrollAhead = self.width / 3.0 # distance ahead of time cursor
		self.scrollPartner = None
		self.pageX = (self.width / 4.0 ) * 3
		

	def canvasAdjusted(self, args=None):
		"""
		Responds to main window being adjusted in size.
		"""
		self.width = self.canvas.winfo_width()
		self.height = self.canvas.winfo_height()
		self.scrollAhead = self.width / 3.0
		self.pageX = (self.width / 4.0 ) * 3
		if not self.scripts:return
		self.clearCanvas()
		self.drawScriptsFresh()
		self.drawTimeCursor()
		
	def clearCanvas(self):
		"""
		Clears entire canvas.
		"""
		self.canvas.delete(scripthistoryviews_const.FILE_TAG)
		self.canvas.delete(scripthistoryviews_const.FILE_SELECTOR_TAG)
		self.canvas.delete(scripthistoryviews_const.FILE_BG_TAG)
		self.canvas.delete(scripthistoryviews_const.FILE_LABEL_TAG)
		self.canvas.delete(scripthistoryviews_const.FILE_LABEL_BG_TAG)
		self.canvas.delete(scripthistoryviews_const.TIME_CURSOR_TAG)
		self.timeCursorSprite = None

	def addScript(self, scriptName):
		"""
		Adds a new script to view, this becomes a column within the display.

		If C{scriptName} already exists within the view's scripts, it is ignored.
		"""
		if self.scripts.has_key(scriptName):return
		self.scripts[scriptName] = {'revisions':{}, 'most_recent':None}

	def addScriptRevision(self, scriptName, timeIndex, diffValue):
		"""
		Adds a revision to the specified script.

		If the script is not already within the view's scripts, it is added.
		"""
		if not self.scripts.has_key(scriptName):self.scripts[scriptName] = {'revisions':{}, 'most_recent':None}
		if self.scripts[scriptName]['revisions'].has_key(timeIndex):return
		sprite = ScriptHistorySprite(self, scriptName, timeIndex, diffValue)
		self.scripts[scriptName]['revisions'][timeIndex] = sprite
		self.scripts[scriptName]['most_recent'] = sprite

	def drawScriptsFresh(self):
		"""
		Draws new view graphics for scripts.
		"""
		if len(self.orderedScriptNames) == 0: return
		scriptY = self.verticalMargin
		scriptX = self.horizontalMargin + self.labelAreaWidth + self.scrollOffsetX
		scriptSpace = self.scriptHeight + self.scriptSpacing
		# scrolling
		self.scrollTop = scriptY
		self.scrollRight = self.canvas.winfo_width()
		# script sprites
		for scriptName in self.orderedScriptNames:
			scriptSprites = self.scripts[scriptName]['revisions']
			for script in scriptSprites.values():
				script.drawFresh(scriptX, scriptY, self.startTime, self.timeSpaceUnit, self.scriptWidth, self.scriptHeight)
				# update scroll right
				if script.xMax > self.scrollRight: self.scrollRight = script.xMax
			# background strip
			self.canvas.create_rectangle(self.horizontalMargin + self.labelAreaWidth, scriptY, self.width, scriptY + self.scriptHeight, fill=demo_const.VIEW_AREA_COLOUR_01, width=0, tag=scripthistoryviews_const.FILE_BG_TAG)
			scriptY += scriptSpace
		self.canvas.lower(scripthistoryviews_const.FILE_BG_TAG)
		# scrolling
		self.scrollBottom = scriptY + scriptSpace
		# labels
		self.canvas.create_rectangle(0,0, self.horizontalMargin + self.labelAreaWidth, self.height, fill=demo_const.BG_COLOUR, width=0, tag=scripthistoryviews_const.FILE_LABEL_BG_TAG)
		self.scriptLabels.drawFresh(self.horizontalMargin, self.verticalMargin, self.scriptHeight + self.scriptSpacing, self.orderedScriptNames)
		# top margin
		self.canvas.create_rectangle(0,0, self.width, self.verticalMargin, fill=demo_const.BG_COLOUR, width=0, tag=scripthistoryviews_const.FILE_LABEL_BG_TAG)


	def onMouseDown_3(self, event):
		self.prevX, self.prevY = event.x, event.y
		self.pauseTimeCursor()
		self.scrollPartner.pauseTimeCursor()

	def onMouseUp_3(self, event):
		self.unpauseTimeCursor()
		self.scrollPartner.unpauseTimeCursor()

	def onMouseDrag_3(self, event):
		dX = self.prevX - event.x
		dY = self.prevY - event.y
		if self.scrollBottom < self.height:dY = 0
		bounds =  self.canvas.bbox(scripthistoryviews_const.FILE_TAG) 
		rightLimit = self.timeCursorX + self.scrollOffsetX + self.scrollAhead
		if rightLimit < bounds[2]:rightLimit = bounds[2] ### ??? needs work
		if dX < 0 and bounds[2] < self.width:dX = 0 # scroll right
		elif dX > 0 and bounds[0] > self.scrollLeft:dX = 0 # scroll left
		if dY < 0 and bounds[3] < self.height:dY = 0 # scroll up
		elif dY > 0 and bounds[1] >= self.scrollTop:dY = self.scrollTop - bounds[1] # scroll down
		self.prevX, self.prevY = event.x, event.y
		self.scrollXY(dX, dY)

	def scrollXY(self, dX, dY, synch=True):
		"""
		Scrolls view components by specified amounts on x and y axes.
		"""
		self.canvas.move(scripthistoryviews_const.FILE_TAG, dX, dY)
		self.canvas.move(scripthistoryviews_const.FILE_BG_TAG, 0, dY)
		self.canvas.move(scripthistoryviews_const.FILE_LABEL_TAG, 0, dY)
		self.scrollOffsetX += dX
		self.scrollOffsetY += dY
		if synch:self.scrollPartner.synchScroll(dX)

	def synchScroll(self, dX):
		"""
		Synchronises horizontal scroll.
		"""
		self.canvas.move(scripthistoryviews_const.FILE_TAG, dX, 0)
		self.prevX += dX
		self.scrollOffsetX += dX

	def pageForward(self):
		"""
		Scrolls the view forward by page size.
		"""
		if (self.width - self.pageX) < (self.horizontalMargin + self.labelAreaWidth):
			self.scrollXY(self.horizontalMargin + self.labelAreaWidth -self.pageX, 0)
		else:self.scrollXY(-self.pageX, 0)

	def pageBack(self):
		"""
		Scrolls the view backward by page size.
		"""
		self.scrollXY(self.pageX, 0)

	def updateView(self, data):
		"""
		Performs update of view in response to change in data source.
		"""
		scriptData = data[0]
		self.orderedScriptNames = data[1]
		for scriptName, revisions in scriptData.items():
			self.addScript(scriptName)
			for revision in revisions:
				self.addScriptRevision(scriptName, revision.getRevisionTime(), revision.diffValue)
		self.clearCanvas()
		self.drawScriptsFresh()
		self.drawTimeCursor()

	def scriptSelected(self, script):
		"""
		Called by script view components when selected, passes info onto client.
		"""
		if self.selectedScript:self.selectedScript.deselect()
		self.selectedScript = script
		self.selectedScript.select()
		self.scriptLabels.selectLabel(script.scriptName)

	def scriptLabelSelected(self, scriptName):
		"""
		Called by script label components when selected, passes info onto client.

		Selects most recently added version of script.
		"""
		if self.selectedScript and self.selectedScript.scriptName != scriptName:self.selectedScript.deselect()
		if not self.scripts.has_key(scriptName):return
		self.selectedScript = self.scripts[scriptName]['most_recent']
		self.selectedScript.select()
		self.scriptLabels.selectLabel(self.selectedScript.scriptName)

	def selectItem(self, selectedItem):
		"""
		Selects specified item if present within view.

		This method is used to synchronise selections 
		between the different tracker views.
		"""
		self.clearSelection()
		if selectedItem.selectionSource == self:self.scriptSelected(selectedItem.item)
		elif self.scripts.has_key(selectedItem.itemName):self.scriptLabelSelected(selectedItem.itemName)
			
	def clearSelection(self):
		"""
		Clears any selected items in view.
		"""
		if self.selectedScript:
			self.scriptLabels.deselectLabel(self.selectedScript.scriptName)
			self.selectedScript.deselect()
			self.selectedScript = None

	def makeSelection(self, selectedItem):
		"""
		Called by component sprites when selected.

		Sends message to parent for synchronisation.
		"""
		self.context.syncSelection(encodeSelectedItem(selectedItem.scriptName, selectedItem, self))


	def startTimeCursor(self, startTime):
		"""
		Sets start time used by view.
		"""
		self.startTime = startTime

	def synchTimeCursor(self, synchTime):
		"""
		Sets time cursor in view to synchronise with C{synchTime}.
		"""
		self.timeCursorX = self.horizontalMargin + self.labelAreaWidth + (synchTime  * self.timeSpaceUnit)
		self.drawTimeCursor()

	def pauseTimeCursor(self):
		"""
		Sets time cursor in pause mode.
		"""
		self.timeCursorPaused = True
		self.canvas.itemconfig(self.timeCursorSprite, fill=demo_const.DFT_COLOUR)

	def unpauseTimeCursor(self):
		"""
		Sets time cursor in pause mode.
		"""
		self.timeCursorPaused = False
		self.canvas.itemconfig(self.timeCursorSprite, fill=demo_const.TIME_CURSOR_COLOUR)

	def drawTimeCursor(self):
		"""
		Draws the time cursor sprite.
		"""
		if self.timeCursorPaused:return
		x = self.timeCursorX + self.scrollOffsetX
		if x > self.width:
			self.pageForward()
			x += self.scrollOffsetX
		if not self.timeCursorSprite:
			self.timeCursorSprite = self.canvas.create_line(x,0, x, self.height, fill=demo_const.TIME_CURSOR_COLOUR, width=1, tag=scripthistoryviews_const.TIME_CURSOR_TAG)
		else:self.canvas.coords(self.timeCursorSprite, x,0, x, self.height)
		
		
		

#############################
# SCRIPT HISTORY SPRITE
#############################
class ScriptHistorySprite(GenericComponentView):
	"""
	Displays state of script at particular moment in time.
	"""
	def __init__(self, context, scriptName, timeIndex, diffValue):
		GenericComponentView.__init__(self, context)
		self.scriptName = scriptName
		self.timeIndex = timeIndex
		self.diffValue = diffValue
		colourValue = (1.0 - diffValue) * 255
		self.colour = "#%02x%02x%02x" % (colourValue, colourValue, colourValue)
	
	def drawFresh(self, locX, locY, startTime, timeSpaceUnit, drawWidth, drawHeight):
		"""
		Draws script view onto canvas.
		"""
		self.xMin = locX + ((self.timeIndex - startTime) * timeSpaceUnit)
		self.yMin = locY
		self.xMax = self.xMin + drawWidth
		self.yMax = self.yMin + drawHeight
		self.sprite = self.canvas.create_rectangle(self.xMin, self.yMin, self.xMax, self.yMax, fill=self.colour, outline=demo_const.SPRITE_HILIGHT_COLOUR, width=0, tag=scripthistoryviews_const.FILE_TAG)
		self.canvas.tag_bind(self.sprite, '<Button-1>', self.onMouseClick)
		self.canvas.tag_bind(self.sprite, "<Enter>", self.hilight)
		self.canvas.tag_bind(self.sprite, "<Leave>", self.unHighlight)
		if self.selected:self.canvas.itemconfig(self.sprite, fill=demo_const.SPRITE_SELECTED_COLOUR)
		return self.yMax

	def onMouseClick(self, event):
		"""
		Responds to double-click from mouse.
		"""
		self.context.makeSelection(self)

	def hilight(self, event):
		"""
		Highlights sprite.
		"""
		self.canvas.itemconfig(self.sprite, width=1)
		self.context.scriptLabels.hilightLabel(self.scriptName)

	def unHighlight(self, event):
		"""
		Highlights sprite.
		"""
		self.canvas.itemconfig(self.sprite, width=0)
		self.context.scriptLabels.unHilightLabel(self.scriptName)

#############################
# SCRIPT HISTORY LABELS
#############################
class ScriptHistoryLabels(GenericComponentView):
	"""
	Displays name of script.
	"""
	def __init__(self, context):
		GenericComponentView.__init__(self, context)
		self.colour = demo_const.DFT_COLOUR
		self.labelSprites = {}
		self.selectedLabel = None

	def addLabel(self, labelText):
		"""
		Adds label for script listing.
		"""
		self.labelSprites[labelText] = None
		
	
	def drawFresh(self, locX, locY, spacer, orderedScriptNames):
		"""
		Draws script labels onto canvas.
		"""
		self.xMin = locX
		self.yMin = locY
		self.spacer = spacer
		verticalSpacer = spacer/2
		for labelText in orderedScriptNames:
			sprite = self.canvas.create_text(self.xMin, self.yMin + verticalSpacer, fill=self.colour, text=labelText, anchor='w' , tag=scripthistoryviews_const.FILE_LABEL_TAG)
			self.labelSprites[labelText] = sprite
			# handlers for enter and leave actions
			def onEnter(event, self=self, sprite=sprite):
                		self.hilightSprite(sprite)
			def onLeave(event, self=self, sprite=sprite):
                		self.unHighlightSprite(sprite)
			def onDoubleClick(event, self=self, sprite=sprite):
                		self.selectSprite(sprite)
			# end handlers
			self.canvas.tag_bind(sprite, "<Enter>", onEnter)
			self.canvas.tag_bind(sprite, "<Leave>", onLeave)
			#self.canvas.tag_bind(sprite, '<Button-1>', onDoubleClick)
			verticalSpacer += self.spacer

	
	def hilightLabel(self, labelText):
		"""
		Highlights label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.hilightSprite(sprite)

	def hilightSprite(self, sprite):
		"""
		Highlights sprite.
		"""
		if self.selectedLabel == sprite:return
		self.canvas.itemconfig(sprite, fill=demo_const.SPRITE_HILIGHT_COLOUR)

	def unHilightLabel(self, labelText):
		"""
		Unhighlights label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.unHighlightSprite(sprite)

	def unHighlightSprite(self, sprite):
		"""
		Unhighlights sprite.
		"""
		if self.selectedLabel == sprite:return
		self.canvas.itemconfig(sprite, fill=self.colour)

	def selectLabel(self, labelText):
		"""
		Selects label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.selectSprite(sprite)

	def selectSprite(self, sprite):
		"""
		Selects sprite.
		"""
		if self.selectedLabel:self.canvas.itemconfig(self.selectedLabel, fill=self.colour)
		self.selectedLabel = sprite
		self.canvas.itemconfig(self.selectedLabel, fill=demo_const.SPRITE_SELECTED_COLOUR)

	def deselectLabel(self, labelText):
		"""
		Deselects label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.deselectSprite(sprite)

	def deselectSprite(self, sprite):
		"""
		Deselects sprite.
		"""
		if self.selectedLabel:self.canvas.itemconfig(self.selectedLabel, fill=self.colour)
		self.selectedLabel = None



#############################
# SCRIPT DIFF
#############################
class ScriptDiffView(GenericDataView):
	"""
	Displays difference comparisions between versions of a script.
	"""
	def __init__(self, parent=None, context=None):
		GenericDataView.__init__(self)
		self.context = context
		self.width = 0
		self.height = 0
		self.canvas = Canvas(parent, bg='white', borderwidth=0, highlightthickness=0)
		#self.canvas.bind('<Configure>', self.canvasAdjusted)
		#self.canvas.bind('<Button-1>', self.mouseClicked)
		self.horizontalMargin = 10
		self.verticalMargin = 10
		self.maxRadius = 0
		self.spriteRadius = 5
		self.centreX = 0
		self.centreY = 0
		self.nodes = {}
		self.orderedNodeNames = []
		self.selectedNode = None
		#self.canvasAdjusted()

