# svs_demogame.messagehistoryviews

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Graphical representation of message messages between players.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Tkinter import *
from time import time
import math

# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.views import GenericDataView, GenericComponentView, encodeSelectedItem
from svs_core.utilities.lib import Constants
from svs_core.utilities.constants import svs_const
from svs_core.geometry.geomlib import geom_const



#############################
# CONSTANTS
#############################
messagehistoryviews_const = Constants()
# tkinter tags
messagehistoryviews_const.MSG_TAG = 'msg'
messagehistoryviews_const.MSG_LABEL_TAG = 'msg_label'
messagehistoryviews_const.MSG_LABEL_BG_TAG = 'msg_label_bg'
messagehistoryviews_const.MSG_SELECTOR_TAG = 'msg_selector'
messagehistoryviews_const.MSG_BG_TAG = 'msg_bg'
messagehistoryviews_const.TIME_CURSOR_TAG = 'time_cursor'


#############################
# MESSAGE HISTORY
#############################
class MessageHistoryView(GenericDataView):
	"""
	Displays chat messages over time.
	"""
	def __init__(self, parent, context):
		GenericDataView.__init__(self)
		self.context = context
		self.canvas = Canvas(parent, bg=demo_const.BG_COLOUR, borderwidth=0, highlightthickness=0)
		self.canvas.bind('<Configure>', self.canvasAdjusted)
		self.canvas.bind('<Button-3>', self.onMouseDown_3)
		self.canvas.bind('<ButtonRelease-3>', self.onMouseUp_3)
		self.canvas.bind('<B3-Motion>', self.onMouseDrag_3)
		self.width = self.canvas.winfo_width()
		self.height = self.canvas.winfo_height()
		self.timeSpaceUnit = 1.0
		self.messageWidth = 5.0
		self.messageHeight = 10.0
		self.messageSpacing = 1.0
		self.horizontalMargin = 10
		self.verticalMargin = 10
		self.labelAreaWidth = 80
		self.messages = {}
		self.orderedClientNames = []
		self.selectedMessage = None
		self.messageLabels = MessageHistoryLabels(self)
		# time display
		self.startTime = time()
		self.timeCursorSprite = None
		self.pauseTimeCursor()
		# scrolling
		self.prevX, self.prevY = 0,0
		self.scrollLeft = self.horizontalMargin + self.labelAreaWidth
		self.scrollRight = self.canvas.winfo_width()
		self.scrollTop = self.verticalMargin
		self.scrollBottom = self.canvas.winfo_height()
		self.scrollOffsetX = 0.0 # amount moved by scroll operation
		self.scrollOffsetY = 0.0
		self.scrollAhead = self.width / 3.0 # distance ahead of time cursor
		self.pageX = (self.width / 4.0 ) * 3
		self.scrollPartner = None
		self.timeCursorPaused = False

	def canvasAdjusted(self, args=None):
		"""
		Responds to main window being adjusted in size.
		"""
		self.width = self.canvas.winfo_width()
		self.height = self.canvas.winfo_height()
		self.scrollAhead = self.width / 3.0
		self.pageX = (self.width / 4.0 ) * 3
		if not self.messages:return
		self.clearCanvas()
		self.drawMessagesFresh()
		self.drawTimeCursor()
		
	def clearCanvas(self):
		"""
		Clears entire canvas.
		"""
		self.canvas.delete(messagehistoryviews_const.MSG_TAG)
		self.canvas.delete(messagehistoryviews_const.MSG_LABEL_TAG)
		self.canvas.delete(messagehistoryviews_const.MSG_LABEL_BG_TAG)
		self.canvas.delete(messagehistoryviews_const.MSG_SELECTOR_TAG)
		self.canvas.delete(messagehistoryviews_const.MSG_BG_TAG)
		self.canvas.delete(messagehistoryviews_const.TIME_CURSOR_TAG)
		self.timeCursorSprite = None

	def addMessageClient(self, clientName):
		"""
		Adds a new message client to view, this becomes a row within the display.

		If C{clientName} already exists within the view's messages, it is ignored.
		"""
		if self.messages.has_key(clientName):return
		self.messages[clientName] = {'messages':{}, 'most_recent':None}

	def addMessage(self, clientName, message):
		"""
		Adds a message to the specified client.

		If the client is not already within the view's listing, it is added.
		"""
		if not self.messages.has_key(clientName):self.messages[clientName] = {'messages':{}, 'most_recent':None}
		if self.messages[clientName]['messages'].has_key(message.created):return
		sprite = MessageHistorySprite(self, message)
		self.messages[clientName]['messages'][message.created] = sprite
		self.messages[clientName]['most_recent'] = sprite

	def drawMessagesFresh(self):
		"""
		Draws new view graphics for messages.
		"""
		if len(self.orderedClientNames) == 0: return
		messageY = self.verticalMargin
		messageX = self.horizontalMargin + self.labelAreaWidth + self.scrollOffsetX
		messageSpace = self.messageHeight + self.messageSpacing
		# scrolling
		self.scrollTop = messageY
		self.scrollRight = self.canvas.winfo_width()
		# script sprites
		for clientName in self.orderedClientNames:
			messageSprites = self.messages[clientName]['messages']
			for message in messageSprites.values():
				message.drawFresh(messageX, messageY, self.startTime, self.timeSpaceUnit, self.messageWidth, self.messageHeight)
				# update scroll right
				if message.xMax > self.scrollRight: self.scrollRight = message.xMax
			# background strip
			self.canvas.create_rectangle(self.horizontalMargin + self.labelAreaWidth, messageY, self.width, messageY + self.messageHeight, fill=demo_const.VIEW_AREA_COLOUR_01, width=0, tag=messagehistoryviews_const.MSG_BG_TAG)
			messageY += messageSpace
		self.canvas.lower(messagehistoryviews_const.MSG_BG_TAG)
		# scrolling
		self.scrollBottom = messageY + messageSpace
		# labels
		self.canvas.create_rectangle(0,0, self.horizontalMargin + self.labelAreaWidth, self.height, fill=demo_const.BG_COLOUR, width=0, tag=messagehistoryviews_const.MSG_LABEL_BG_TAG)
		self.messageLabels.drawFresh(self.horizontalMargin, self.verticalMargin, self.messageHeight + self.messageSpacing, self.orderedClientNames)
		# top margin
		self.canvas.create_rectangle(0,0, self.width, self.verticalMargin, fill=demo_const.BG_COLOUR, width=0, tag=messagehistoryviews_const.MSG_LABEL_BG_TAG)

	def onMouseDown_3(self, event):
		self.prevX, self.prevY = event.x, event.y
		self.pauseTimeCursor()
		self.scrollPartner.pauseTimeCursor()

	def onMouseUp_3(self, event):
		self.unpauseTimeCursor()
		self.scrollPartner.unpauseTimeCursor()

	def onMouseDrag_3(self, event):
		dX = self.prevX - event.x
		dY = self.prevY - event.y
		if self.scrollBottom < self.height:dY = 0
		bounds =  self.canvas.bbox(messagehistoryviews_const.MSG_TAG) 
		rightLimit = self.timeCursorX + self.scrollOffsetX + self.scrollAhead
		if rightLimit < bounds[2]:rightLimit = bounds[2] ### ??? needs work
		if dX < 0 and bounds[2] < self.width:dX = 0 # scroll right
		elif dX > 0 and bounds[0] > self.scrollLeft:dX = 0 # scroll left
		if dY < 0 and bounds[3] < self.height:dY = 0 # scroll up
		elif dY > 0 and bounds[1] >= self.scrollTop:dY = 0 # scroll down
		self.prevX, self.prevY = event.x, event.y
		self.scrollXY(dX, dY)

	def scrollXY(self, dX, dY, synch=True):
		"""
		Scrolls view components by specified amounts on x and y axes.
		"""
		self.canvas.move(messagehistoryviews_const.MSG_TAG, dX, dY)
		self.canvas.move(messagehistoryviews_const.MSG_BG_TAG, 0, dY)
		self.canvas.move(messagehistoryviews_const.MSG_LABEL_TAG, 0, dY)
		self.scrollOffsetX += dX
		self.scrollOffsetY += dY
		if synch:self.scrollPartner.synchScroll(dX)

	def synchScroll(self, dX):
		"""
		Synchronises horizontal scroll.
		"""
		self.canvas.move(messagehistoryviews_const.MSG_TAG, dX, 0)
		self.prevX += dX
		self.scrollOffsetX += dX
		
	def pageForward(self):
		"""
		Scrolls the view forward by page size.
		"""
		if (self.width - self.pageX) < (self.horizontalMargin + self.labelAreaWidth):
			self.scrollXY(self.horizontalMargin + self.labelAreaWidth -self.pageX, 0)
		else:self.scrollXY(-self.pageX, 0)

	def pageBack(self):
		"""
		Scrolls the view backward by page size.
		"""
		self.scrollXY(self.pageX, 0)

	def updateView(self, data):
		"""
		Performs update of view in response to change in data source.
		"""
		messageData = data[0]
		self.orderedClientNames = data[1]
		for clientName, messages in messageData.items():
			self.addMessageClient(clientName)
			for message in messages:
				self.addMessage(clientName, message)
		self.clearCanvas()
		self.drawMessagesFresh()
		self.drawTimeCursor()

	def messageSelected(self, messageSprite):
		"""
		Called by message view components when selected, passes info onto client.
		"""
		if self.selectedMessage:self.selectedMessage.deselect()
		self.selectedMessage = messageSprite
		self.selectedMessage.select()
		self.messageLabels.selectLabel(messageSprite.message.sender)

	def messageLabelSelected(self, clientName):
		"""
		Called by message label components when selected, passes info onto client.
		"""
		if self.selectedMessage and self.selectedMessage.message.sender != clientName:self.selectedMessage.deselect()
		if not self.messages.has_key(clientName):return
		self.selectedMessage = self.messages[clientName]['most_recent']
		self.selectedMessage.select()
		self.messageLabels.selectLabel(self.selectedMessage.message.sender)

	def selectItem(self, selectedItem):
		"""
		Selects specified item if present within view.

		This method is used to synchronise selections 
		between the different tracker views.
		"""
		if selectedItem.selectionSource == self:self.messageSelected(selectedItem.item)
		elif self.messages.has_key(selectedItem.itemName):self.messageLabelSelected(selectedItem.itemName)
		else:self.clearSelection()
			
	def clearSelection(self):
		"""
		Clears any selected items in view.
		"""
		if self.selectedMessage:
			self.messageLabels.deselectLabel(self.selectedMessage.message.sender)
			self.selectedMessage.deselect()
			self.selectedMessage = None

	def makeSelection(self, selectedItem):
		"""
		Called by component sprites when selected.

		Sends message to parent for synchronisation.
		"""
		self.context.syncSelection(encodeSelectedItem(selectedItem.message.sender, selectedItem, self))
	
	def coordinateForClient(self, clientName):
		"""
		Returns screen y coordinate for client.
		"""
		if clientName not in self.orderedClientNames: return 0
		return self.verticalMargin + ((self.orderedClientNames.index(clientName) + 1) * (self.messageHeight + self.messageSpacing))

	def startTimeCursor(self, startTime):
		"""
		Sets start time used by view.
		"""
		self.startTime = startTime

	def synchTimeCursor(self, synchTime):
		"""
		Sets time cursor in view to synchronise with C{synchTime}.
		"""
		self.timeCursorX = self.horizontalMargin + self.labelAreaWidth + (synchTime  * self.timeSpaceUnit)
		self.drawTimeCursor()

	def pauseTimeCursor(self):
		"""
		Sets time cursor in pause mode.
		"""
		self.timeCursorPaused = True
		self.canvas.itemconfig(self.timeCursorSprite, fill=demo_const.DFT_COLOUR)

	def unpauseTimeCursor(self):
		"""
		Sets time cursor in pause mode.
		"""
		self.timeCursorPaused = False
		self.canvas.itemconfig(self.timeCursorSprite, fill=demo_const.TIME_CURSOR_COLOUR)

	def drawTimeCursor(self):
		"""
		Draws the time cursor sprite.
		"""
		if self.timeCursorPaused:return
		x = self.timeCursorX + self.scrollOffsetX
		if x > self.width:
			self.pageForward()
			x += self.scrollOffsetX
		if not self.timeCursorSprite:
			self.timeCursorSprite = self.canvas.create_line(x,0, x, self.height, fill=demo_const.TIME_CURSOR_COLOUR, width=1, tag=messagehistoryviews_const.TIME_CURSOR_TAG)
		else:self.canvas.coords(self.timeCursorSprite, x,0, x, self.height)

		

#############################
# MESSAGE HISTORY SPRITE
#############################
class MessageHistorySprite(GenericComponentView):
	"""
	Displays state of message at particular moment in time.
	"""
	def __init__(self, context, message):
		GenericComponentView.__init__(self, context)
		self.message = message
		self.colour = demo_const.DFT_COLOUR
		self.lines = []
	
	def drawFresh(self, locX, locY, startTime, timeSpaceUnit, drawWidth, drawHeight):
		"""
		Draws message view onto canvas.
		"""
		drawHeightMid =  drawHeight/2
		self.x = locX + ((self.message.created - startTime) * timeSpaceUnit)
		self.y = locY + drawHeightMid
		# draw marker for sender
		self.xMin = self.x - 2
		self.yMin = self.y - 2
		self.xMax = self.x + 2
		self.yMax = self.y + 2
		self.sprite = self.canvas.create_rectangle(self.xMin, self.yMin, self.xMax, self.yMax, outline=self.colour, fill=demo_const.VIEW_AREA_COLOUR_01, width=1, tag=messagehistoryviews_const.MSG_TAG)
		if self.selected:self.canvas.itemconfig(self.sprite, outline=demo_const.SPRITE_SELECTED_COLOUR, fill=demo_const.SPRITE_SELECTED_COLOUR)
		self.canvas.tag_bind(self.sprite, '<Button-1>', self.onMouseClick)
		self.canvas.tag_bind(self.sprite, "<Enter>", self.hilight)
		self.canvas.tag_bind(self.sprite, "<Leave>", self.unHighlight)
		# init connector line values
		lineStartY = self.y
		lineEndY = self.y
		# draw recipients
		self.lines = []
		if self.message.recipient == svs_const.RECIPIENTS_ALL:
			recipients = self.context.orderedClientNames
		else:recipients = self.message.recipient
		for rcpCleint in recipients:
			if rcpCleint == self.message.sender:continue
			# draw marker for recipient
			rcpY = self.context.coordinateForClient(rcpCleint) - drawHeightMid
			sprite = self.canvas.create_line(self.xMin, rcpY, self.xMax, rcpY, fill=self.colour, width=1, tag=messagehistoryviews_const.MSG_TAG)
			self.canvas.tag_bind(sprite, '<Button-1>', self.onMouseClick)
			self.canvas.tag_bind(sprite, "<Enter>", self.hilight)
			self.canvas.tag_bind(sprite, "<Leave>", self.unHighlight)
			if self.selected:self.canvas.itemconfig(sprite, fill=demo_const.SPRITE_SELECTED_COLOUR)
			self.lines.append(sprite)
			# update connector line
			if rcpY < lineStartY:lineStartY = rcpY
			if rcpY > lineEndY:lineEndY = rcpY
		# draw connector line
		sprite = self.canvas.create_line(self.x, lineStartY, self.x, lineEndY, fill=self.colour, width=1, tag=messagehistoryviews_const.MSG_TAG)
		if self.selected:self.canvas.itemconfig(sprite, fill=demo_const.SPRITE_SELECTED_COLOUR)
		self.canvas.tag_bind(sprite, '<Button-1>', self.onMouseClick)
		self.canvas.tag_bind(sprite, "<Enter>", self.hilight)
		self.canvas.tag_bind(sprite, "<Leave>", self.unHighlight)
		self.lines.append(sprite)
		self.canvas.lift(self.sprite)

	def onMouseClick(self, event):
		"""
		Responds to double-click from mouse.
		"""
		self.context.makeSelection(self)

	def hilight(self, event):
		"""
		Highlights sprite.
		"""
		if self.selected:return
		self.canvas.itemconfig(self.sprite, outline=demo_const.SPRITE_HILIGHT_COLOUR)
		for lineSprite in self.lines:
			self.canvas.itemconfig(lineSprite, fill=demo_const.SPRITE_HILIGHT_COLOUR)
		self.context.messageLabels.hilightLabel(self.message.sender)

	def unHighlight(self, event):
		"""
		Highlights sprite.
		"""
		if self.selected:return
		self.canvas.itemconfig(self.sprite, outline=self.colour)
		for lineSprite in self.lines:
			self.canvas.itemconfig(lineSprite, fill=self.colour)
		self.context.messageLabels.unHilightLabel(self.message.sender)

	def select(self):
		"""
		Selects sprite.
		"""
		self.selected = True
		self.canvas.itemconfig(self.sprite, fill=demo_const.SPRITE_SELECTED_COLOUR, outline=demo_const.SPRITE_SELECTED_COLOUR)
		for lineSprite in self.lines:
			self.canvas.itemconfig(lineSprite, fill=demo_const.SPRITE_SELECTED_COLOUR)

	def deselect(self):
		"""
		Deselects sprite.
		"""
		if not self.selected:return
		self.selected = False
		self.canvas.itemconfig(self.sprite, outline=self.colour, fill=demo_const.VIEW_AREA_COLOUR_01)
		for lineSprite in self.lines:
			self.canvas.itemconfig(lineSprite, fill=self.colour)

#############################
# MESSAGE HISTORY LABELS
#############################
class MessageHistoryLabels(GenericComponentView):
	"""
	Displays name of message.
	"""
	def __init__(self, context):
		GenericComponentView.__init__(self, context)
		self.colour = demo_const.DFT_COLOUR
		self.labelSprites = {}
		self.selectedLabel = None

	def addLabel(self, labelText):
		"""
		Adds label for message listing.
		"""
		self.labelSprites[labelText] = None
		
	
	def drawFresh(self, locX, locY, spacer, orderedClientNames):
		"""
		Draws message labels onto canvas.
		"""
		self.xMin = locX
		self.yMin = locY
		self.spacer = spacer
		verticalSpacer = spacer/2
		for labelText in orderedClientNames:
			sprite = self.canvas.create_text(self.xMin, self.yMin + verticalSpacer, fill=self.colour, text=labelText, anchor='w' , tag=messagehistoryviews_const.MSG_LABEL_TAG)
			self.labelSprites[labelText] = sprite
			# handlers for enter and leave actions
			def onEnter(event, self=self, sprite=sprite):
                		self.hilightSprite(sprite)
			def onLeave(event, self=self, sprite=sprite):
                		self.unHighlightSprite(sprite)
			def onDoubleClick(event, self=self, sprite=sprite):
                		self.selectSprite(sprite)
			# end handlers
			self.canvas.tag_bind(sprite, "<Enter>", onEnter)
			self.canvas.tag_bind(sprite, "<Leave>", onLeave)
			#self.canvas.tag_bind(sprite, '<Double-1>', onDoubleClick)
			verticalSpacer += self.spacer

	
	def hilightLabel(self, labelText):
		"""
		Highlights label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.hilightSprite(sprite)

	def hilightSprite(self, sprite):
		"""
		Highlights sprite.
		"""
		if self.selectedLabel == sprite:return
		self.canvas.itemconfig(sprite, fill=demo_const.SPRITE_HILIGHT_COLOUR)

	def unHilightLabel(self, labelText):
		"""
		Unhighlights label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.unHighlightSprite(sprite)

	def unHighlightSprite(self, sprite):
		"""
		Unhighlights sprite.
		"""
		if self.selectedLabel == sprite:return
		self.canvas.itemconfig(sprite, fill=self.colour)

	def selectLabel(self, labelText):
		"""
		Selects label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.selectSprite(sprite)

	def selectSprite(self, sprite):
		"""
		Selects sprite.
		"""
		if self.selectedLabel:self.canvas.itemconfig(self.selectedLabel, fill=self.colour)
		self.selectedLabel = sprite
		self.canvas.itemconfig(self.selectedLabel, fill=demo_const.SPRITE_SELECTED_COLOUR)

	def deselectLabel(self, labelText):
		"""
		Deselects label.
		"""
		sprite = self.labelSprites.get(labelText, None)
		if sprite:self.deselectSprite(sprite)

	def deselectSprite(self, sprite):
		"""
		Deselects sprite.
		"""
		if self.selectedLabel:self.canvas.itemconfig(self.selectedLabel, fill=self.colour)
		self.selectedLabel = None





