# svs_demogame.gameworld

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Main manager class for game.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from time import time

# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.terrain import *
from svs_demogame.agents import *
from svs_demogame.gamesettings import GameSettings

		
#############################
# GAME WORLD CLASS
#############################	
class Gameworld:
	def __init__(self):
		self.name = None
		self.terrain = None
		self.agents = None
		self.prevTime = 0
		self.changes = None
		self.running = False
		self.agentGroupManager = AgentGroupManager()

	def createFromFile(self, filepath):
		"""
		Creates a gameworld from data read in from external file.

		Reads settings from C{filepath} which should be
		to a file containing the following:
		
		Essential:

		- dimensions of world
		- number of areas in x and y axes
		- number of agents at start
		- start location for each agent

		Optional:

		- density values for areas
		"""
		settings = GameSettings()
		settings.loadFromFile(filepath, self)


	def createTerrain(self, dimX, dimY, areasX, areasY):
		"""
		Creates a gameworld of specified dimensions.
		"""
		self.age = 0
		self.terrain = Terrain(dimX, dimY, areasX, areasY)
		self.agents = {}
		return self.terrain

	def addAgent(self, agent):
		"""
		Adds agent to gameworld.

		Makes sure that the agent belongs to and fits within the gameworld terrain.
		"""
		agent.terrain = self.terrain
		agent.setLocation(agent.location.x, agent.location.y)
		self.agents[agent.idNum] = agent

	def getAreaAtIndices(self, index_x, index_y):
		"""
		Returns area at specified index coordinates.  If not
		found returns None.
		"""
		return self.terrain.getAreaAtIndices(index_x, index_y)

	def getAreaWithId(self, areaId):
		"""
		Returns area with specified id number.
		"""
		return self.terrain.getAreaWithId(areaId)

	def getAgentWithId(self, areaId):
		"""
		Returns agent with specified id number.
		"""
		return self.agents.get(areaId, None)

	def addAgentGroup(self, groupName, groupColour):
		"""
		Adds new agent group.

		See L{svs_demogame.agents.AgentGroupManager.addAgentGroup}.
		"""
		self.agentGroupManager.addAgentGroup(groupName, groupColour)


	def start(self):
		"""
		Starts gameworld.
		"""
		self.terrain.start()
		for agent in self.agents.values():agent.startPlay()
		self.prevTime = time()
		self.running = True
		self.reportChange(self.terrain.getChanges(), demo_const.TERRAIN_LABEL)
		self.reportChange(self.running, demo_const.WORLD_RUNNING)

	def stop(self):
		"""
		Stops gameworld.
		"""
		self.running = False
		self.changes = None
		for agent in self.agents.values():agent.stopPlay()
		self.terrain.stop()
		self.reportChange(self.terrain.getChanges(), demo_const.TERRAIN_LABEL)
		self.reportChange(self.running, demo_const.WORLD_RUNNING)

	def update(self):
		"""
		Updates world and all active components.
		"""
		if not self.running:return
		timeNow = time()
		timeInterval = timeNow - self.prevTime
		self.changes = None
		self.terrain.update(timeInterval)
		for agent in self.agents.values():agent.update(timeInterval)
		self.reportChange(self.terrain.getChanges(), demo_const.TERRAIN_LABEL)
		self.prevTime = timeNow
		

	def reportChange(self, changeData, keyName):
		"""
		Collects changes in condition of terrain and agents.
		"""
		if changeData == None:return
		if not self.changes:self.changes = {}
		self.changes[keyName] = changeData

	def getChanges(self):
		"""
		Get reported changes in game world.
		"""
		if not self.changes:return None
		if len(self.changes) == 0:return None
		return self.changes

	def getModel(self):
		"""
		Returns model of game world in current condition.
		"""
		model = {demo_const.WORLD_NAME:self.name, demo_const.WORLD_RUNNING:self.running, demo_const.TERRAIN_LABEL:self.terrain.getModel(), demo_const.AGENTS_LABEL:[]}
		for agent in self.agents.values():model[demo_const.AGENTS_LABEL].append(agent.getProfile())
		return model


