# svs_demogame.gameviews

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Graphical representation of game elements.

The game is displayed using the Tkinter library:
U{http://www.pythonware.com/library/tkinter/introduction/}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Tkinter import *
import math

# internal imports
from svs_demogame.views import GenericComponentView, encodeSelectedItem
from svs_demogame.utils import demo_const
from svs_core.utilities.lib import Constants
#from svs_core.geometry.geomlib import geom_const
from svs_simulation.numdata.geomlib import geom_const


#############################
# CONSTANTS
#############################
game_view_elements = Constants()
game_view_elements.TERRAIN_BG = 'terrain_bg'
game_view_elements.TERRAIN_AREA = 'terrain_area'
game_view_elements.AGENT_BODY = 'agent_body'
game_view_elements.AGENT_POINTER = 'agent_pointer'


#############################
# CLASSES
#############################
class GameView:
	"""
	Handles drawing of all game components, and can be added
	to other Tkinter components.
	"""
	def __init__(self, parent=None, context=None):
		self.context = context
		self.width = 0
		self.height = 0
		self.canvas = Canvas(parent, bg=demo_const.BG_COLOUR, borderwidth=0, highlightthickness=0)
		self.canvas.bind('<Configure>', self.canvasAdjusted)
		self.areaViews = {}
		self.agentViews = {}
		self.areaSpacing = 1
		self.areasX = 0
		self.areasY = 0
		self.selectedArea = None
		self.selectedAgent = None


	def canvasAdjusted(self, args=None):
		"""
		Responds to main window being adjusted in size.
		"""
		self.width = self.canvas.winfo_width()
		self.height = self.canvas.winfo_height()
		if not self.areasX or not self.areasY:return
		if self.height <= self.width:
			self.marginY = self.height/20.0
			self.worldViewDim = self.height - (self.marginY * 2)
			self.marginX = (self.width - self.worldViewDim)/2.0
			self.areaDim = ((self.worldViewDim - self.areaSpacing) / self.areasY) - self.areaSpacing
		else:
			self.marginX = self.width/20.0
			self.worldViewDim = self.width - (self.marginX * 2)
			self.marginY = (self.height - self.worldViewDim)/2.0
			self.areaDim = ((self.worldViewDim - self.areaSpacing) / self.areasX) - self.areaSpacing
		self.drawTerrainFresh()
		self.drawAgentsFresh()
	
	def clearCanvas(self):
		"""
		Clears entire game world from canvas.
		"""
		self.canvas.delete(game_view_elements.TERRAIN_BG)
		self.canvas.delete(game_view_elements.TERRAIN_AREA)
		self.canvas.delete(game_view_elements.AGENT_BODY)
		self.canvas.delete(game_view_elements.AGENT_POINTER)

	def drawTerrainFresh(self):
		"""
		Clears canvas and draws the game world, scaled to canvas area.
		"""
		self.clearCanvas()
		# terrain background
		self.sprite = self.canvas.create_rectangle(self.marginX, self.marginY, self.marginX + self.worldViewDim, self.marginY + self.worldViewDim, fill=demo_const.DFT_COLOUR, width=0, tag=game_view_elements.TERRAIN_BG)
		self.canvas.tag_bind(self.sprite, '<Button-1>', self.onSingleClick)
		# terrain areas
		aIdx = 0
		areas = self.areaViews.values()
		for aX in range(self.areasX):
			aXLoc = (self.marginX + self.areaSpacing) + (aX * (self.areaDim + self.areaSpacing))
			for aY in range(self.areasY):
				aYLoc = (self.marginY + self.areaSpacing) + (aY * (self.areaDim + self.areaSpacing))
				areas[aIdx].drawFresh(aXLoc, aYLoc, aXLoc + self.areaDim, aYLoc + self.areaDim)
				aIdx += 1
		
		
	def setWorldDimensions(self, worldDimX, worldDimY):
		"""
		Sets dimensions of game world.
		"""
		self.worldDimX = worldDimX 
		self.worldDimY = worldDimY

	def worldLocationToScreen(self, locX, locY):
		"""
		Converts location in world coordinates to screen coordinates.
		"""
		drawX = ((locX / self.worldDimX) * self.worldViewDim) + self.marginX
		drawY = ((locY / self.worldDimY) * self.worldViewDim) + self.marginY
		return (drawX, drawY)

	def displayPosition(self, viewX, viewY):
		"""
		Displays gameworld coordinates for specified view location.
		"""
		worldX = ((viewX - self.marginX) / self.worldViewDim) * self.worldDimX
		worldY = ((viewY - self.marginY) / self.worldViewDim) * self.worldDimY
		self.context.displayPosition(worldX, worldY)

	def onSingleClick(self, event):
		"""
		Responds to single-click from mouse.
		"""
		self.displayPosition(event.x, event.y)


	def setupTerrainAreas(self, areasX, areasY, areaData):
		"""
		Maps gamewolrd dimensions to canvas, creates terrainAreaViews.
		"""
		self.areasX = areasX
		self.areasY = areasY
		self.selectedArea = None
		aIdx = 0
		for x in range(self.areasX):
			for y in range(self.areasY):
				areaView = TerrainAreaView(self, idNum=areaData[aIdx][demo_const.AREA_ID_LABEL], density=areaData[aIdx][demo_const.DENSITY_LABEL])
				self.areaViews[areaData[aIdx][demo_const.AREA_ID_LABEL]] = areaView
				aIdx += 1


	def setupAgents(self, agentData):
		"""
		Creates agent sprites.
		"""
		self.agentViews = {}
		for aData in agentData:
			agentView = AgentView(self, aData[demo_const.AGENT_ID_LABEL])
			agentView.setLocation(aData[demo_const.LOC_X_LABEL], aData[demo_const.LOC_Y_LABEL])
			agentView.setOrientation(aData[demo_const.FACING_LABEL])
			self.agentViews[aData[demo_const.AGENT_ID_LABEL]] = agentView

	def drawAgentsFresh(self):
		"""
		Draws new view graphics for agents.
		"""
		for agent in self.agentViews.values():
			agent.drawFresh(self.areaDim, self.areaDim)
		

	def updateTerrainAreas(self, areaData):
		"""
		Updates terrainAreaViews with specified data.
		"""
		if not areaData:return
		for data in areaData:
			area = self.areaViews.get(data[demo_const.AREA_ID_LABEL], None)
			if not area:continue
			area.setDensity(data[demo_const.DENSITY_LABEL])

	def updateAgents(self, agentData):
		"""
		Updates agentViews with specified data.
		"""
		if not agentData:return
		for data in agentData:
			agent = self.agentViews.get(data[demo_const.AGENT_ID_LABEL], None)
			if not agent:continue
			agent.setLocation(data[demo_const.LOC_X_LABEL], data[demo_const.LOC_Y_LABEL])
			agent.setOrientation(data[demo_const.FACING_LABEL])
				

	def areaSelected(self, area):
		"""
		Called by area view components when selected, passes info onto client.
		"""
		if self.selectedArea:self.selectedArea.deselect()
		if self.selectedAgent:self.selectedAgent.deselect()
		self.selectedArea = area
		self.selectedAgent = None
		self.selectedArea.select()
		self.context.areaSelected(area.idNum)

	def agentSelected(self, agent):
		"""
		Called by agent view components when selected, passes info onto client.
		"""
		if self.selectedAgent:self.selectedAgent.deselect()
		if self.selectedArea:self.selectedArea.deselect()
		self.selectedAgent = agent
		self.selectedArea = None
		self.selectedAgent.select()
		self.context.agentSelected(agent.idNum)

	def selectAreaWithIdNum(self, idNum):
		"""
		Selects area with matching id number.
		"""
		area = self.areaViews.get(idNum, None)
		if area:self.areaSelected(area)

	def selectAgentWithIdNum(self, idNum):
		"""
		Selects agent with matching id number.
		"""
		agent = self.agentViews.get(idNum, None)
		if agent:self.agentSelected(agent)

	def clearSelection(self):
		"""
		Clears any selected items in view.
		"""
		if self.selectedAgent:self.selectedAgent.deselect()
		if self.selectedArea:self.selectedArea.deselect()

	def getScriptForAgent(self, agent):
		"""
		Gets script for selected agent.
		"""
		if self.selectedAgent is not agent:
			self.selectedAgent.deselect()
			self.selectedAgent = agent
			self.selectedAgent.select()
		self.context.getScriptForAgent(agent.idNum)


class TrackerGameView(GameView):
	"""
	Extended version of L{GameView} providing additional methods for Tracker client.
	"""
	def __init__(self, parent=None, context=None):
		GameView.__init__(self, parent=parent, context=context)

	def areaSelected(self, area):
		"""
		Called by area view components when selected, passes info onto client.
		"""
		if self.selectedArea:self.selectedArea.deselect()
		if self.selectedAgent:self.selectedAgent.deselect()
		self.selectedArea = area
		self.selectedAgent = None
		self.selectedArea.select()
		self.makeSelection("<area_%d>" % area.idNum, area)

	def agentSelected(self, agent):
		"""
		Called by agent view components when selected, passes info onto client.
		"""
		if self.selectedAgent:self.selectedAgent.deselect()
		if self.selectedArea:self.selectedArea.deselect()
		self.selectedAgent = agent
		self.selectedArea = None
		self.selectedAgent.select()
		self.makeSelection("<agent_%d>" % agent.idNum, agent)

	def selectItem(self, selectedItem):
		"""
		Selects specified item if present within view.

		This method is used to synchronise selesctions 
		between the different tracker views.
		"""
		if selectedItem.selectionSource == self:return
		if selectedItem.itemName.startswith('<area_'):
			idNum = selectedItem.itemName[selectedItem.itemName.index('_')+1:-1]
			try:self.selectAreaWithIdNum(int(idNum))
			except:self.clearSelection()
		elif selectedItem.itemName.startswith('<agent_'):
			idNum = selectedItem.itemName[selectedItem.itemName.index('_')+1:-1]
			try:self.selectAgentWithIdNum(int(idNum))
			except:self.clearSelection()
		else:self.clearSelection()

	def makeSelection(self, identifier, selectedItem):
		"""
		Called by component sprites when selected.

		Sends message to parent for synchronisation.

		NOTE: this version of the method is slightly different from
		that of others, as normally an identifier is dervied from the
		C{selectedItem} object itself. This should really be fixed in
		later versions.
		"""
		self.context.syncSelection(encodeSelectedItem(identifier, selectedItem, self))
		

class TerrainAreaView(GenericComponentView):
	"""
	Draws terrain area.
	"""
	def __init__(self, context, idNum=0, density=0.0, gameData=None, ):
		GenericComponentView.__init__(self, context)
		self.idNum = idNum
		colourValue = (1.0 - density) * 255
		self.colour = "#%02x%02x%02x" % (colourValue, colourValue, colourValue)
		self.selected = False
	

	def updateWithData(self, data):
		"""
		Handles info from L{svs_demogame.terrain.TerrainAreaState} object.
		"""
		if data.idNum != self.idNum: return
		colourValue = data.density * 255
		self.colour = "#%02x%02x%02x" % (colourValue, colourValue, colourValue)
		self.context.canvas.itemconfig(self.sprite, fill=self.colour)

	def setDensity(self, density, updateView=True):
		"""
		Changes colour of area to represent density.

		Black = density of 1.0, white = density of 0.0
		"""
		colourValue = (1.0 - density) * 255
		self.colour = "#%02x%02x%02x" % (colourValue, colourValue, colourValue)
		if updateView:self.context.canvas.itemconfig(self.sprite, fill=self.colour)
		

	def onSingleClick(self, event):
		"""
		Responds to single-click from mouse.
		"""
		self.context.displayPosition(event.x, event.y)


	def onDoubleClick(self, event):
		"""
		Responds to double-click from mouse.
		"""
		self.context.areaSelected(self)


	def drawFresh(self, xMin, yMin, xMax, yMax):
		"""
		Creates component on display context.
		"""
		self.xMin = xMin
		self.yMin = yMin
		self.xMax = xMax
		self.yMax = yMax
		self.sprite = self.canvas.create_rectangle(xMin, yMin, xMax, yMax, fill=self.colour, width=0, tag=game_view_elements.TERRAIN_AREA)
		self.canvas.tag_bind(self.sprite, '<Button-1>', self.onSingleClick)
		self.canvas.tag_bind(self.sprite, '<Double-1>', self.onDoubleClick)
		if self.selected:self.context.canvas.itemconfig(self.sprite, fill=demo_const.SPRITE_SELECTED_COLOUR)
		
	

class AgentView(GenericComponentView):
	"""
	Draws agent.
	"""
	def __init__(self, context, idNum, gameData=None):
		GenericComponentView.__init__(self, context)
		self.idNum = idNum
		self.colour = 'black'
		self.facing = 0
		self.spritePointer = None

	def setLocation(self, locX, locY):
		"""
		Handles info from L{svs_demogame.agents.AgentState} object.
		"""
		self.worldX = locX
		self.worldY = locY
		self.updateLocation()

	def setOrientation(self, newfacing):
		"""
		Sets angle in which agent is facing.
		"""
		self.facing = newfacing
		self.updateDirection()

	def drawFresh(self, drawWidth, drawHeight):
		"""
		Creates component on display context.
		"""
		# calc positions
		self.drawWidth = drawWidth
		self.drawHeight = drawHeight
		self.drawX, self.drawY = self.context.worldLocationToScreen(self.worldX, self.worldY)
		self.radius = self.drawWidth
		self.xMin = self.drawX - (self.drawWidth / 2.0)
		self.yMin = self.drawY - (self.drawHeight / 2.0)
		self.xMax = self.xMin + self.drawWidth
		self.yMax = self.yMin + self.drawHeight
		# draw body
		self.sprite = self.context.canvas.create_oval(self.xMin, self.yMin, self.xMax, self.yMax, outline=self.colour, fill='white', width=1, tag=game_view_elements.AGENT_BODY)
		self.canvas.tag_bind(self.sprite, '<Button-1>', self.onSingleClick)
		self.canvas.tag_bind(self.sprite, '<Double-1>', self.onDoubleClick)
		# draw directional pointer
		self.spritePointer = self.context.canvas.create_line(self.drawX, self.drawY, self.xMin, self.yMax, fill=self.colour, width=1, tag=game_view_elements.AGENT_POINTER)
		self.updateDirection()
		
	def updateDirection(self):
		"""
		Draws updated direction pointer for agent.
		"""
		if not self.spritePointer:return
		self.pointerX = self.drawX + (self.radius * math.cos(geom_const.deg2rad * self.facing))
		self.pointerY = self.drawY + (self.radius * math.sin(geom_const.deg2rad * self.facing))
		self.context.canvas.coords(self.spritePointer, self.drawX, self.drawY, self.pointerX, self.pointerY)
		
	def updateLocation(self):
		"""
		Draws updated position of agent.
		"""
		if not self.sprite:return
		self.drawX, self.drawY = self.context.worldLocationToScreen(self.worldX, self.worldY)
		self.xMin = self.drawX - (self.drawWidth / 2.0)
		self.yMin = self.drawY - (self.drawHeight / 2.0)
		self.xMax = self.xMin + self.drawWidth
		self.yMax = self.yMin + self.drawHeight
		self.canvas.coords(self.sprite, self.xMin, self.yMin, self.xMax, self.yMax)

	def onSingleClick(self, event):
		"""
		Responds to single-click from mouse.
		"""
		self.context.agentSelected(self)

	def onDoubleClick(self, event):
		"""
		Responds to double-click from mouse.
		"""
		self.context.canvas.itemconfig(self.sprite, outline=demo_const.SPRITE_SELECTED_COLOUR)
		self.context.canvas.itemconfig(self.spritePointer, fill=demo_const.SPRITE_SELECTED_COLOUR)
		self.context.getScriptForAgent(self)

	def select(self):
		"""
		Selects sprite.
		"""
		self.selected = True
		self.context.canvas.itemconfig(self.sprite, outline=demo_const.SPRITE_SELECTED_COLOUR)
		self.context.canvas.itemconfig(self.spritePointer, fill=demo_const.SPRITE_SELECTED_COLOUR)

	def deselect(self):
		"""
		Deselects sprite.
		"""
		if not self.selected:return
		self.selected = False
		self.context.canvas.itemconfig(self.sprite, outline=self.colour, fill='white')
		self.context.canvas.itemconfig(self.spritePointer, fill=self.colour)
