# svs_demogame.gameworld

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Loads and parses settings file for game.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from string import upper, lower, split, strip
from xml.sax import saxutils
from xml.sax import make_parser
from xml.sax.handler import feature_namespaces
from xml.dom.minidom import getDOMImplementation

# local imports
from svs_demogame.utils import xml_tags
from svs_demogame.scripts import makeScriptFromCode, makeEmptyScript
from svs_demogame.agents import Agent
from svs_core.filehandling.loaders import *


#############################
# GAME WORLD SETTINGS
#############################
class GameSettings:
	"""
	Class encapsulating settings for gameworld.
	"""
	def __init__(self):
		self.name = ''
		self.dimension_x = 0
		self.dimension_y = 0
		self.areas_x = 0
		self.areas_y = 0
		self.agents = []
		self.areas = []
		self.default_agent = None
		self.default_area = None
		self.groups = []
		self.scripts = {}

	def setDefaultAgentValues(self, facing=0.0, speed=0.0, script=None):
		"""
		Sets default values that can be applied to all agents.
		"""
		self.default_agent = Bunch(facing=facing, speed=speed, script=script)

	def setDefaultAreaValues(self, density=0.0, script=None):
		"""
		Sets default values that can be applied to all areas.
		"""
		self.default_area = Bunch(density=density, script=script)
		
	def addArea(self, index_x=0, index_y=0, density=0.0, script=None, areaObject=None):
		"""
		Adds information about an area.
		"""
		if areaObject:self.areas.append(areaObject)
		else:self.areas.append(Bunch(index_x=index_x, index_y=index_y, density=density, script=script))

	def addAgent(self, x=0.0, y=0.0, radius=1.0, facing=0.0, name=None, speed=0.0, script=None, agentObject=None):
		"""
		Adds information about an agent.
		"""
		if agentObject:self.agents.append(agentObject)
		else:self.agents.append(Bunch(x=x, y=y, radius=radius, facing=facing, name=name, speed=speed, script=script)) 

	def addAgentGroup(self, name, colour):
		"""
		Adds a group that agents can join.
		"""
		self.groups.append(Bunch(name=name, colour=colour))

	def addScript(self, name, code):
		"""
		Adds a script that can be attached to an area or agent.
		"""
		if name == 'noscript':return
		self.scripts[name] = code

	def applyScripts(self):
		"""
		Applies scripts to areas and agents.
		"""
		if self.default_area:
			if self.scripts.has_key(self.default_area.script):
				self.default_area.script = self.scripts[self.default_area.script]
		for agent in self.agents:
			if not agent.script:agent.script = self.default_agent.script
			elif agent.script == 'noscript':agent.script = None
			elif self.scripts.has_key(agent.script):agent.script = self.scripts[agent.script]
		for area in self.areas:
			if not area.script:area.script = self.default_area.script
			elif area.script == 'noscript':area.script = None
			elif self.scripts.has_key(area.script):area.script = self.scripts[area.script]

	def loadFromFile(self, filepath, gameWorld):
		"""
		Loads and parses XML file storing settings.

		Note: if the file is malformed this will throw errors 
		which should be caught at a higher level.
		"""
		loader = GameSettingsLoaderXML(self)
		loader.read(filepath)
		self.applyScripts()
		gameWorld.name = self.name
		terrain = gameWorld.createTerrain(self.dimension_x, self.dimension_y, self.areas_x, self.areas_y)

		if self.default_area:
			terrain.setAreaDefaults(density=self.default_area.density, script=makeScriptFromCode(self.default_area.script))

		for areaData in self.areas:
			gameArea = gameWorld.getAreaAtIndices(areaData.index_x, areaData.index_y)
			if not gameArea:continue
			gameArea.setDensity(areaData.density)
			if not areaData.script:gameArea.clearScripts()
			else:
				gameArea.clearScripts()
				gameArea.setScript(makeScriptFromCode(areaData.script))
		for agentData in self.agents:
			agent = Agent(agentData.name, terrain)
			agent.setLocation(agentData.x, agentData.y)
			agent.setDirection(agentData.facing)
			agent.speed(agentData.speed)
			agent.setScript(makeScriptFromCode(agentData.script))
			gameWorld.addAgent(agent)
		for groupData in self.groups:
			agentGroup = AgentGroup(groupData.name, groupData.colour)
			gameWorld.addAgentGroup(agentGroup)
			
				
			 

#############################
# LOADER AND PARSER
#############################
class GameSettingsLoaderXML(DocumentLoaderXML):
	"""
	Class for reading and writing game settings in xml format.
	"""
	def __init__(self, target):
		DocumentLoaderXML.__init__(self, target, GameSettingsXMLContentHandler)

		
	def toxml(self):
		"""
		Converts game settings to xml text.
		
		@rtype:	string
		@return: actor data in xml encoding
		"""
		return self.target.data.toxml()
		
		
class GameSettingsXMLContentHandler(GenericXMLContentHandler):
	"""
	Handler for parser game settings xml document.
	"""
	
	def __init__(self, target):
		GenericXMLContentHandler.__init__(self, target)
		self.chardata = ""
		self.tmpdata = None
		self.parentNode = None
	
	##########################
	# GENERAL XML HANDLING
	##########################
	def startElement(self, name, attrs):
		# clean char data
		self.chardata = ""
		# parse tags
		name = upper(name)
		if name == xml_tags.GAMEWORLD_TAG:self.handleGameWorldTag(attrs)
		elif name == xml_tags.AGENTS_TAG:self.handleAgentsTag(attrs)
		elif name == xml_tags.AGENT_TAG:self.handleAgentTag(attrs)
		elif name == xml_tags.AREAS_TAG:self.handleAreasTag(attrs)
		elif name == xml_tags.AREA_TAG:self.handleAreaTag(attrs)
		elif name == xml_tags.SCRIPTS_TAG:self.handleScriptsTag(attrs)
		elif name == xml_tags.SCRIPT_TAG:self.handleScriptTag(attrs)

		
	def endElement(self, name):
		# parse tags
		name = upper(name)
		self.chardata = strip(self.chardata)
		if len(self.chardata) > 0:
			if name == xml_tags.SCRIPT_TAG:self.handleScriptTagEnd(self.chardata)
			# clean char data
			self.chardata = ""
		if name == xml_tags.AREA_TAG:self.handleAreaTagEnd()
		elif name == xml_tags.AGENT_TAG:self.handleAgentTagEnd()

	
	def characters(self, content):
		self.chardata += content.encode("utf-8")
		
	
	##########################
	# SPECIFIC TAG HANDLING
	##########################	
	def handleGameWorldTag(self, attrs):
		"""
		Parses C{GAMEWORLD} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = xml_tags.GAMEWORLD_TAG
		try:
			name = attrs.get(xml_tags.NAME_TAG)
			dim_x = attrs.get(xml_tags.DIM_X_TAG)
			dim_y = attrs.get(xml_tags.DIM_Y_TAG)
			areas_x = attrs.get(xml_tags.AREAS_X_TAG)
			areas_y = attrs.get(xml_tags.AREAS_Y_TAG)
		except ValueError:raise DocumentLoadingException("Essential data missing from input document.")

		if name:self.target.name = name.encode("utf-8")
			
		if dim_x:
			try:self.target.dimension_x = float(dim_x.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		
		if dim_y:
			try:self.target.dimension_y = float(dim_y.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)

		if areas_x:
			try:self.target.areas_x = int(areas_x.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		
		if areas_y:
			try:self.target.areas_y = int(areas_y.encode("utf-8"))
			except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		
	
	def handleAgentsTag(self, attrs):
		"""
		Parses C{AGENTS} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = xml_tags.AGENTS_TAG
		self.target.agents = []
		facing = attrs.get(xml_tags.FACING_TAG, '0.0')
		speed = attrs.get(xml_tags.SPEED_TAG, '0.0')
		script = attrs.get(xml_tags.SCRIPT_TAG, None)
		try:facing = float(facing.encode("utf-8"))
		except ValueError:facing = 0.0
		try:speed = float(speed.encode("utf-8"))
		except ValueError:speed = 0.0
		if script:script = script.encode("utf-8")
		self.target.setDefaultAgentValues(facing, speed, script)

	def handleAgentTag(self, attrs):
		"""
		Parses C{AGENT} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = xml_tags.AGENT_TAG
		name = attrs.get(xml_tags.NAME_TAG, None)
		loc_x = attrs.get(xml_tags.LOC_X_TAG, '0.0')
		loc_y = attrs.get(xml_tags.LOC_Y_TAG, '0.0')
		radius = attrs.get(xml_tags.RADIUS_TAG, '1.0')
		facing = attrs.get(xml_tags.FACING_TAG, '%f' % self.target.default_agent.facing)
		speed = attrs.get(xml_tags.SPEED_TAG, '%f' % self.target.default_agent.speed)
		script = attrs.get(xml_tags.SCRIPT_TAG, self.target.default_agent.script)
		if name:name = name.encode("utf-8")
		try:loc_x = float(loc_x.encode("utf-8"))
		except ValueError:loc_x = 0.0
		try:loc_y = float(loc_y.encode("utf-8"))
		except ValueError:loc_y = 0.0
		try:radius = float(radius.encode("utf-8"))
		except ValueError:radius = 1.0
		try:facing = float(facing.encode("utf-8"))
		except ValueError:facing = self.target.default_agent.facing
		try:speed = float(speed.encode("utf-8"))
		except ValueError:speed = self.target.default_agent.speed
		if script:script = script.encode("utf-8")
		self.tmpdata = Bunch(x=loc_x, y=loc_y, radius=radius, facing=facing, name=name, speed=speed, script=script)

	def handleAgentTagEnd(self):
		"""
		Stores data collected from C{AGENT} tag along with script code for agent.
		"""
		if not self.tmpdata:return
		self.target.addAgent(agentObject=self.tmpdata)
		self.tmpdata = None
		self.currentNode = None
		

	def handleAreasTag(self, attrs):
		"""
		Parses C{AREAS} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = xml_tags.AREAS_TAG
		self.target.areas = []
		density = attrs.get(xml_tags.DENSITY_TAG, '0.0')
		script = attrs.get(xml_tags.SCRIPT_TAG, None)
		try:density = float(density.encode("utf-8"))
		except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		if script:script = script.encode("utf-8")
		self.target.setDefaultAreaValues(density=density, script=script)

	def handleAreaTag(self, attrs):
		"""
		Parses C{AREA} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = xml_tags.AREA_TAG
		index_x = attrs.get(xml_tags.INDEX_X_TAG, '0.0')
		index_y = attrs.get(xml_tags.INDEX_Y_TAG, '0.0')
		density = attrs.get(xml_tags.DENSITY_TAG, '%f' % self.target.default_area.density)
		script = attrs.get(xml_tags.SCRIPT_TAG, self.target.default_area.script)
		try:index_x = int(index_x.encode("utf-8"))
		except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		try:index_y = int(index_y.encode("utf-8"))
		except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		try:density = float(density.encode("utf-8"))
		except ValueError:raise malformedInputForXMLNodeException(self.currentNode)
		if script:script = script.encode("utf-8")
		self.tmpdata = Bunch(index_x=index_x, index_y=index_y, density=density, script=script)

	def handleAreaTagEnd(self):
		"""
		Stores data collected from C{AREA} tag along with script code for area.
		"""
		if not self.tmpdata:return
		self.target.addArea(areaObject=self.tmpdata)
		self.tmpdata = None
		self.currentNode = None

	def handleScriptsTag(self, attrs):
		"""
		Parses C{SCRIPTS} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.currentNode = xml_tags.SCRIPTS_TAG
		self.target.scripts = {}
	
	def handleScriptTag(self, attrs):
		"""
		Parses C{SCRIPT} tag.
		
		@type 	attrs: list
		@param 	attrs: attributes for tag
		"""
		self.parentNode = self.currentNode
		if self.parentNode == xml_tags.SCRIPTS_TAG:
			name = attrs.get(xml_tags.NAME_TAG, None)
			if name:name = name.encode("utf-8")
			self.tmpdata = Bunch(name=name, script=None)

	def handleScriptTagEnd(self, chardata):
		"""
		Adds script onto data collected for area.
		"""
		if self.parentNode == xml_tags.SCRIPTS_TAG:
			if not self.tmpdata:return
			try:self.tmpdata.script = chardata
			except AttributeError:return
			else:self.target.addScript(self.tmpdata.name, self.tmpdata.script)
		elif self.parentNode == xml_tags.AREA_TAG:
			if not self.tmpdata:return
			try:self.tmpdata.script = chardata
			except AttributeError:return
		elif self.parentNode == xml_tags.AGENT_TAG:
			if not self.tmpdata:return
			try:self.tmpdata.script = chardata
			except AttributeError:return
