# svs_demogame.gamelogicviews

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Display current changes in overall state of game logic, etc.

The game is displayed using the Tkinter library:
U{http://www.pythonware.com/library/tkinter/introduction/}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Tkinter import *
from time import time, gmtime, strftime

# internal imports
from svs_demogame.utils import demo_const


#############################
# CLASSES
#############################
class GameLogicView:
	"""
	Displays data about changes in game logic.
	"""
	def __init__(self, parent, context):
		self.context = context
		self.startTime = None
		self.running = False
		self.horizontalMargin = 10
		self.verticalMargin = 5
		self.labelAreaWidth = 80
		self.lineHeight = 10.0
		self.lineSpacing = 4.0
		self.build(parent)


	def build(self, parent):
		"""
		Builds view components.
		"""
		self.xMin = self.horizontalMargin
		self.yMin = self.verticalMargin
		spacer = self.lineHeight + self.lineSpacing
		self.frame = Frame(parent, bg=demo_const.BG_COLOUR)
		# name label
		self.nameLabelText = StringVar()
		self.nameLabel = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), text='name:')
		self.nameLabel.place(x=self.xMin, y=self.yMin)
		self.nameDisplay = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), textvariable=self.nameLabelText)
		self.nameDisplay.place(x=self.xMin + self.labelAreaWidth, y=self.yMin)
		# current time
		textSpacer = spacer
		self.currentTimeLabelText = StringVar()
		self.currentTimeLabel = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), text='current time:')
		self.currentTimeLabel.place(x=self.xMin, y=self.yMin + textSpacer)
		self.currentTimeDisplay = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), textvariable=self.currentTimeLabelText)
		self.currentTimeDisplay.place(x=self.xMin + self.labelAreaWidth, y=self.yMin + textSpacer)
		# start time
		textSpacer += spacer
		self.startTimeLabelText = StringVar()
		self.startTimeLabel = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), text='start time:')
		self.startTimeLabel.place(x=self.xMin, y=self.yMin + textSpacer)
		self.startTimeDisplay = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), textvariable=self.startTimeLabelText)
		self.startTimeDisplay.place(x=self.xMin + self.labelAreaWidth, y=self.yMin + textSpacer)
		# play time
		textSpacer += spacer
		self.playTimeLabelText = StringVar()
		self.playTimeLabel = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), text='play time:')
		self.playTimeLabel.place(x=self.xMin, y=self.yMin + textSpacer)
		self.playTimeDisplay = Label(self.frame, bg=demo_const.BG_COLOUR, fg=demo_const.DFT_COLOUR, justify=LEFT, anchor=W, font=("Helvetica", 12), textvariable=self.playTimeLabelText)
		self.playTimeDisplay.place(x=self.xMin + self.labelAreaWidth, y=self.yMin + textSpacer)
		# set initial display values
		self.setNameLabel('')
		self.setCurrentTimeLabel(0.0)
		self.setStartTimeLabel(0.0)
		self.setPlayTimeLabel(0.0)
		
	def setNameLabel(self, text):
		"""
		Sets displayed text for name label.
		"""
		self.nameLabelText.set(text)

	def setCurrentTimeLabel(self, timeFloat):
		"""
		Sets display for current time label.

		@type timeFloat: float
		"""
		self.currentTimeLabelText.set(strftime("%H:%M:%S", gmtime(timeFloat)))

	def setStartTimeLabel(self, timeFloat):
		"""
		Sets displayed text for start time label.

		@type timeFloat: float
		"""
		self.startTimeLabelText.set(strftime("%H:%M:%S", gmtime(timeFloat)))

	def setPlayTimeLabel(self, timeFloat):
		"""
		Sets displayed text for play time label.

		@type timeFloat: float
		"""
		self.playTimeLabelText.set(strftime("%H:%M:%S", gmtime(timeFloat)))

	def startGame(self):
		"""
		Shows game has started.
		"""
		self.startTime = time()
		self.setStartTimeLabel(self.startTime)
		self.setPlayTimeLabel(0.0)
		self.running = True

	def stopGame(self):
		"""
		Shows game has stopped.
		"""
		self.running = False

	def updateDisplay(self, currentTime):
		"""
		Updates displayed information.
		"""
		self.setCurrentTimeLabel(currentTime)
		if self.running:self.setPlayTimeLabel(currentTime - self.startTime)
