# svs_demogame.agents

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Simple AI agents for simulation.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
import math

# internal imports
from svs_demogame.utils import demo_const
from svs_demogame.base_entities import ScriptableEntity
#from svs_core.geometry.geomlib import Point2D
from svs_simulation.numdata.geomlib import Point2D, angleFromAtoB


#############################
# EXCEPTIONS
#############################
class AgentException(Exception):pass

		
#############################
# AGENT
#############################
class Agent(ScriptableEntity):
	"""
	Simple agent with steering capabilities, and able to respond to
	commands embedded in the terrain.
	"""
	def __init__(self, name=None, terrain=None):
		ScriptableEntity.__init__(self)
		self.name = name
		self.terrain = terrain
		self.location = Point2D(0.0, 0.0)
		self.facing = 0.0
		self.velocity = Point2D(0.0, 0.0)
		self.accelleration = 1.0
		self.terrainFriction = 1.0
		self.moving = False
		self.currentTerrainArea = None

	def getName(self):
		"""
		Returns name of agent.
		"""
		return self.name

	def getProfile(self):
		"""
		Returns L{AgentProfile} object for sending over network.
		"""
		return {demo_const.NAME_LABEL:self.name, demo_const.AGENT_ID_LABEL:self.idNum, demo_const.LOC_X_LABEL:self.location.x, demo_const.LOC_Y_LABEL:self.location.y, demo_const.FACING_LABEL:self.facing}

	def stop(self):
		"""
		Changes agent to 'rest' state, stops movement.
		"""
		self.moving = False

	def go(self, newSpeed=None, newFacing=None):
		"""
		Changes agent to 'move' state, starts movement.
		"""
		if newSpeed:self.speed(newSpeed)
		if newFacing:self.setDirection(newFacing)
		self.moving = True

	def setLocation(self, newPosX, newPosY):
		"""
		Sets location of agent, making sure that it is within
		the bounds of the terrain.
		"""
		if self.terrain.containsLocation(newPosX, newPosY):self.location.setLocation(newPosX, newPosY)

	def setDirection(self, newFacing):
		"""
		Sets new facing direction for agent, making sure it is withing 360 degrees.
		"""
		self.facing = newFacing % 360
		self.terrain.reportChange(self.getProfile(), demo_const.AGENTS_LABEL)

	def placeAt(self, newX, newY):
		"""
		Places agent at new location.
		"""
		# check new location is within bounds of terrain
		if not self.terrain.containsLocation(newX, newY):return
		# do collision detection - to do
		# apply new values to agent
		self.location.setLocation(newX, newY)
		self.terrain.reportChange(self.getProfile(), demo_const.AGENTS_LABEL)

	def move(self, deltaX, deltaY):
		"""
		Moves agent by specified distances, making sure that
		it remains within the terrain.
		"""
		self.placeAt(self.location.x + deltaX, self.location.y + deltaY)

	def moveTowards(self, deltaX, deltaY):
		"""
		Moves agent by specified distances and facing in that direction, 
		making sure that it remains within the terrain.
		"""
		newX = self.location.x + deltaX
		newY = self.location.y + deltaY
		self.turnTo(newX, newY)
		self.placeAt(newX, newY)

	def turn(self, angle):
		"""
		Turns agent by specified angle.
		"""
		self.setDirection(self.facing + angle)

	def turnTo(self, x, y):
		"""
		Turns agent to face specified position.
		"""
		angle = angleFromAtoB(self.location.x, self.location.y, x, y)
		self.setDirection(angle)

	def speed(self, newSpeed):
		"""
		Sets new speed for agent.
		"""
		self.accelleration = newSpeed

	def getSurroundingAreas(self):
		"""
		Returns a list of areas surrounding agent.
		"""
		if not self.currentTerrainArea:return None
		return self.currentTerrainArea.getNeighbours('N', 'S', 'E', 'W', markBoundary=True)
			
	def startPlay(self):
		"""
		Called when play starts.
		"""
		self.scriptHandler.loadCurrentScript()
		self.executeStartPlayScript()

	def stopPlay(self):
		"""
		Called when play stops.
		"""
		self.scriptHandler.loadCurrentScript()
		self.executeStopPlayScript()


	def update(self, timeInterval):
		"""
		Updates agent.
		"""
		self.scriptHandler.loadCurrentScript()
		#self.checkObstacles()
		self.updateLocation(timeInterval)
		self.updateTerrainLocation()
		self.executeScriptOnSelf()

	def updateTerrainLocation(self):
		"""
		Checks current location of agent in terrain to determine
		current terrain area of agent.

		This method is called within the main update loop and is
		used to call C{enterArea} and C{exitArea} methods.
		"""
		area = self.terrain.getAreaAtLocation(self.location.x, self.location.y)
		#if not area:raise AgentException("no terrain area for agent location: <%s, %s>" % (self.location.x, self.location.y))
		if not area:return
		# do nothing if in current area
		if area == self.currentTerrainArea:return
		# exit current area
		if self.currentTerrainArea:
			self.currentTerrainArea.agentExited(self)
			self.executeExitAreaScript(self.currentTerrainArea)
		# enter new area
		self.currentTerrainArea = area
		self.currentTerrainArea.agentEntered(self)
		self.terrainFriction = 1.0 - self.currentTerrainArea.density
		self.executeEnterAreaScript(self.currentTerrainArea)
		

	def updateLocation(self, timeInterval):
		"""
		Updates location of agent.
		"""
		if not self.moving or not self.accelleration:return
		theta = math.radians(self.facing)
		self.velocity.x = math.cos(theta) * self.accelleration * self.terrainFriction * timeInterval
		self.velocity.y = math.sin(theta) * self.accelleration * self.terrainFriction * timeInterval
    		newPosX = self.location.x + self.velocity.x
		newPosY = self.location.y + self.velocity.y
		if self.terrain.containsLocation(newPosX, newPosY):
			self.location.setLocation(newPosX, newPosY)
			self.terrain.reportChange(self.getProfile(), demo_const.AGENTS_LABEL)

	def checkObstacles(self):
		"""
		Checks ahead of agent in terrain for obstacles.
		"""
		if not self.moving or not self.currentTerrainArea:return
		compass = int(math.floor(self.facing / 45.0))
		#print "compass:", compass
		front = demo_const.COMPASS_POINTS[compass]
		front_left = demo_const.COMPASS_POINTS[(compass + 1) % 8]
		front_right = demo_const.COMPASS_POINTS[(compass - 1) % 8]
		left = demo_const.COMPASS_POINTS[(compass + 2) % 8]
		right = demo_const.COMPASS_POINTS[(compass - 2) % 8]
		areas = self.currentTerrainArea.getNeighbours(front, front_left, front_right, left, right)
		if compass % 2 == 0:
			if areas.has_key(front) and areas[front].isAccessible():
				#print "ahead ok:", front
				return
			elif areas.has_key(front_left) and areas[front_left].isAccessible():
				self.turn(-45)
				return
			elif areas.has_key(front_right) and areas[front_right].isAccessible():
				self.turn(45)
				return
			elif areas.has_key(left) and areas[left].isAccessible():
				self.turn(-90)
				return
			elif areas.has_key(right) and areas[right].isAccessible():
				self.turn(90)
				return
			else:self.turn(180)
		else:
			newAngle = 0.0

			if areas.has_key(front) and areas[front].isAccessible():
				newAngle = compass
			if areas.has_key(front_left) and areas[front_left].isAccessible():
				newAngle -= 45
			if areas.has_key(front_right) and areas[front_right].isAccessible():
				newAngle += 45
			if math.fabs(newAngle - compass) < 0.1:return
			if areas.has_key(left) and areas[left].isAccessible():
				newAngle -= 45
				self.turn(newAngle)
				return
			if areas.has_key(right) and areas[right].isAccessible():
				newAngle += 45
				self.turn(newAngle)
		



	#############################
	# SCRIPTING
	#############################
	def getScriptIdentifier(self):
		"""
		Returns a string name for the object that can be 
		used by the L{ScriptHandler}.

		Overridden by extending class.
		"""
		return "<agent_%d>" % self.idNum

	def executeStartPlayScript(self):
		"""
		Executes C{startPlay} handler in script attached to agent.
		"""
		self.scriptHandler.executeCurrentScriptMethod('startPlay')

	def executeStopPlayScript(self):
		"""
		Executes C{stopPlay} handler in script attached to agent.
		"""
		self.scriptHandler.executeCurrentScriptMethod('stopPlay')

	def executeEnterAreaScript(self, area):
		"""
		Executes C{enterArea} handler in script attached to agent.
		"""
		self.scriptHandler.executeCurrentScriptMethod('enterArea', area)

	def executeExitAreaScript(self, area):
		"""
		Executes C{exitArea} handler in script attached to agent.
		"""
		self.scriptHandler.executeCurrentScriptMethod('exitArea', area)
		




#############################
# AGENT GROUPS
#############################
class AgentGroupManagerException(Exception):
    pass

class AgentGroupManager:
	"""
	Manages membership of agent groups.

	Agents can join groups which allow them to behave collectively.  
	Each group is defined by a unique name and colour.
	"""
	def __init__(self):
		self.groups = {}
		self.colours = []

	def addAgentGroup(self, groupName, groupColour):
		"""
		Adds new group to manager.

		A group can only be added once, if a group with
		the same name or colour already exists, the new group is rejected.
		"""
		if self.groups.has_key(groupName):raise AgentGroupManagerException("Group <%s> already exists." % groupName)
		if groupColour in self.colours:raise AgentGroupManagerException("Colour <%s> already in use." % groupColour)
		self.groups[groupName] = AgentGroup(groupName, groupColour)
		self.colours.append(groupColour)

	def getGroup(self, groupName):
		"""
		Returns group with specified name, if not
		present retruns None.
		"""
		return self.groups.get(groupName, None)

	def joinAgentGroup(self, agent, groupName):
		"""
		Adds agent to specified group.

		Checks that the agent does not alreday belong to 
		the group, if so ignores request.  If group 
		does not exist raises L{AgentGroupManagerException}.
		"""
		if not self.groups.has_key(groupName): raise AgentGroupManagerException("Group <%s> does not exist." % groupName)
		self.groups[groupName].addAgent(agent)
		
class AgentGroup:
	"""
	Holds a collection of agents who can behave as a single group.
	
	Each group is defined by a unique name and colour.
	"""
	def __init__(self, name, colour):
		self.name = name
		self.colour = colour
		self.agents = []

	def addAgent(self, agent):
		"""
		Adds agent to group making sure that it is not duplicated.
		"""
		if agent not in self.agents:self.agents.append(agent)
	
	
