# svs_core.utilities.objectmanager

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Provides facilities for dynamic object generation and persistence.

Based on the data manager exmaple in Sean Riley, 2004,
'GameProgramming with Python', Charles River Media.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
#############################
# EXCEPTIONS
#############################
class PersistentObjectException(Exception):pass

#############################
# FUNCTIONS
#############################
def importCodeFromFile(codeFile,name,add_to_sys_modules=0):
	"""
	Import dynamically generated code as a module. C{code} is the
	object containing the code (a string, a file handle or an
	actual compiled code object, same types as accepted by an
	exec statement). The C{name} is the name to give to the module,
	and the final argument says whether to add it to sys.modules
	or not. If it is added, a subsequent import statement using
	name will return this module. If it is not added to sys.modules
	import will try to load it in the normal fashion.

	import foo

	is equivalent to

	foofile = open("/path/to/foo.py")
	foo = importCode(foofile,"foo",1)

	Returns a newly generated module.

	Written by Anders Hammarquist, U{http://aspn.activestate.com/ASPN/Cookbook/Python/Recipe/82234}
	"""
	import sys,imp
	module = imp.new_module(name)
	code = open(codeFile)
	exec code in module.__dict__
	if add_to_sys_modules:
		sys.modules[name] = module
	return module

#############################
# CLASSES
#############################
class ObjectCategory:
	"""
	Base class for dynamic objects created from data source.
	"""
	def __init__(self, name):
		self.name = name

	def create(self):
		"""
		Creates new instance of object from data source.

		This should be overridden by extending classes.
		"""
		return None

class PersistentObject:
	"""
	Supports an object that can be stored and retrived externally.
	"""
	def __init__(self):
		pass

	def read(self, filename):
		"""
		Reads object data.
		"""	
		from svs_core.utilities.lib import validateFilePath
		filename = validateFilePath(filename, create=False)
		if not filename:raise PersistentObjectException("File not found: %s" % filename)
		localvars = {}
		execfile(filename, globals(), localvars)
		self.__dict__.update(localvars)

	def write(self, filename):
		"""
		Writes object data to file.
		"""
		from svs_core.utilities.lib import validateFilePath
		filename = validateFilePath(filename)
		pass



class ObjectManager:
	"""
	Provides generators for dynamic classes, created from loaded data.
	"""
	def __init__(self):
		self.categories = {}

	def initCategory(self, label, objectData, categoryClass):
		"""
		Adds factory functions for new category of object.
		"""
		newCatList = []
		for categoryTuple in objectData:
			newCatObj = apply(categoryClass, categoryTuple)
			newCatList.append(newCatObj)
		self.categories[label] = newCatList

	def findCategory(self, name, categoryName):
		"""
		Searches stored object categories for specified category.
		"""
		category = self.categories.get(categoryName, None)
		if not category:return None
		for cat in category:
			if cat.name == name:return cat

	def createInstance(self, name, categoryName, *args):
		"""
		Creates an instance of a specified category object.
		"""
		category = self.findCategory(name, categoryName)
		if category:return apply(category.create, args)


