# svs_core.network.clustermanager

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Provides central management of cluster.

It uses the twisted framework: U{http://twistedmatrix.com}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org

TO DO:

- add multicast for zero-conf and heartbeat
- service browsing functionality
- adverts
- media retrieval (FTP)
- more robust server (Twisted tap)
"""
# external imports
from twisted.cred import portal, checkers
from twisted.spread import pb
from twisted.internet import reactor
from time import time

# internal imports
from svs_core.network.clientavatar import ClientAvatar
from svs_core.network.clustergroups import ClusterGroup
from svs_core.network.packets import makeDataPacket
from svs_core.utilities.constants import svs_const
from svs_core.utilities.notification import Listenable



class ClusterManager:
	"""
	This class provides basic server functionality and client
	authentication.  
	
	For another kind of application class to use the server it must provide a
	L{twisted.cred.portal.IRealm} instance and implement the following methods:
	
		- getClientProxyRealm()
		- update(time)
		- statusMessage(text)
		- errorMessage(text)
	"""
	def __init__(self, port, tickDuration=1000):
		self.port = port
		self.realm = ClusterRealm()
		self.realm.server = self
		self.checker = checkers.InMemoryUsernamePasswordDatabaseDontUse()
		self.clusterPortal = portal.Portal(self.realm, [self.checker])
		self.clusterGroupManager = ClusterGroupManager(self)
		self.isRunning = 0
		self.tickDelay = tickDuration / 1000.0 	
		self.quiet = True
		
		
	def setupServices(self):
		"""
		Activates services.
		
		NOTE: left over from earlier version, does nothing at 
		present but might be useful later.
		"""
		pass
		
	def loadClientAccounts(self, path):
		"""
		Loads client authentication data.
		
		NOTE: currently not implemented.

		@type 	path: string
		@param 	path: location of account data
		"""
		pass
		#loader = ClientAccountLoaderXML(self)
		#loader.read(path)
		
		
	def addClientAuthentication(self, name, password):
		"""
		Create authentication account for client.
		"""
		#print "addClientAuthentication: %s, %s" % (name, password)
		self.checker.addUser(name, password)
		
		
	def run(self):
		"""
		Sets up and starts networking facilities for cluster.
		"""
		self.statusMessage("starting server ...")
		# start networking
		self.isRunning = 1
		self.tickCount = 0
		reactor.callLater(self.tickDelay, self.tick)
		reactor.listenTCP(self.port, pb.PBServerFactory(self.clusterPortal))
		reactor.run()
	
	
	def tick(self):
		"""
		Called by reactor as clock event to update simulation.
		"""
		if self.isRunning:
			# reset timer
			reactor.callLater(self.tickDelay, self.tick)
			# update simulation
			self.tickCount += 1
			self.clusterGroupManager.update(self.tickCount)
	
		
	def stop(self):
		"""
		Stops running server.
		"""
		self.statusMessage("stopping server ...")
		if self.isRunning:
			self.isRunning = 0
			reactor.stop()
		self.statusMessage("server stopped")


	#######################
	# CLIENTS
	#######################
	def joinClusterGroup(self, client):
		"""
		Join client proxy group within world.
		
		@type 	user: L{ClientAvatar <svs.network.clientavatar.ClientAvatar>}
		@param 	user: user to register in world
		"""
		self.statusMessage("Client '%s' is joining cluster group '%s'" % (client.name, client.groupName))
		return self.clusterGroupManager.joinGroup(client)
		
	def leaveClusterGroup(self, client):
		"""
		Remove client from group.
		
		@type 	user: L{ClientAvatar <svs.network.clientavatar.ClientAvatar>}
		@param 	user: user to remove from world
		"""
		self.statusMessage("Client '%s' of type '%s' is leaving world" % (client.name, client.groupName))
		self.clusterGroupManager.leaveGroup(client)
	
	def sendCommand(self, cmd):
		"""
		Send command to recipient.
		
		@type 	cmd: L{Command <svs.commands.scriptcommands.Command>}
		@param 	cmd: command for cleint
		"""
		self.statusMessage("Command '%s' for '%s'" % (cmd.name, cmd.recipient))
		fwdCmd = Command()
		fwdCmd.__dict__ = cmd.__dict__
		self.clusterGroupManager.sendCommand(fwdCmd)

	#######################
	# UTILITY
	#######################
	def statusMessage(self, text):
		"""
		Handles status messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		if not self.quiet:print "STATUS MESSAGE:", text
		
	
	def errorMessage(self, text):
		"""
		Handles error messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		if not self.quiet:print "ERROR MESSAGE:", text


#############################
# EXCEPTIONS
#############################
class ClientRosterException(Exception):
	"""
	Raised when an exception or error occurs in
	handling client proxy groups.
	"""
	pass


#############################
# CLIENT PROXY REALM
#
# used for authentication
# by twisted server
#############################
class ClusterRealm:
	"""
	The ClusterRealm is used by twisted to authenticate clients
	and link them to proxy objects.
	
	The main proxy object used in SVSa is the L{ClientAvatar}.
	
	For more information see authentication in the Perspective Broker system
	within the twisted framework: U{http://twistedmatrix.com/documents/howto/pb-cred}.
	"""
	__implements__ = portal.IRealm
	def requestAvatar(self, avatarID, mind, *interfaces):
		assert pb.IPerspective in interfaces
		avatar = ClientAvatar(avatarID)
		avatar.server = self.server
		avatar.attached(mind)
		return pb.IPerspective, avatar, lambda a=avatar:a.detached(mind)



#############################
# CLIENT PROXY GROUPS
#############################
class ClusterGroupManager(Listenable):
	"""
	This class manages groups of client proxies.
	"""
	def __init__(self, clusterManager):
		Listenable.__init__(self)
		self.clusterManager = clusterManager
		self.groups = {} # indexed by name

	def joinGroup(self, client):
		"""
		Adds client to group.  The group is determined by
		the client's C{clientType} property.
		
		@type 	client: object
		@param 	client: new client to be added
		@rtype:	tuple (string, list)
		@return: name of group and list of members
		"""
		groupName = client.groupName
		if not self.groups.has_key(groupName):
			self.groups[groupName] = ClusterGroup(groupName)
		self.groups[groupName].addClient(client)
		self.sendClientJoinedNotification(client.name, groupName)
		return self.groups[groupName]
		
	def leaveGroup(self, client):
		"""
		Removes client from group. The group is determined by
		the client's C{clientType} property.
		
		@type 	client: object
		@param 	client: old client to be removed
		"""
		groupName = client.groupName
		if not self.groups.has_key(groupName):raise ClientRosterException("Client '%s' attempted to leave non-existant group '%s'." % (client.name, groupName))
		try:self.groups[groupName].removeClient(client)
		except:raise ClientRosterException("Client '%s' not found in group '%s'." % (client.name, groupName))
		self.sendClientDepartedNotification(client.name, groupName)
		
	def update(self, simTime):
		"""
		Forwards C{update} message from simulation cluster to all clients.
		"""
		for groupname, group in self.groups.items():group.update(simTime)

	def sendClientJoinedNotification(self, clientName, groupName):
		"""
		Sends notification that new client has joined a group.
		"""
		data = {svs_const.CLIENT_NAME:clientName, svs_const.CLIENT_GROUP:groupName}
		self.sendData(makeDataPacket(svs_const.SERVER_MESSAGE, content=data, label=svs_const.CLIENT_JOINED))

	def sendClientDepartedNotification(self, clientName, groupName):
		"""
		Sends notification that new client has departed from group.
		"""
		data = {svs_const.CLIENT_NAME:clientName, svs_const.CLIENT_GROUP:groupName}
		self.sendData(makeDataPacket(svs_const.SERVER_MESSAGE, content=data, label=svs_const.CLIENT_DEPARTED))
			
		
	def sendData(self, data):
		"""
		Forwards data to all clients.
		
		@type 	data: object
		@param 	data: data to send
		"""
		for groupname, group in self.groups.items():group.sendData(data)
			
	def sendDataToGroup(self, data, groupName):
		"""
		Forwards data to specified group.
		
		@type 	data: object
		@param 	data: data to send
		@type 	groupName: string
		@param 	groupName: name of group to send data to
		"""
		group = self.groups.get(groupName, None)
		if group:group.sendData(data)

	#############################
	# LISTENERS 
	#############################
	def notifyListeners(self, listenerGroup, dataPacket):
		"""
		Forwards data packet to specified list of clients.
		"""
		for listener in listenerGroup:
			client = self.clients.get(listener, None) 
			if client:client.receiveData(dataPacket)
	

		
