# svs_core.network.clustergroups

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Handles grousp of clients within the cluster.

It uses the twisted framework: U{http://twistedmatrix.com}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from twisted.spread import pb
from time import time

# internal imports
from svs_core.network.clientavatar import ClientAvatar
from svs_core.network.packets import makeDataPacket
from svs_core.commands.scriptcommands import copyCommand, copyCommandResult, makeCommandResult
from svs_core.utilities.constants import svs_const
from svs_core.utilities.notification import Listenable


class ClusterGroup(pb.Viewable, Listenable):
	"""
	This class encapsulates a single group of client proxy
	objects.  It adds and removes proxies from the group
	as well as handling any collective communication to
	group members.
	"""
	
	def __init__(self, groupname):
		Listenable.__init__(self)
		self.name = groupname
		self.clients = {}
		
	def addClient(self, client):
		"""
		Adds client to group.
		
		@type 	client: object
		@param 	client: new client to be added
		"""
		if self.clients.has_key(client.name):return
		self.clients[client.name] = client
		client.clusterGroup = self
		
	def removeClient(self, client):
		"""
		Removes client from group.
		
		@type 	client: object
		@param 	client: old client to be removed
		"""
		if self.clients.has_key(client.name):del self.clients[client.name]

	def view_getClientList(self, sendingClient):
		"""
		Returns a list of clients currently in group.
		"""
		clientList = self.clients.keys()
		clientList.sort()
		return makeDataPacket(self, content=clientList, label=svs_const.GROUP_LIST)
		
	def view_sendChatMessage(self, from_client, msgPacket):
		"""
		Forwards a message from a client to the group.
		
		If a client has registered as a 'process listener' 
		(see L{ClusterGroup.view_handleListenRequestForProcess}) for 
		chat messages, the message is also sent onto them.

		@type 	from_client: object
		@param 	from_client: client sending message
		@type 	message: string
		@param 	message: message to send
		"""
		msgPacket.sender = from_client.name
		if msgPacket.recipient == svs_const.RECIPIENTS_ALL:
			for client in self.clients.values():
				if client == from_client:continue
				client.sendChatMessage(msgPacket)
		else:
			for clientName in msgPacket.recipient:
				client = self.clients.get(clientName, None)
				if client:client.sendChatMessage(msgPacket)
		# forward to process listeners
		dataPacket = makeDataPacket(self, content=msgPacket, label=svs_const.CHAT_MESSAGE) 
		self.notifyListeners(self.listeners.get(svs_const.CHAT_MESSAGE, []), dataPacket)
			

	def sendData(self, dataPacket):
		"""
		Forwards C{dataPacket} to all clients in group.
		"""
		for client in self.clients.values():client.receiveData(dataPacket)

	def update(self, simTime):
		"""
		Forwards C{update} message from simulation cluster to clients in group.
		
		@type 	simTime: list
		@param 	simTime: current simulation time
		"""
		for client in self.clients.values():client.sendUpdate(simTime)
			
	def view_sendCommand(self, sendingClient, cmd):
		"""
		Sends command to client in group.
		"""
		client = self.clients.get(cmd.recipient, None)
		if client:
			fwdCmd = copyCommand(cmd)
			fwdCmd.sender = sendingClient.name
			client.sendCommand(fwdCmd)
			return True
		return False
	
	def view_sendCommandResult(self, sendingClient, cmdResult):
		"""
		Sends command to client in group.
		"""
		client = self.clients.get(cmdResult.cmd.sender, None)
		if client:
			client.sendCommandResult(copyCommandResult(cmdResult))
			return True
		return False


	#######################
	# DATA EXCHANGE
	#######################
	def view_getData(self, sendingClient, dataRequest):
		"""
		Retrieves data from specified source (other client in cluster).
		"""
		client = self.clients.get(dataRequest.recipient, None)
		dataRequest.sender = sendingClient.name
		if client:
			client.handleDataRequest(dataRequest)
			return makeCommandResult(message="asking '%s' for data" % dataRequest.recipient, status=svs_const.OK)
		return makeCommandResult(message="client '%s' not found" % dataRequest.recipient, status=svs_const.ERROR)

	def view_sendData(self, sendingClient, dataPacket):
		"""
		Retrieves data from specified source (other client in cluster).
		"""
		client = self.clients.get(dataPacket.recipient, None)
		dataPacket.sender = sendingClient.name
		if client:client.receiveData(dataPacket)
	
	def sendDataResult(self, cmdResult):
		"""
		Forwards data from avatar to clients in group.
		
		@type 	cmdResult: CommandResult
		@param 	cmdResult: result of retrieval
		"""
		client = self.clients.get(cmdResult.recipient, None)
		if client:client.receiveDataResult(cmdResult)
	
	def view_sendDataResult(self, sendingClient, cmdResult):
		"""
		Forwards data from client to other client in group.
		
		@type 	cmdResult: CommandResult
		@param 	cmdResult: result of retrieval
		"""
		client = self.clients.get(cmdResult.recipient, None)
		if client:client.receiveDataResult(cmdResult)

	#######################
	# PROFILES
	#######################
	def view_getProfileForClient(self, sendingClient, requestedClient):
		"""
		Retrieves profile from specified client.
		"""
		client = self.clients.get(requestedClient, None)
		if client:
			profile = client.getProfile(sendingClient.name)
			return makeCommandResult(result=profile, status=svs_const.OK)
		return makeCommandResult(message="profile for client '%s' not found" % requestedClient, status=svs_const.ERROR)
	
	#######################
	# LISTENER METHODS
	#######################
	def view_handleListenRequest(self, sendingClient, listenRequest):
		"""
		Forwards request from one client to become the listener of another.
		"""
		listenRequest.sender = sendingClient.name
		if not listenRequest.recipient:
			return makeCommandResult(message="recipient '%s' not specified" % listenRequest.recipient, status=svs_const.ERROR)
		client = self.clients.get(listenRequest.recipient, None)
		if client:
			return client.handleListenRequest(listenRequest)
		return makeCommandResult(message="recipient '%s' not found" % listenRequest.recipient, status=svs_const.ERROR)

	def view_handleListenRequestForProcess(self, sendingClient, listenRequest):
		"""
		Forwards request from client to listen for network processes.
		"""
		listenRequest.sender = sendingClient.name
		if not listenRequest.recipient:
			return makeCommandResult(message="recipient '%s' not specified" % listenRequest.recipient, status=svs_const.ERROR)
		return self.handleListenRequest(listenRequest)

	def notifyListeners(self, listenerGroup, dataPacket):
		"""
		Forwards data packet to specified list of clients.
		"""
		if not listenerGroup:return
		for listener in listenerGroup:
			client = self.clients.get(listener, None) 
			if client:
				client.receiveData(dataPacket)

	
