# svs_core.network.clientuser

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
This is the module from which classes and methods implemented by a remote
client are derived.

It uses the twisted framework: U{http://twistedmatrix.com}.  For more information
relating the clients see the documentation on twisted's Perspective Broker system:
U{http://twistedmatrix.com/documents/howto}.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org

TO DO:

- add multicast for zero-conf and heartbeat
- service browsing functionality
- adverts
- media retrieval (P2P, HTTP)
- descriptor classes for services
"""

# external imports
from twisted.spread import pb
from twisted.cred import credentials

# internal imports
from svs_core.gui.utilities import printFormatCommandList, printFormatProfile
from svs_core.commands.scriptcommands import *
from svs_core.network.packets import *
from svs_core.utilities.constants import svs_const




#############################
# MODULE METHODS
#############################
def __remote_callable(obj, methodName):
	"""
	Checks if method for specified object is callable, and then if 
	it conforms to the twisted remote call signature, ie. begins with
	'remote_' or 'perspective_'

	@returns:	string
	"""
	if not callable(getattr(obj, methodName)):return None
	if methodName.startswith("remote_"):return __stripTwistedSignature(methodName)
	elif  methodName.startswith("perspective_"):return __stripTwistedSignature(methodName)
	else:return None

def __stripTwistedSignature(methodName):
	"""
	Removes all characters from the specified method name
	up to, and including, the first '_', returning shortened
	string or None.
	
	@returns:	string
	"""
	parts = methodName.split("_", 1)
	if len(parts) < 2:return None
	return parts[1]


def addRemoteMethodsToProfile(obj):	
	"""
	Checks if methods for specified object can be called remotely,
	and adds them to profile if so.
	"""
	for method in dir(obj):
		rmeth = __remote_callable(obj, method)
		if rmeth:
			obj.profile.remote_call_methods[rmeth] = getattr(obj, method).__doc__


#############################
# PROFILE
#############################
class ClientProfile(pb.Copyable, pb.RemoteCopy):
	"""
	Holds information about a client, used to advertise its
	functionality.
	"""
	def __init__(self):
		self.name = "" 
		self.remote_call_methods = {} 
		self.script_methods = {}
		self.services_provided = {}
		self.services_sought = {}
		self.description = ""

	def setCopyableState(self, state):
		self.__dict__ = state

	def getStateToCopy(self):
		d = self.__dict__.copy()
		return d

# register with perspective broker
pb.setUnjellyableForClass(ClientProfile, ClientProfile)


#############################
# EXCEPTIONS
#############################
class ClientException(Exception):
	pass



#############################
# GENERIC CLIENT
#############################
class GenericClient(pb.Referenceable):
	"""
	GenericClient provides the basic client capabilities for 
	SVS clients.  It is able to connect to a cluster,
	retrieve data and disconnect from a cluster
	but little else.
	
	Specialised client types extend from this class to provide
	more capabilities.

	When talking to server, communicate with avatar.
	When talking to other clients, communicate with clusterGroup.

	Callbacks from message to avatar start with 'avatarResult_'.
	Callbacks from message to clusterGroup start with 'groupResult_'.
	
	@type	clientName: string
	@ivar	clientName: name for client (uses class name as default)
	@type	clientPassword: string
	@ivar	clientPassword: password for client (uses class signature as default)
	@type	avatar: object
	@ivar	avatar: proxy class for cluster supplied by server
	@type	clusterGroupName: string
	@ivar	clusterGroupName: name of cluster group
	@type	worker: object
	@ivar	worker: plugin class for handling specialised processes
	@type	dataCache: dict
	@ivar	dataCache: store for temporary data
	@type	logging: boolean
	@ivar	logging: flag to turn logging output on and off
	@type	profile: L{ClientProfile}
	@ivar	profile: client profile
	"""
	def __init__(self, name, passwd):
		#self.user = None
		self.clientPassword = passwd #tmp!
		self.reactor = None
		self.avatar = None
		self.clusterGroupName = None
		self.worker = None
		self.dataCache = {}
		self.logging = True
		self.profile = ClientProfile()
		self.profile.name = name
		self.profile.description = self.__doc__		
		addRemoteMethodsToProfile(self)

	def getName(self):
		"""
		Returns name of client.
		"""
		return self.profile.name
	
	def setWorker(self, worker):
		"""
		Add delegate class for handling work tasks on client.
		"""
		self.worker = worker
		self.worker.setClient(self)

	#######################
	# CONNECTION
	#######################
	def connect(self, group, host, port):
		"""
		Connect to server.
		
		@type 	group: string
		@param 	group: name of group
		@type 	host: string
		@param 	host: name of server host
		@type 	port: integer
		@param 	port: server port number
		"""
		if self.reactor == None:
			from twisted.internet import reactor
			self.reactor = reactor
		self.clusterGroupName = group
		self.host = host
		self.port = port
		
		self.logMessage("connecting to server %s:%d ..." % (self.host, self.port))
		
		factory = pb.PBClientFactory()
		self.reactor.connectTCP(self.host, self.port, factory)
		d = factory.login(credentials.UsernamePassword(self.profile.name, self.clientPassword), client=self)
		d.addCallback(self.result_connected)
		self.reactor.run()
		

	def result_connected(self, perspective):
		"""
		Called by server when connection made.
		
		This passes a C{perspective} object.  This is a
		reference to the server connection which is used
		for invoking remote calls on the server.
		
		@type 	perspective: object
		@param 	perspective: reference to server connection
		"""
		self.logMessage("connected to server.")
		self.avatar = perspective
		self.joinClusterGroup()
		
		
	def disconnect(self):
		"""
		Disconnect from cluster.
		"""
		self.reactor.disconnectAll()
		self.reactor.stop()
		
	#######################
	# SHUTDOWN
	#######################
	def shutdown(self, result):
		"""
		Called by the server when it shuts down.
		"""
		self.reactor.disconnectAll()
		self.reactor.stop()
 
		
	#######################
	# GROUP CONNECTIONS
	# local methods
	#######################
	def joinClusterGroup(self):
		"""
		Client attempts to join cluster group.  If succesful,
		it will be added to a list of similar client types also
		currently connected.
		
		This list is returned in the callback method
		L{joinClusterResult}.
		"""
		if not self.clusterGroupName:
			msg = "No cluster specified."
			self.errorMessage(msg)
			raise ClientException(msg)
			
		if not self.avatar:
			msg = "No connection to server."
			self.errorMessage(msg)
			raise ClientException(msg)
			
		d = self.avatar.callRemote("joinClusterGroup", self.clusterGroupName)
		d.addCallback(self.avatarResult_joinClusterGroup)
		
	def haveJoinedClusterGroup(self):
		"""
		Called after a client has succesfully joined a
		cluster group.
		
		This does nothing by itself but can be overidden
		by extending clients to handle any actions
		necessary at this point.
		"""
		pass
		
	def leaveCluster(self):
		"""
		Client leaves cluster.  On the server side
		it will be removed from the list of clients it was
		previously grouped with.
		
		This list is returned in the callback method
		L{leaveClusterResult}.
		"""
		if not self.avatar:
			msg = "No connection to server."
			self.errorMessage(msg)
			raise ClientException(msg)
			
		d = self.avatar.callRemote("leaveCluster")
		d.addCallback(self.avatarResult_leaveCluster)

	def getGroupMembers(self):
		"""
		Retrieves a list of all clients currently
		connected to cluster group.
		"""
		if not self.clusterGroup:raise Exception("not part of group")
		d = self.clusterGroup.callRemote("getClientList")
		d.addCallback(self.groupResult_getClientList)

	def handleGroupMemberList(self, groupList):
		"""
		Deals with received list of group members.

		This can be overridden by implementing clients.
		"""
		groupListStr = "group '%s' members:\n" % self.clusterGroupName
		for name in groupList:
			groupListStr += "  %s\n" % name
		self.statusMessage(groupListStr)
		
		
	#######################
	# GROUP CONNECTIONS
	# callbacks
	#######################
	def avatarResult_joinClusterGroup(self, group):
		"""
		Handles result of L{joinClusterGroup} action.
		"""
		self.clusterGroup = group
		self.logMessage("joined cluster group '%s'" % self.clusterGroupName)
		self.haveJoinedClusterGroup()
		
		
	def avatarResult_leaveCluster(self, result):
		"""
		Handles result of L{leaveCluster} action.
		
		@type 	result: dict
		@param 	result: result packet
		"""
		self.logMessage("have departed from cluster.")

	def groupResult_getClientList(self, result):
		"""
		Handles receievd list of clients in group.

		@type 	result: dict
		@param 	result: result packet
		"""
		if not result:
			self.errorMessage("no members for group '%s' found" % self.clusterGroupName)
			return
		self.handleGroupMemberList(result.content)

	#######################
	# PROFILES
	#######################
	def getProfileForClient(self, clientName):
		"""
		Retrieves profile for specified client from server.
		"""
		self.clusterGroup.callRemote("getProfileForClient", clientName).addCallback(self.groupResult_getProfileForClient)

	def groupResult_getProfileForClient(self, commandResult):
		"""
		Receives profile requested from other client.
		"""
		if commandResult.status == svs_const.OK:self.handleReceivedProfile(commandResult.result)
		else:self.errorMessage(commandResult.message)
			
	def handleReceivedProfile(self, profile):
		"""
		Performs actions in response to profile
		receievd from another client.
		
		This should be overidden to provide desired
		functionality for extending clients.
		"""
		self.statusMessage(printFormatProfile(profile))
		

	def searchForServices(self, serviceList):
		"""
		Queries server for clients providing specified services.
		"""
		pass

	def searchForUsers(self, serviceList):
		"""
		Queries server for clients using specified services.
		"""
		pass

	def searchForMethods(self, methodsList):
		"""
		Queries server for clients responding to specified method names.
		"""
		pass

	def searchForCommands(self, commandList):
		"""
		Queries server for clients responding to specified command names.
		"""
		pass

	def searchProfiles(self, profileRequest):
		"""
		Generic handler for searching client profiles.
		"""
		self.clusterGroup.callRemote("searchProfiles", profileRequest).addCallback(self.groupResult_searchProfiles)

	def advertise(self, advertPacket):
		"""
		Places advert for services on server.
		"""
		pass
		
	def remote_profileRequested(self, requester):
		"""
		Receives name of other client that has requested 
		this client's profile.
		"""
		self.handleProfileRequest(requester)
		
		
	def handleProfileRequest(self, requester):
		"""
		Responds notification of request for profile.
		"""
		self.statusMessage("client '%s' has requested profile." % requester)
		

	#######################
	# HTTP
	#######################
	def sendHttpRequest(self, url, method=None):
		"""
		Sends HTTP request to specified URL.
		"""
		pass
		

	#######################
	# LOCAL DATA CACHE
	#######################
	def storeLocalData(self, dataPacket):
		"""
		Stores local data in cache.
		
		Data is stored as DataPacket.
		"""
		dataPacket.sender = self.profile.name
		if dataPacket.label:
			self.dataCache[dataPacket.label] = dataPacket
		else:
			self.dataCache[svs_const.CURRENT_DATA] = dataPacket
		print self.dataCache
	
	def getLocalData(self, label=None):
		"""
		Retrieves local data from cache.
		"""
		if not label:
			return self.dataCache.get(svs_const.CURRENT_DATA, None)
		else:
			return self.dataCache.get(label, None)
	
	def clearLocalData(self, label=None):
		"""
		Empties cache of data deposits held by client.
	
		If a label is specified then only data stored under that
		label is removed.
		"""
		if label:
			self.dataCache[label] = None
		else:
			self.dataCache = {}

	#######################
	# DATA EXCHANGE
	# sending and receiving
	# data with other clients
	#######################
	def handleDataPacket(self, dataPacket):
		"""
		Handles data packet received from network.
		""" 
		self.logMessage("data packet received: '%s'" % dataPacket.content)
		
	def remote_handleDataRequest(self, dataRequest):
		"""
		Responds to request for data from another client.
		"""
		self.logMessage("data request received from: '%s'" % dataRequest.sender)
		dataPacket = self.getDataForLabel(dataRequest)
		if dataPacket:
			result = makeCommandResult(status=svs_const.OK, result=dataPacket, recipient=dataRequest.sender)
		else:
			result = makeCommandResult(status=svs_const.ERROR, message="data not found", recipient=dataRequest.sender)
		self.sendDataResult(result)

	def getDataForLabel(self, dataRequest):
		"""
		Allows speacialised handling of labelled data.

		This can be overridden by extending classes to provide
		custom methods for providing requested data.
		"""
		if dataRequest.label:return self.dataCache.get(dataRequest.label, None)
		else:return self.dataCache.get(svs_const.CURRENT_DATA, None)

	def getData(self, dataRequest):
		"""
		Retrieves data from specified source on cluster,
		and performs 'callback' on receipt.
		
		This method may often be used by delegate worker classes.

		Uses L{groupResult_getData} as callback.
		"""
		if not self.clusterGroup:
			msg = "No cluster specified."
			self.errorMessage(msg)
			raise ClientException(msg)
			
		d = self.clusterGroup.callRemote("getData", dataRequest)
		d.addCallback(self.groupResult_getData)


	def groupResult_getData(self, result):
		"""
		Receives result of L{getData} from avatar.
		"""
		if result.status is svs_const.ERROR:
			self.errorMessage(result.message)
		else:
			self.statusMessage(result.message)
		
	
	def remote_receiveData(self, dataPacket):
		"""
		Handles data sent from client proxy on server.
		
		@type 	dataPacket: L{DataPacket}
		@param 	dataPacket: data from simulation
		"""
		self.handleDataPacket(dataPacket)

	def remote_receiveDataResult(self, cmdResult):
		"""
		Handles data sent, inside makeCommandResult, from client proxy on server.
		
		@type 	cmdResult: L{makeCommandResult}
		@param 	cmdResult: data result from other client
		"""
		if cmdResult.result:
			self.handleDataPacket(cmdResult.result)
		elif cmdResult.status is svs_const.ERROR:
			self.errorMessage(cmdResult.message)
		else:
			self.errorMessage("unknown error retrieveing data")
	
	def sendData(self, dataPacket):
		"""
		Sends data to another client.
		
		Uses L{groupResult_sendData} as callback.
		"""
		d = self.clusterGroup.callRemote("sendData", dataPacket)
		d.addCallback(self.groupResult_sendData)

	def groupResult_sendData(self, result):
		"""
		Receives result of L{sendData} from avatar.
		"""
		self.logMessage("__data sent")

	def sendDataResult(self, dataResult):
		"""
		Sends data to another client.
		
		Uses L{groupResult_sendData} as callback.
		"""
		d = self.clusterGroup.callRemote("sendDataResult", dataResult)
		d.addCallback(self.groupResult_sendDataResult)
	
	def groupResult_sendDataResult(self, result):
		"""
		Receives result of L{sendDataResult} from avatar.
		"""
		self.logMessage("data result sent")

	#######################
	# DATA DEPOSITS
	# leaving data on server
	# for collection by
	# other clients
	#######################
	def depositData(self, dataPacket):
		"""
		Sends data packet to avatar on server where it is cached
		for other clients to collect.
		"""
		if not self.avatar:
			msg = "No connection to server."
			self.errorMessage(msg)
			raise ClientException(msg)
			
		d = self.avatar.callRemote("depositData", dataPacket)
		d.addCallback(self.avatarResult_depositData)

	def avatarResult_depositData(self, result):
		"""
		Receives result of L{depositData} from avatar.
		"""
		self.logMessage("data deposited")
		
	def clearDepositedData(self, label=None):
		"""
		Empties cache of data deposits held by avatar.

		Uses L{avatarResult_clearDepositedData} as callback.
		"""
		if not self.avatar:
			msg = "No connection to server."
			self.errorMessage(msg)
			raise ClientException(msg)
			
		d = self.avatar.callRemote("clearDepositedData", label)
		d.addCallback(self.avatarResult_clearDepositedData)
	
	def avatarResult_clearDepositedData(self, result):
		"""
		Received result of L{clearDepositedData} from avatar.
		"""
		self.logMessage("deposits cleared")
	
	def retrieveOwnData(self, dataRequest):
		"""
		Collects previously deposited data packet from avatar.
		Uses L{avatarResult_retrieveOwnData} as callback.
		"""
		if not self.avatar:
			msg = "No connection to server."
			self.errorMessage(msg)
			raise ClientException(msg)
			
		d = self.avatar.callRemote("retrieveOwnData", dataRequest)
		d.addCallback(self.avatarResult_retrieveOwnData)
	
	def avatarResult_retrieveOwnData(self, result):
		"""
		Received result of L{retrieveOwnData} from avatar.
		"""
		dataPacket = result.result
		if dataPacket:
			self.statusMessage("deposit collected: '%s'" % dataPacket.content)
		elif result.status is svs_const.ERROR:
			self.errorMessage(result.message)
		else:
			self.errorMessage("Unknown error retrieving data from deposit")

	#######################
	# LISTENER METHODS
	#######################
	def startListeningTo(self, sourceClient, listenFor=None):
		"""
		Register as listener with another client.
		"""
		listenRequest = makeListenRequest(self, recipient=sourceClient, label=listenFor, action=svs_const.START_LISTEN)
		d = self.clusterGroup.callRemote("handleListenRequest", listenRequest)
		d.addCallback(self.groupResult_listenTo)
	
	def stopListeningTo(self, sourceClient, listenFor=None):
		"""
		De-register as listener with another client.
		"""
		listenRequest = makeListenRequest(self, recipient=sourceClient, label=listenFor, action=svs_const.STOP_LISTEN)
		d = self.clusterGroup.callRemote("handleListenRequest", listenRequest)
		d.addCallback(self.groupResult_stopListeningTo)

	def groupResult_listenTo(self, result):
		"""
		Callback for L{listenTo} method.
		"""
		self.logMessage("<groupResult_listenTo>") 
	
	
	def groupResult_stopListeningTo(self, result):
		"""
		Callback for L{stopListeningTo} method.
		"""
		self.logMessage("<groupResult_stopListeningTo>")
	
	def notifyListeners(self, dataPacket):
		"""
		Forward data to listeners.

		Uses L{avatarResult_notifyListeners} as callback.
		"""
		d = self.avatar.callRemote("notifyListeners", dataPacket)
		d.addCallback(self.avatarResult_notifyListeners)
	
	def avatarResult_notifyListeners(self, result):
		"""
		Callback for L{notifyListeners} method.
		"""
		pass
		#self.logMessage("notification sent")
		
	def remote_notify(self, notification):
		"""
		Receives notification packets from other
		clients to which this has registered as
		listener.
		"""
		self.logMessage("<remote_notify>")

	#######################
	# PROCESS LISTENER METHODS
	#
	# provide listeners for
	# network proceses
	#######################
	def startListeningToProcess(self, listenFor):
		"""
		Register as listener to network processes.
		"""
		listenRequest = makeListenRequest(self, recipient=self.clusterGroup, label=listenFor, action=svs_const.START_LISTEN)
		d = self.clusterGroup.callRemote("handleListenRequestForProcess", listenRequest)
		d.addCallback(self.groupResult_startListeningToProcess)
	
	def stopListeningToProcess(self, listenFor):
		"""
		De-register as listener to network processes.
		"""
		listenRequest = makeListenRequest(self, recipient=self.clusterGroup, label=listenFor, action=svs_const.STOP_LISTEN)
		d = self.clusterGroup.callRemote("handleListenRequestForProcess", listenRequest)
		d.addCallback(self.groupResult_stopListeningToProcess)

	def groupResult_startListeningToProcess(self, result):
		"""
		Callback for L{listenTo} method.
		"""
		self.logMessage("<groupResult_startListeningToProcess>") 
	
	
	def groupResult_stopListeningToProcess(self, result):
		"""
		Callback for L{stopListeningTo} method.
		"""
		self.logMessage("<groupResult_stopListeningToProcess>")

	#######################
	# CHAT METHODS
	#######################
	def sendChatMessage(self, msgPacket):
		"""
		Sends chat message to other clients.
		"""
		d = self.clusterGroup.callRemote("sendChatMessage", msgPacket)
		d.addCallback(self.groupResult_sendChatMessage)
	
	def groupResult_sendChatMessage(self, result):
		"""
		Callback for L{sendChatMessage} method.
		"""
		self.logMessage("message sent")

	def remote_receiveChatMessage(self, msgPacket):
		"""
		Receives chat message from other clients.
		"""
		msgText = "message from '%s':\n%s\n" % (msgPacket.sender, msgPacket.content)
		self.statusMessage(msgText)
		
	#######################
	# REMOTE METHODS
	#
	# these are methods
	# called from the server
	#######################
	def remote_getProfile(self):
		"""
		Retrieves profile for cleint
		"""
		return self.profile

	def remote_statusMessage(self, message):
		"""
		Handles status message sent from client proxy on server.
		
		@type 	message: string
		@param 	message: message from proxy
		"""
		self.statusMessage(message)
		
		
	def remote_errorMessage(self, message):
		"""
		Handles error message sent from client proxy on server.
		
		@type 	message: string
		@param 	message: message from proxy
		"""
		self.errorMessage(message)
		
		
	def remote_update(self, time):
		"""
		Handles update sent from client proxy on server.
		
		@type 	time: list
		@param 	time: current simulation time
		"""
		#self.user.update(time)
		#print "update:", time
		#self.statusMessage(time)
		pass 
	
	
	#######################
	# UTILITY
	#######################
	def statusMessage(self, text):
		"""
		Handles status messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		print "STATUS MESSAGE:", text

	def logMessage(self, text):
		"""
		Handles log messages for client. Log messages can be turned on and off.

		This should be overridden by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		if self.logging: print "LOG MESSAGE:", text
		
	
	def errorMessage(self, text):
		"""
		Handles error messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		print "ERROR MESSAGE:", text
		

#############################
# SCRIPTABLE CLIENT
#############################
class ScriptableClient(GenericClient):
	"""
	Scriptable SVS client.
	"""
	def __init__(self, name, passwd):
		GenericClient.__init__(self, name, passwd)
		self.parser = CommandParser()
		self.commandHandler = CommandHandler(self)
		self.commandHandler.createCommandList()
		self.profile.script_methods = self.commandHandler.getPublicCommandDoc()

	def setWorker(self, worker):
		"""
		Add delegate class for handling work tasks on client.
		"""
		self.worker = worker
		self.worker.setClient(self)
		self.commandHandler.createCommandList()
		self.profile.script_methods = self.commandHandler.getPublicCommandDoc()

	def execute(self, commandText, sender=None):
		"""
		Executes script command.
		"""
		if not sender:
			sender = self
		cmd = self.parser.parse(commandText, sender)
		if not cmd:
			cmd = Command()
			cmd.sender = sender
			return makeCommandResult(cmd, "invalid command: %s" % commandText, svs_const.ERROR)
		if cmd.recipient:
			self.sendCommand(cmd)
			return makeCommandResult(cmd, "command sent to '%s'" % cmd.recipient, svs_const.OK)
		return self.commandHandler.handleCommand(cmd)
		
	
	def returnCommandResult(self, cmdResult):
		"""
		Returns result of script command.
		"""
		if cmdResult.cmd.sender is self:
			if cmdResult.status == svs_const.OK:
				if cmdResult.message:
					self.statusMessage(cmdResult.message)
				if cmdResult.cmd.callback:
					cmdResult.cmd.callback(cmdResult.result)
			else:
				self.errorMessage(cmdResult.message)
		else:
			self.sendCommandResult(cmdResult)

	def handleLocalCommand(self, inputText):
		"""
		Handles command script from own interface.
		"""
		self.returnCommandResult(self.execute(inputText, self))
	
	def displayCommandList(self, listData):
		"""
		Displays command list. Can be overridden bu GraphicalClients.
		"""
		self.statusMessage(printFormatCommandList(listData))
	
	def sendCommand(self, cmd):
		"""
		Sends command to other network client.
		"""
		self.logMessage("sending command to '%s'" % cmd.recipient)
		d = self.clusterGroup.callRemote("sendCommand", cmd)
		d.addCallback(self.commandSent)

	def sendCommandResult(self, cmdResult):
		"""
		Sends command to other network client.
		"""
		self.logMessage("sending result to '%s'" % cmdResult.cmd.sender)
		d = self.clusterGroup.callRemote("sendCommandResult", cmdResult)
		d.addCallback(self.commandResultSent)

	#########################
	# REMOTE COMMANDS
	#########################
	def remote_receiveCommand(self, cmd):
		"""
		Handles command sent from client proxy on server.
		
		@type 	cmd: object
		@param 	cmd: command
		"""
		#print "cmd:", cmd
		self.returnCommandResult(self.commandHandler.handleCommand(cmd))
	
	def remote_receiveCommandResult(self, cmdResult):
		"""
		Handles command result sent from client proxy on server.
		
		@type 	cmdResult: object
		@param 	cmdResult: command result
		"""
		self.logMessage("result received from'%s'" % cmdResult.cmd.recipient)

	
	#########################
	# CALLBACKS
	#########################
	def commandSent(self, result):
		"""
		Handles result of L{sendCommand} action.
		
		@type 	result: dict
		@param 	result: result packet
		"""
		self.logMessage("acknowledge command sent")
	
	def commandResultSent(self, result):
		"""
		Handles result of L{sendCommandResult} action.
		
		@type 	result: dict
		@param 	result: result packet
		"""
		self.logMessage("acknowledge command result sent")


	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_disconnect(self, cmd):
		"""
		Disconnect from server.
		"""
		self.disconnect()
		return makeCommandResult(cmd, message="disconnected", status=svs_const.OK)

	def cmdprivate_quit(self, cmd):
		"""
		Disconnect from server and quit client application.
		"""
		self.disconnect()
		return makeCommandResult(cmd, message="quitting", status=svs_const.OK)

	def cmdprivate_logging(self, cmd):
		"""
		Turns log messages on and off.
		"""
		try:
			state = cmd.args[0]
		except IndexError:
			state = 'on'
		if state == 'off':
			self.logging = False
		else:
			self.logging =  True
		return makeCommandResult(cmd, status=svs_const.OK) 
	
	def cmdprivate_deposit(self, cmd):
		"""
		Deposit data with server-side proxy.
		"""
		try:
			content = cmd.args[1]
		except IndexError:
			return makeCommandResult(cmd, message="No data to deposit", status=svs_const.ERROR)
		try:
			label = cmd.args[0]
		except IndexError:
			label = None
	
		try:
			timeToLive = cmd.args[2]
		except IndexError:
			timeToLive = None

		dataPacket = makeDataPacket(self, content=content, label=label, timeToLive=timeToLive)
		self.depositData(dataPacket)
		return makeCommandResult(cmd, message="depositing...", status=svs_const.OK)

	def cmdprivate_retrieve(self, cmd):
		"""
		Collect data from server-side proxy.
		"""
		try:
			label = cmd.args[0]
		except IndexError:
			label = None
	
		dataRequest = makeDataRequest(self, label=label)
		self.retrieveOwnData(dataRequest)
		return makeCommandResult(cmd, message="retrieving...", status=svs_const.OK)

	def cmdprivate_cleardeposit(self, cmd):
		"""
		Delete data held by server-side proxy.
		"""
		try:
			label = cmd.args[0]
		except IndexError:
			label = None
	
		self.clearDepositedData(label)
		return makeCommandResult(cmd, message="clearing deposit...", status=svs_const.OK)


	def cmdprivate_getfrom(self, cmd):
		"""
		Collect data from another client.
		"""
		try:
			clientName = cmd.args[0]
		except IndexError:
			return makeCommandResult(cmd, message="get <client> <label>", status=svs_const.ERROR)

		try:
			label = cmd.args[1]
		except IndexError:
			label = None
	
		dataRequest = makeDataRequest(self, recipient=clientName, label=label)
		self.getData(dataRequest)
		return makeCommandResult(cmd, message="getting data ...", status=svs_const.OK)

	def cmdprivate_put(self, cmd):
		"""
		Store data in client-side cache.
		"""
		try:
			data = cmd.args[1]
		except IndexError:
			return makeCommandResult(cmd, message="no data defined", status=svs_const.ERROR)
		
		try:
			label = cmd.args[0]
		except IndexError:
			label = None
		
		dataPacket = makeDataPacket(self, content=data, label=label)
		self.storeLocalData(dataPacket)
		return makeCommandResult(cmd, message="data stored", status=svs_const.OK)
	
	def cmdprivate_get(self, cmd):
		"""
		Get data from client-side cache.
		"""
		try:
			label = cmd.args[0]
		except IndexError:
			label = None
		
		dataPacket = self.getLocalData(label)
		if dataPacket:
			return makeCommandResult(cmd, message="data: '%s'" % dataPacket.content, result=dataPacket, status=svs_const.OK)
		return makeCommandResult(cmd, message="no data found", status=svs_const.ERROR)
 	
	def cmdprivate_clear(self, cmd):
		"""
		Delete data from client-side cache.
		"""
		try:
			label = cmd.args[0]
		except IndexError:
			label = None
		
		self.clearLocalData(label)
		if label:
			return makeCommandResult(cmd, message="data cleared from '%s'" % label, status=svs_const.OK)
		return makeCommandResult(cmd, message="data cleared", status=svs_const.OK)


	def cmdprivate_listen(self, cmd):
		"""
		Start/stop listening to another client.
		"""
		try:
			clientName = cmd.args[0]
		except IndexError:
			return makeCommandResult(cmd, message="get <client> <label>", status=svs_const.ERROR)

		try:
			action = cmd.args[1]
		except IndexError:
			action = "on"
	
		if action is "on":
			self.startListeningTo(clientName)
			return makeCommandResult(cmd, message="registering as listener ...", status=svs_const.OK)
		else:
			self.stopListeningTo(clientName)
			return makeCommandResult(cmd, message="de-registering as listener ...", status=svs_const.OK)

	def cmdprivate_notify(self, cmd):
		"""
		Send information to listeners.
		"""
		try:
			data = cmd.args[0]
		except IndexError:
			return makeCommandResult(cmd, message="no data defined", status=svs_const.ERROR)
		
		try:
			label = cmd.args[1]
		except IndexError:
			label = None

		dataPacket = makeDataPacket(self, content=data, label=label) 
		self.notifyListeners(dataPacket)
		return makeCommandResult(cmd, message="notification sent...", status=svs_const.OK)

	def cmdprivate_tell(self, cmd):
		"""
		Sends plain text message to other clients, used for chat.
		"""
		try:
			msgPacket = makeMessagePacket(self, cmd.args)
		except DataPacketException:return makeCommandResult(cmd, message="tell <recipient(,recipient2, recipient3)> <message>", status=svs_const.ERROR)
		self.sendChatMessage(msgPacket)
		return makeCommandResult(cmd, status=svs_const.OK)

	def cmdprivate_grouplist(self, cmd):
		"""
		Gets list of clients in cluster group.
		"""
		try:self.getGroupMembers()
		except:return makeCommandResult(cmd, message="Not part of group", status=svs_const.ERROR)
		return makeCommandResult(cmd, status=svs_const.OK)

	def cmdprivate_profile(self, cmd):
		"""
		Gets profile for specified client.
		"""
		try:otherClient = cmd.args[0]
		except:return makeCommandResult(cmd, message="profile <client_name>", status=svs_const.ERROR)
		self.getProfileForClient(otherClient)
		return makeCommandResult(cmd, status=svs_const.OK)


	#########################
	# PUBLIC SCRIPT COMMANDS
	#########################
	def cmdpublic_commands(self, cmd):
		"""
		Get list of script commands.
		"""
		if cmd.sender is self:
			cmd.callback = self.displayCommandList
			return makeCommandResult(cmd, 
				status=svs_const.OK,
				result=self.commandHandler.getAllCommandDoc())
		return makeCommandResult(cmd, 
			status=svs_const.OK,
			result=self.commandHandler.getPublicCommandDoc())
		




#############################
# GUI CLIENT
#############################
class GraphicalClient(ScriptableClient):
	"""
	SVS client with graphical interface.
	"""
	def __init__(self, name, passwd, guiClass=None):
		from svs_core.gui.clientgui import ClientGUI
		ScriptableClient.__init__(self, name, passwd)
		if not guiClass: guiClass = ClientGUI
		self.gui = guiClass(self)
		self.gui.build()
		
	
	#######################
	# UTILITY
	#######################
	def statusMessage(self, text):
		"""
		Handles status messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		self.gui.statusMessage(text)

	def logMessage(self, text):
		"""
		Handles log messages for client. Log messages can be turned on and off.

		This should be overridden by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		if self.logging: self.gui.statusMessage(text)
		
	
	def errorMessage(self, text):
		"""
		Handles error messages for client.  This should be overridden
		by implementing classes.
		
		@type 	text: string
		@param 	text: message
		"""
		self.gui.errorMessage(text)
	
	#########################
	# REMOTE METHODS
	#########################
	def remote_update(self, time):
		"""
		Handles update sent from client proxy on server.
		
		@type 	time: list
		@param 	time: current simulation time
		"""
		self.gui.update(time)
		
	
	#########################
	# PRIVATE SCRIPT COMMANDS
	#########################
	def cmdprivate_quit(self, cmd):
		"""
		Disconnect from server and quit client application.
		"""
		self.gui.destroy()
		return makeCommandResult(cmd, message="quitting", status=svs_const.OK)

	def cmdprivate_clearlog(self, cmd):
		"""
		Clear text from output console.
		"""
		self.gui.clearLogDisplay()
		return makeCommandResult(cmd, status=svs_const.OK)
	
	def cmdprivate_openview(self, cmd):
		"""
		Open visualisation in window.
		"""
		self.gui.openView(False)
		return makeCommandResult(cmd, status=svs_const.OK)

	def cmdprivate_hideview(self, cmd):
		"""
		Hide visualisation in window.
		"""
		self.gui.hideView()
		return makeCommandResult(cmd, status=svs_const.OK)
	
	def cmdprivate_fullscreen(self, cmd):
		"""
		Open fullscreen visualisation.
		"""
		self.gui.openView(True)
		return makeCommandResult(cmd, status=svs_const.OK)
	
	def cmdprivate_echo(self, cmd):
		"""
		Turns echo of input in console on and off.
		"""
		try:
			state = cmd.args[0]
		except IndexError:
			state = 'on'
		if state == 'off':
			self.gui.setEchoInput(False)
		else:
			self.gui.setEchoInput(True)
		return makeCommandResult(cmd, status=svs_const.OK)


	
	
