# svs_core.network.client

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Module for generic client object and related utilities.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
from twisted.spread import pb

# internal imports
from svs_core.network.clientuser import ClientProfile
from svs_core.network.packets import *
from svs_core.commands.scriptcommands import Command, makeCommandResult
from svs_core.utilities.constants import svs_const
from svs_core.utilities.notification import Listenable
# tmp
from svs_demogame.scripts import Script # for network exchange, this needs to become dynamic


#############################
# CLASSES
#############################
class ClientAvatar(pb.Avatar, Listenable):
	"""
	Generic client, provides base client functionality
	as represented to the server.

	Clients interact with the simulation world by calling
	remote methods on the proxy.
	
	For more information see the Perspective Broker system
	within the twisted framework: 
	U{http://twistedmatrix.com/documents/howto/}.
	"""
	def __init__(self, name):
		Listenable.__init__(self)
		self.name = name
		self.clusterGroup = None
		self.dataCache = {}
		self.remote = None
	
		
	#############################
	# CONNECTION 
	#############################
	def attached(self, mind):
		"""
		Called by the L{ClusterRealm} when a client connects to 
		the server. It passes a C{mind} object to the proxy.  This is 
		a reference to the client connection and is used to pass
		remote method calls back to the client from the server.
		
		@type 	mind: object
		@param 	mind: reference to client connection
		"""
		if self.remote != None:return # reject multiple clients
		self.remote = mind
		self.remote.callRemote("getProfile").addCallback(self.profileReceived)
		#print "ASKING FOR PROFILE"
		
	def detached(self, mind):
		"""
		Called by the L{ClusterRealm} when a client disconnects from 
		the server. It passes a C{mind} object to the proxy.  This is 
		a reference to the client connection and is used to pass
		remote method calls back to the client from the server.
		
		The client is also removed from the group it joined earlier.
		
		@type 	mind: object
		@param 	mind: reference to client connection
		"""
		self.remote = None
		# leave group
		if self.groupName:
			self.server.leaveClusterGroup(self)
		self.server.statusMessage("Client '%s' disconnected from server" % self.name)
	
	#############################
	# PROFILES 
	#############################
	def profileReceived(self, profile):
		"""
		Deferred callback for receiving profile from real client.
		"""
		if not profile:
			self.server.errorMessage("Unable to retrieve profile")
		else:
			self.profile = profile
			self.server.statusMessage("Client '%s' connected to server" % self.profile.name)

	def perspective_getProfile(self):
		return self.profile

	def getProfile(self, requester=None):
		"""
		Returns profile.  If requester given notifies remote self with
		name of requester.
		"""
		if requester:self.remote.callRemote("profileRequested", requester)
		return self.profile


	#############################
	# COMMUNICATION 
	#
	# client-to-server
	#############################
	def perspective_joinClusterGroup(self, groupName):
		"""
		Proxy invocation of 
		L{svs.network.clientuser.GenericClient.joinClusterGroup} called by client.
		
		@type 	groupName: string
		@param 	groupName: name of cluster group to join
		@rtype:	object
		"""
		self.groupName = groupName
		return self.server.joinClusterGroup(self)
		
	def perspective_leaveClusterGroup(self):
		"""
		Proxy invocation of 
		L{svs.network.clientuser.GenericClient.leaveCluster} called by client.
		
		@rtype:	object
		@return: result of C{leaveCluster} call to server
		"""
		return self.server.leaveCluster(self)
		
	def perspective_sendCommand(self, cmd):
		"""
		Sends commands to another client.
		"""
		#print "perspective_sendCommand from '%s' to '%s'" % (self.name, cmd.recipient)
		return self.server.sendCommand(cmd)
	
	def perspective_sendCommandResult(self, cmdResult):
		"""
		Sends command result to sending client.
		
		Note: not implemented.
		"""
		pass
		#print "perspective_sendCommandResult:", cmdResult
		#return self.server.sendCommand(cmd)
	
	
	#############################
	# COMMUNICATION 
	#
	# server-to-client
	#############################
	def sendStatusMessage(self, message):
		"""
		Sends status message to client.
		
		@type 	message: string
		@param 	message: status message
		"""
		self.remote.callRemote("statusMessage", message)
		
		
	def sendErrorMessage(self, message):
		"""
		Sends error message to client.
		
		@type 	message: string
		@param 	message: error message
		"""
		self.remote.callRemote("errorMessage", message)
		
		
	def sendUpdate(self, simTime):
		"""
		Calls update method on client.
		
		@type 	simTime: list
		@param 	simTime: current simulation time
		"""
		if self.remote:self.remote.callRemote("update", simTime)

	#############################
	# CHAT MESSAGES 
	#############################
	def sendChatMessage(self, message):
		"""
		Sends char message to client.
		
		@type 	message: string
		@param 	message: status message
		"""
		self.remote.callRemote("receiveChatMessage", message)
		
	#############################
	# COMMANDS 
	#############################
	def sendCommand(self, cmd):
		"""
		Sends command to client.
		
		@type 	data: list
		@param 	data: data to send
		"""
		#print "'%s' avatar.sendCommand from '%s' to '%s'" % (self.name, cmd.sender, cmd.recipient)
		self.remote.callRemote("receiveCommand", cmd)
	
	def sendCommandResult(self, cmdResult):
		"""
		Sends comamnd result to client.
		
		@type 	data: list
		@param 	data: data to send
		"""
		#print "'%s' avatar.sendCommandResult" % self.name
		self.remote.callRemote("receiveCommandResult", cmdResult)
	
	#############################
	# LISTENERS 
	#############################
	def perspective_notifyListeners(self, dataPacket):
		"""
		Forwards data onto listeners.
		"""
		dataPacket.sender = self.name
		listenFor = dataPacket.label
		if not listenFor:listenFor='_all_'
		if not self.listeners.has_key(listenFor):return
		if self.clusterGroup:self.clusterGroup.notifyListeners(self.listeners[listenFor], dataPacket)

	#######################
	# DATA EXCHANGE
	#######################
	def handleDataRequest(self, dataRequest):
		"""
		Checks if requested data is in cache and returns that.  If not,
		then forwards request to client.
		"""
		if dataRequest.label:dataPacket = self.dataCache.get(dataRequest.label, None)
		else:dataPacket = self.dataCache.get(svs_const.CURRENT_DATA, None)
		if dataPacket:self.sendDataResult(makeCommandResult(status=svs_const.OK, result=dataPacket, recipient=dataRequest.sender))
		else:self.remote.callRemote("handleDataRequest", dataRequest)
	
	def receiveData(self, dataPacket):
		"""
		Receive data from another client, forward to own client.
		"""
		self.remote.callRemote("receiveData", dataPacket)

	def receiveDataResult(self, cmdResult):
		"""
		Receive data from another client, forward to own client.

		The data is wrapped inside a makeCommandResult object.
		"""
		self.remote.callRemote("receiveDataResult", cmdResult)
	
	def sendData(self, dataPacket):
		"""
		Sends data packet to cluster group for forwarding to
		other client.
		"""
		if self.clusterGroup:self.clusterGroup.sendData(dataPacket)

	def sendDataResult(self, cmdResult):
		"""
		Sends data packet to cluster group for forwarding to
		other client.
		"""
		if self.clusterGroup:self.clusterGroup.sendDataResult(cmdResult)
		
	
	#######################
	# DATA DEPOSITS
	# leaving data on server
	# for collection by
	# other clients
	#######################
	def perspective_depositData(self, dataPacket):
		"""
		Stores data packet in cache for other clients to collect.

		If the data packet is labelled it is stored under this label, 
		if not it replaces the current unlabelled data.
		"""
		dataPacket.sender = self.name
		if dataPacket.label:self.dataCache[dataPacket.label] = dataPacket
		else:self.dataCache[svs_const.CURRENT_DATA] = dataPacket
		return makeCommandResult(status=svs_const.OK)


	def perspective_clearDepositedData(self, label=None):
		"""
		Empties cache of data deposits held by avatar.
	
		If a label is specified then only data stored under that
		label is removed.
		"""
		if label:self.dataCache[label] = None
		else:self.dataCache = {}
		return makeCommandResult(status=svs_const.OK)

	
	def perspective_retrieveOwnData(self, dataRequest):
		"""
		Returns requested in cache data back to client.
	
		This can be used by the client as a temporary store 
		for its own information.
		"""
		label = dataRequest.label
		if not label:
			data = self.dataCache.get(svs_const.CURRENT_DATA, None)
			label = "current data"
		else:data = self.dataCache.get(label, None)
		if data:return makeCommandResult(status=svs_const.OK, result=data)
		return makeCommandResult(status=svs_const.ERROR, message="Requested data not found: '%s'" % label)




	
		

