# svs_core.gui.clientgui_wx

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Basic Graphical User Interface for SVS client using wxPython toolkit.


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
import sys
import wx
from twisted.internet import wxreactor
# if a reactor has already been installed it must be removed for wxreactor
if sys.modules.has_key('twisted.internet.reactor'):del sys.modules['twisted.internet.reactor']
wxreactor.install()
from twisted.internet import reactor

# wxPython events
ID_ENTER_BUTTON=110
ID_HISTORY_BUTTON=111
ID_RESET_BUTTON=112
ID_QUIT_BUTTON=113
ID_INPUTFIELD=120


class wxBaseApp(wx.App):
	"""
	Provides a basic delegate class for handling wx
	application calls.
	"""
	def __init__(self, gui):
		self.gui = gui
		wx.App.__init__(self, 0)
		
	def OnInit(self):
		"""
		Called by twisted reactor to initiate display.
		"""
		return True


class ClientGUI:
	def __init__(self, client):
		self.client = client
		self.viswin = None
		self.echoInput = False
		self.client.reactor = reactor

	def build(self):
		"""
		Creates interface components.
		"""
		# make wx base application
		self.app = wxBaseApp(self)
		# make root window
		self.root = wx.Frame(None, -1, 
				title='SVS client: %s' % self.client.profile.name, 
				size =(400,200),
				style=wx.DEFAULT_FRAME_STYLE|wx.NO_FULL_REPAINT_ON_RESIZE)
		wx.EVT_CLOSE(self.root, self.onClose)
		# add standard components
		self.buildStandardComponents()
		# show
		self.root.Show(True)
		self.app.SetTopWindow(self.root)
		# register with twisted
		self.client.reactor.registerWxApp(self.app)
			

	def buildStandardComponents(self):
		"""
		Creates interface components.
		"""
		### components ###
		self.mainSizer = wx.BoxSizer(wx.VERTICAL)
		#console
		self.consoleSizer = wx.BoxSizer(wx.VERTICAL)
		self.inputField = wx.TextCtrl(self.root, ID_INPUTFIELD)
		wx.EVT_CHAR(self.inputField, self.inputFieldAdjusted)
		self.console = wx.TextCtrl(self.root, -1, style=wx.TE_MULTILINE | wx.TE_READONLY)
		# input
		self.inputBuffer = []
		self.inputBufferMaxSize = 20
		self.inputBufferIndex = 0
		self.consoleSizer.Add(self.inputField, 0, wx.EXPAND)
		self.consoleSizer.Add(self.console, 1, wx.EXPAND)
		self.mainSizer.Add(self.consoleSizer, 1, wx.EXPAND)
		#Layout sizers
		self.root.SetSizer(self.mainSizer)
		self.root.SetAutoLayout(1)
		# show
		self.inputField.SetFocus()


		
	def onClose(self, event):
		self.root.Hide()
		self.client.disconnect()
			
	def statusMessage(self, text):
		"""
		Displays status message in console.
		"""
		self.console.AppendText('%s\n' % text)
	
	def errorMessage(self, text):
		"""
		Displays error message in console.
		"""
		self.console.AppendText('ERROR: %s\n' % text)

	def destroy(self, args=None):
		"""
		Closes and destroys interface components.
		"""
		self.client.disconnect()
		
	def clearLogDisplay(self):
		"""
		Clear console.
		"""
		self.console.Clear()
	
	def setEchoInput(self, state=True):
		"""
		If 'True' echoes input in console, if "False' then no echo.
		"""
		self.echoInput = state
		
	def inputFieldAdjusted(self, event):
		"""
		Deals with text from L{ClientGUI.inputField}.
		"""
		keycode = event.GetKeyCode()
		if keycode == wx.WXK_RETURN:
			command = self.inputField.GetValue()
			self.inputField.Clear()
			self.inputBuffer.append(command)
			if len(self.inputBuffer) > self.inputBufferMaxSize:
				self.inputBuffer = self.inputBuffer[-self.inputBufferMaxSize:]
			self.inputBufferIndex = len(self.inputBuffer) - 1
			self.enterCommand(command)
		elif keycode == wx.WXK_UP:
			self.stepBackCommand()
		elif keycode == wx.WXK_DOWN:
			self.stepForwardCommand()
		event.Skip()
		

	def enterCommand(self, command):
		if self.echoInput:self.console.AppendText(command+"\n")
		self.client.handleLocalCommand(command.encode('ascii'))
		

	def stepBackCommand(self):
		"""
		Step back one place in the buffer of input commands.
		"""
		if len(self.inputBuffer) > 0:
			cmdText = self.inputBuffer[self.inputBufferIndex]
			self.inputField.Clear()
			self.inputField.AppendText(cmdText)
			if self.inputBufferIndex >= 1:
				self.inputBufferIndex -= 1
		
	def stepForwardCommand(self):
		"""
		Step forward one place in the buffer of input commands.
		"""
		if 0 <= self.inputBufferIndex < len(self.inputBuffer)-1:
			self.inputBufferIndex += 1
			self.inputField.Clear()
			self.inputField.AppendText(self.inputBuffer[self.inputBufferIndex])
			

	def openView(self, fullscreen=False):
		"""
		Open visualisation view.  If 'fullscreen' is 'True' open
		as fullscreen display, otherwise in window.

		This can be overridden by extending classes.
		"""
		pass


	def hideView(self):
		"""
		Hides visualisation view.
		"""
		if self.viswin:self.viswin.hide()

	def destroyView(self, args=None):
		"""
		Destroys visualisation view.
		"""
		if not self.viswin:return
		self.viswin.destroy()
		self.viswin = None
		

	def update(self, timestamp):
		"""
		Updates view components.

		This should be overridden by extending classes.
		"""
		pass

			
