# svs_core.gui.clientgui

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Basic Graphical User Interface for SVS client.


@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""
# external imports
from Tkinter import *
from twisted.internet import tksupport



class ClientGUI:
	def __init__(self, client):
		self.client = client
		self.viswin = None
		self.echoInput = False

	def build(self):
		"""
		Creates interface components.
		"""
		# make root window
		self.root = Tk()
		self.root.title('SVS client: %s' % self.client.profile.name)
		self.root.geometry("%dx%d%+d%+d" % (400, 200, 0, 0))
		self.root.bind('<Destroy>', self.destroy)
		# register with twisted
		tksupport.install(self.root)
		# add standard components
		self.buildStandardComponents()

	def buildStandardComponents(self):
		"""
		Creates interface components.
		"""
		### components ###
		self.inputFrame = Frame(self.root)
		self.inputFrame.pack(side=TOP, expand=YES, fill=X)
		# input field
		self.inputField = Entry(self.inputFrame)
		self.inputField.pack(side=LEFT, expand=YES, fill=X)
		self.inputField.config(bg='white', fg='black')
		self.inputField.bind('<Return>', (lambda event: self.enterCommand()))
		self.inputField.bind('<Up>', (lambda event: self.stepBackCommand()))
		self.inputField.bind('<Down>', (lambda event: self.stepForwardCommand()))
		self.inputBuffer = []
		self.inputBufferMaxSize = 20
		self.inputBufferIndex = 0
		self.consoleFrame = Frame(self.root)
		self.consoleFrame.pack(side=BOTTOM, expand=YES, fill=X)
		self.sbar = Scrollbar(self.consoleFrame)
		self.console = Text(self.consoleFrame, relief=SUNKEN)
		self.sbar.config(command=self.console.yview)
		self.console.config(yscrollcommand=self.sbar.set)
		self.console.config(bg='white', fg='black')
		self.sbar.pack(side=RIGHT, fill=Y)
		self.console.pack(side=LEFT, expand=YES, fill=X)
		self.root.protocol("WM_DELETE_WINDOW", self.destroy)


		
	def close(self, args=None):
		"""
		Closes interface window.
		"""
		self.root.withdraw()

	def destroy(self, args=None):
		"""
		Closes interface components.
		"""
		if self.viswin:
			self.viswin.destroy()
		self.client.disconnect()
		self.root.quit()
		
	def statusMessage(self, text):
		"""
		Displays status message in console.
		"""
		self.console.insert(END, '%s\n' % text)
		self.console.mark_set(INSERT, END)
		self.console.see(END)

	def errorMessage(self, text):
		"""
		Displays error message in console.
		"""
		self.console.insert(END, 'ERROR: %s\n' % text)
		self.console.mark_set(INSERT, END)
		self.console.see(END)

	def clearLogDisplay(self):
		"""
		Clear console.
		"""
		self.console.delete('1.0', END)
	
	def setEchoInput(self, state=True):
		"""
		If 'True' echoes input in console, if "False' then no echo.
		"""
		self.echoInput = state

	def enterCommand(self):
		"""
		Handles text from input field as command.
		"""
		inputText = self.inputField.get()
		if self.echoInput:
			self.console.insert(END, '>> %s\n' % inputText)
		self.inputField.delete(0, END)
		self.inputBuffer.append(inputText)
		if len(self.inputBuffer) > self.inputBufferMaxSize:
			self.inputBuffer = self.inputBuffer[-self.inputBufferMaxSize:]
		self.inputBufferIndex = len(self.inputBuffer) - 1
		self.client.handleLocalCommand(inputText)

	def stepBackCommand(self):
		"""
		Step back one place in the buffer of input commands.
		"""
		if len(self.inputBuffer) == 0:return
		cmdText = self.inputBuffer[self.inputBufferIndex]
		self.inputField.delete(0, END)
		self.inputField.insert(0, cmdText)
		if self.inputBufferIndex > 0:
			self.inputBufferIndex -= 1
		
	def stepForwardCommand(self):
		"""
		Step forward one place in the buffer of input commands.
		"""
		if len(self.inputBuffer) == 0:return
		if self.inputBufferIndex < len(self.inputBuffer)-1:
			self.inputBufferIndex += 1
		cmdText = self.inputBuffer[self.inputBufferIndex]
		self.inputField.delete(0, END)
		self.inputField.insert(0, cmdText)
		

	def openView(self, fullscreen=False):
		"""
		Open visualisation view.  If 'fullscreen' is 'True' open
		as fullscreen display, otherwise in window.

		This can be overridden by extending classes.
		"""
		pass


	def hideView(self):
		"""
		Hides visualisation view.
		"""
		if self.viswin:self.viswin.hide()

	def destroyView(self, args=None):
		"""
		Destroys visualisation view.
		"""
		if not self.viswin:return
		self.viswin.destroy()
		self.viswin = None
		

	def setConsoleText(self, text='', file=None):
		"""
		Clear console and display new text.
		"""
		if file:
			text = open(file, 'r').read()
		self.console.delete('1.0', END)
		self.console.insert('1.0', text)
		self.console.mark_set(INSERT, END)
		self.console.see(END)

	def getConsoleText(self):
		"""
		Retrieves text contents from console.
		"""
		return self.console.get('1.0', END+'-1c')

	def update(self, timestamp):
		"""
		Updates view components.

		This should be overridden by extending classes.
		"""
		pass

			
