# svs_tracking.visualisation.collections

#    Copyright (c) 2005 Simon Yuill.
#
#    This file is part of 'Social Versioning System' (SVS).
#
#    'Social Versioning System' is free software; you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation; either version 2 of the License, or
#    (at your option) any later version.
#
#    'Social Versioning System' is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with 'Social Versioning System'; if not, write to the Free Software
#    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

"""
Objects for handling groups of visualisation spriteCells.

@author:	Simon Yuill
@copyright:	2005 Simon Yuill
@license:	GNU GPL version 2 or any later version
@contact:	simon@lipparosa.org
"""

# external imports
import math

# internal imports
from svs_core.geometry.geomlib import deg2rad, rad2deg
from svs_tracking.visualisation.sprites3d import Sprite3D

class SpriteCell:
	"""
	Container class for sprite within a collection.

	Handles membership and location within the collection.
	"""
	def __init__(self, sprite=None, locator=None):
		self.sprite = sprite
		self.locator = locator
		self.collection = None
	
	def update(self, timestamp=None):
		"""
		Passes update message to sprite.
		"""
		self.sprite.update(timestamp)

	def addToCollection(self, collection):
		"""
		Called when sprite is added to the specified collection.
		"""
		self.collection = collection
		self.sprite.addToScene(self.collection.scene)

	def removeFromCollection(self, collection=None):
		"""
		Called when sprite is removed from the collection.
		"""
		self.sprite.removeFromScene(self.collection.scene)
		self.collection = None


class SpriteCollection(Sprite3D):
	"""
	Generic collection class with no layout format.
	"""
	def __init__(self, x=0.0, y=0.0, z=0.0):
		Sprite3D.__init__(self, x, y, z)
		self.spriteCells = {}

	def modified(self):
		"""
		Registeres that the sprite will require to
		be re-rendered.
		"""
		if self.scene:self.scene.modified()

	def addSpriteCell(self, spriteCell, updateDisplay=False):
		"""
		Adds a new L{SpriteCell} to the scene.
		"""
		if self.spriteCells.has_key(spriteCell.locator):
			self.spriteCells[spriteCell.locator].removeFromCollection(self)
			self.spriteCells[spriteCell.locator] = None
		spriteCell.addToCollection(self)
		self.validateLocation(spriteCell)
		self.spriteCells[spriteCell.locator] = spriteCell
		if updateDisplay:self.modified()

	def removeSpriteCell(self, spriteCell, updateDisplay=False):
		"""
		Removes L{SpriteCell} from the scene.
		"""
		if self.spriteCells.has_key(spriteCell.locator):
			self.spriteCells[spriteCell.locator].removeFromCollection(self)
			self.spriteCells[spriteCell.locator] = None
		if updateDisplay:self.modified()
	
	def update(self, timestamp=None):
		"""
		Calls update on every L{SpriteCell} within the collection.
		"""
		for spriteCell in self.spriteCells.values():
			spriteCell.update(timestamp)

	def validateLocation(self, spriteCell):
		"""
		Is called by the L{SpriteCell} when it has been added to
		the collection, setting the physical location of the
		L{SpriteCell} in the correct place relative to that of the
		collection itself.

		This should be overridden by extending classes to 
		accomodate any layour features, such as spacings between
		sprite cells, that are particular to them.
		"""
		pass
		



class WheelCollection(SpriteCollection):
	"""
	Holds a collection of visualisation L{SpriteCell}s in a wheel and spoke structure.

	Branches from the centre point are placed on the x and y axes of the collection.
	"""
	def __init__(self, x=0.0, y=0.0, z=0.0, radius=0):
		SpriteCollection.__init__(self, x, y, z)
		self.radius = radius

	def validateLocation(self, spriteCell):
		"""
		Sets position of sprite to valid location within the wheel.
		"""
		spriteCell.locator.radius = self.radius
		spriteX = spriteCell.locator.radius * math.sin(deg2rad * spriteCell.locator.angle)
		spriteY = spriteCell.locator.radius * math.cos(deg2rad * spriteCell.locator.angle)
		spriteZ = self.z
		spriteCell.sprite.setLocation(spriteX, spriteY, spriteZ)


class ClusterCollection(SpriteCollection):
	"""
	This holds a collection of L{SpriteCell}s in a network clustered around a
	central point, and with a specified boundary.

	Branches from the centre point are placed on the x and y axes of the collection.
	"""
	def __init__(self, x=0.0, y=0.0, z=0.0, maxRadius=0):
		SpriteCollection.__init__(self, x, y, z)
		self.maxRadius = maxRadius

	def validateLocation(self, spriteCell):
		"""
		Makes sure the sprite is not outside the maximum radius
		for the cluster collection.  If L{ClusterCollection.maxRadius} is 0
		then this is ignored.
		"""
		if self.maxRadius > 0 and spriteCell.locator.radius > self.maxRadius: 
			spriteCell.locator.radius = self.maxRadius
		
		spriteX = self.x + (spriteCell.locator.radius * math.sin(deg2rad * spriteCell.locator.angle))
		spriteY = self.y + (spriteCell.locator.radius * math.cos(deg2rad * spriteCell.locator.angle))
		spriteZ = self.z
		print "validateLocation: %f, %f, %f" % (spriteX, spriteY, spriteZ)
		spriteCell.sprite.setLocation(spriteX, spriteY, spriteZ)
		


class GridCollection(SpriteCollection):
	"""
	Holds a collection of visualisation L{SpriteCell}s in a grid structure.
	"""
	def __init__(self, x=0.0, y=0.0, z=0.0, cellSpacingX=0.0, cellSpacingY=0.0, cellSpacingZ=0.0):
		SpriteCollection.__init__(self, x, y, z)
		self.cellSpacingX = cellSpacingX
		self.cellSpacingY = cellSpacingY
		self.cellSpacingZ = cellSpacingZ
	
	def validateLocation(self, spriteCell):
		"""
		Sets position of sprite so that it is properly located 
		in relation to the cell spacings for the grid.
		"""
		spriteX = self.x + (self.cellSpacingX * spriteCell.locator.x)
		spriteY = self.y + (self.cellSpacingY * spriteCell.locator.y)
		spriteZ = self.z + (self.cellSpacingZ * spriteCell.locator.z)
		spriteCell.sprite.setLocation(spriteX, spriteY, spriteZ)




	
	
