-- load-nff.lua -- Load a .nff (or .aff) format mesh
--
--  Copyright (C) 2008, 2010  Miles Bader <miles@gnu.org>
--
-- This source code is free software; you can redistribute it and/or
-- modify it under the terms of the GNU General Public License as
-- published by the Free Software Foundation; either version 3, or (at
-- your option) any later version.  See the file COPYING for more details.
--
-- Written by Miles Bader <miles@gnu.org>
--

--
-- NFF ("Neutral File Format") is the scene format used by Eric Haines'
-- SPD (Standard Procedural Databases) ray-tracing test scene generators.
--
-- AFF is an extension of NFF used by the "BART" ray-tracing benchmark
-- suite.
--
-- There are several problems with typical NFF scenes (at least, the
-- scenes generated by SPD package):
--
--  (1) Colors and light-levels are screwed up because the authors
--      apparently attempted to correct for missing gamma correction in
--      the output by mucking with the input colors/light-levels.
-- 
--      To generate output colors similar to reference images, set the
--      output gamma to 1 (and use an output format that doesn't record
--      gamma, so the image viewer thinks the image uses more typical
--      gamma correction).
-- 
--  (2) Light levels are set assuming linear fall-off of light intensity
--      with distance (which apparently used to be the default for POVray!).
--      We only support (physically correct) N^2 light-falloff, so there
--      will inevitably be some difference in lighting.
-- 
--      To fix this generally requires editing the input NFF file to boost
--      the levels of the lights; simply increasing the output exposure
--      using the snogray -e option will work too, but also increases the
--      brightness of any background color, which may be undesirable.
--  

local lp = require 'lpeg'
local lu = require 'lpeg-utils'

local nff_medium_absorption = 10
local nff_light_intens = 50

-- local abbreviations for lpeg primitives
local P, R, S, C = lpeg.P, lpeg.R, lpeg.S, lpeg.C

local HWS = lu.OPT_HORIZ_WS
local WS_NL = HWS * lu.NL
local WS_INT = HWS * lu.INT
local WS_FLOAT = HWS * lu.FLOAT
local WS_POS = (WS_FLOAT * WS_FLOAT * WS_FLOAT) / pos
local WS_VEC = (WS_FLOAT * WS_FLOAT * WS_FLOAT) / vec
local WS_COLOR = (WS_FLOAT * WS_FLOAT * WS_FLOAT) / color

-- this matches a newline, and also updates the error location
local WS_NL_SYNC = WS_NL * lu.ERR_POS

function load_nff (filename, scene, camera)
   local cur_mat = nil
   local cur_mesh = nil

   local vg = mesh_vertex_group ()
   local vng = mesh_vertex_normal_group ()

   local geom_xform = xform_z_to_y
   local norm_xform = geom_xform:inverse():transpose()

   camera:transform (scale (1,1,-1))
   camera:transform (geom_xform)

   local function check_mat ()
      if not cur_mat then
	 lu.parse_err "no current material"
      end
   end
   local function check_mesh ()
      check_mat ()
      if not cur_mesh then
	 cur_mesh = mesh (cur_mat)
	 cur_mesh.left_handed = false  -- NFF format uses right-handed meshes
      end
   end

   local function set_viewpoint (from, at, up, angle, hither, resol_x, resol_y)
      from = geom_xform (from)
      at = geom_xform (at)
      up = norm_xform (up)

      camera:move (from)
      camera:point (at, up)

      camera:set_vert_fov (angle * math.pi / 180)
   end

   local function set_bg_color (color)
      scene:set_background (color)
   end

   local function add_mat (diff, spec, shine, t, ior)
      if t > 0.001 then
	 local absorb = nff_medium_absorption
	    * -math.log (math.max (0.0001, math.min (t, 1)))
	 cur_mat = glass{ior=ior, absorb=absorb}
      else
	 spec = color (spec)
	 if shine > 1 and shine < 10000 then
	    local m = 1 / math.sqrt (shine)
	    cur_mat = cook_torrance{diff=diff, spec=spec, m=m}
	 else
	    cur_mat = lambert (diff)
	 end
	 if spec:intensity() > 0.01 then
	    cur_mat = mirror{reflect=spec, underlying=cur_mat}
	 end
      end
   end
   local function add_f_mat (color, kd, ks, shine, t, ior)
      add_mat (color * kd, ks, shine, t, ior)
   end
   local function add_fm_mat (amb, diff, spec, shine, t, ior)
      add_mat (diff, spec, shine, t, ior)
   end

   local function add_light (pos, color)
      pos = geom_xform (pos)

      if color then
	 color = color * nff_light_intens
      else
	 color = nff_light_intens
      end

      scene:add (point_light (pos, color))
   end

   local function add_cone (base_pos, base_radius, apex, apex_radius)
      check_mat()

      if apex_radius ~= base_radius then
	 lu.parse_err "non-cylinder cone not supported"
      end

      base_pos = geom_xform (base_pos)
      apex_pos = geom_xform (apex_pos)

      scene:add (
	 cylinder (cur_mat, base_pos, (apex_pos - base_pos), base_radius))
   end

   local function add_sphere (pos, radius)
      check_mat()
      scene:add (sphere (cur_mat, geom_xform (pos), radius))
   end

   -- This adds a polygon with many sides in a way that avoids generating
   -- lots of skinny little triangles (one of the SPD NFF scenes uses
   -- polygons with hundreds of sides).
   --
   local function add_many_sided_poly (...)
      local num_verts = select('#', ...)

      local center = pos (0,0,0)
      for i = 1, num_verts do
	 local pi = select (i, ...)
	 center = center + vec (pi.x, pi.y, pi.z)
      end
      center = center / num_verts
      center = geom_xform (center)
      center = cur_mesh:add_vertex (center, vg)

      local prev, v1
      for i = 1, num_verts do
	 local vi = select (i, ...)
	 vi = cur_mesh:add_vertex (geom_xform (vi), vg)

	 if prev then
	    cur_mesh:add_triangle (center, prev, vi)
	 else
	    v1 = vi
	 end

	 prev = vi
      end

      cur_mesh:add_triangle (center, prev, v1)
   end

   -- Add a polygon, without normals
   --
   local function add_poly (num_verts, v1, v2, v3, ...)
      check_mesh()

      if num_verts ~= 3 + select('#', ...) then
	 lu.parse_err "incorrect number of vertices in polygon"
      end

      if num_verts > 5 then
	 return add_many_sided_poly (v1, v2, v3, ...)
      end

      v1 = cur_mesh:add_vertex (geom_xform (v1), vg)
      v2 = cur_mesh:add_vertex (geom_xform (v2), vg)
      v3 = cur_mesh:add_vertex (geom_xform (v3), vg)
      cur_mesh:add_triangle (v1, v2, v3)

      -- Add extra triangles for additional vertices
      --
      local prev = v3
      for i = 1, select ('#', ...) do
	 local vi = select (i, ...)
	 vi = cur_mesh:add_vertex (geom_xform (vi), vg)
	 cur_mesh:add_triangle (v1, prev, vi)
	 prev = vi
      end
   end

   -- Add a polygon, with normals
   --
   local function add_norm_poly (num_verts, v1, n1, v2, n2, v3, n3, ...)
      check_mesh()

      if num_verts * 2 ~= 6 + select('#', ...) then
	 lu.parse_err "incorrect number of vertices in polygon"
      end

      v1 = cur_mesh:add_vertex (geom_xform (v1), norm_xform (n1), vng)
      v2 = cur_mesh:add_vertex (geom_xform (v2), norm_xform (n2), vng)
      v3 = cur_mesh:add_vertex (geom_xform (v3), norm_xform (n3), vng)
      cur_mesh:add_triangle (v1, v2, v3)

      -- Add extra triangles for additional vertices
      --
      local prev = v3
      for i = 1, select ('#', ...) - 1, 2 do
	 local vi = select (i, ...)
	 local nn = select (i+1, ...)
	 vi = cur_mesh:add_vertex (geom_xform (vi), norm_xform (nn), vng)
	 cur_mesh:add_triangle (v1, prev, vi)
	 prev = vi
      end
   end

   -- NFF grammar
   --
   local V_CMD
      = (P"v" * WS_NL_SYNC
	 * P"from" * WS_POS * WS_NL_SYNC
	 * P"at" * WS_POS * WS_NL_SYNC
	 * P"up" * WS_VEC * WS_NL_SYNC
	 * P"angle" * WS_FLOAT * WS_NL_SYNC
	 * P"hither" * WS_FLOAT * WS_NL_SYNC
	 * P"resolution" * WS_INT * WS_INT) / set_viewpoint
   local B_CMD
      = P"b" * WS_COLOR / set_bg_color
   local L_CMD
      = (P"l" * WS_POS * WS_COLOR^-1) / add_light
   local F_CMD
      = (P"f"* WS_COLOR * WS_FLOAT * WS_FLOAT
	  * WS_FLOAT * WS_FLOAT * WS_FLOAT) / add_f_mat
   local FM_CMD
      = (P"fm" * WS_COLOR * WS_COLOR * WS_COLOR
	 * WS_FLOAT * WS_FLOAT * WS_FLOAT) / add_fm_mat
   local C_CMD
      = (P"c"
	 * WS_NL_SYNC * WS_POS * WS_FLOAT
	 * WS_NL_SYNC * WS_POS * WS_FLOAT) / add_cone
   local S_CMD
      = (P"s" * WS_POS * WS_FLOAT) / add_sphere
   local CMD
      = (P"p" * WS_INT * (WS_NL_SYNC * WS_POS)^0) / add_poly
   local PP_CMD
      = (P"pp" * WS_INT * (WS_NL_SYNC * WS_POS * WS_VEC)^0) / add_norm_poly
   local CMD
      = (V_CMD + B_CMD + L_CMD + F_CMD + FM_CMD + C_CMD + S_CMD + CMD + PP_CMD)

   lu.parse_file (filename, CMD * WS_NL)

   if cur_mesh then
      scene:add (cur_mesh)
   end

   return true
end
