-- Copyright (C) 2010-2011 Papavasileiou Dimitris                           
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "physics"
require "frames"
require "switches"
require "transforms"
require "moremath"
require "automotive"
require "morewidgets"

local M = blueprint.masses
local P = blueprint.points
local L = blueprint.locks
local theta = blueprint.headangle

-- Position the vehicle so that the rear chassis
-- center of mass is just behind the start line.
-- This will start timing the lap as soon as the
-- vehicle launches.

local p_0 = {-P[7][1] - 0.01, 0, 0}
local R_0 = transforms.euler (0, 0, options.reverse and 180 or 0)
local V_0 = 0

local twistaxis = {math.cos(blueprint.headangle), 0, math.sin(blueprint.headangle)}
local steeraxis = {-math.sin(blueprint.headangle), 0, math.cos(blueprint.headangle)}

-- Create the machine.

local root = bodies.system {
   -- The swingarm.

   swingarm = bodies.point {
      name = "Swingarm",

      position = math.add (p_0, math.transform (R_0, P[12])),
      orientation = R_0,
      velocity = math.transform (R_0, {V_0, 0, 0}),

      mass = {
	 M[5][1],
	 {0, 0, 0},
	 {M[5][2], M[5][5], M[5][6], 
	  M[5][5], M[5][3], M[5][7], 
	  M[5][6], M[5][7], M[5][4]}
      },

      spindle = joints.hinge {
	 name = "Rear wheel spindle",

	 axis = math.transform (R_0, {0, 1, 0}),
	 anchor = math.add (p_0, math.transform (R_0, P[6])),
	 
	 wheel = automotive.wheel {
	    name = "Rear wheel",

	    position = math.add (p_0, math.transform (R_0, P[6])),
	    orientation = R_0,
	    velocity = math.transform (R_0, {V_0, 0, 0}),

	    iswheel = true,
	    tag = 1,

	    mass = {
	       M[7][1],
	       {0, 0, 0},
	       {M[7][2], M[7][5], M[7][6], 
		M[7][5], M[7][3], M[7][7], 
		M[7][6], M[7][7], M[7][4]}
	    },

	    resistance = blueprint.rearresistance,
	    radii = blueprint.rearradii,
	    elasticity = blueprint.rearelasticity,
	    load = blueprint.rearload,
	    relaxation = blueprint.rearrelaxation,

	    longitudinal = blueprint.rearlongitudinal,  	    
	    lateral = blueprint.rearlateral,
	    moment = blueprint.rearmoment,

	    scaling = blueprint.rearscaling,
	 },
      },

      upperpivot = joints.hinge {
	 name = "Swing-arm pivot",

	 axis = math.transform (R_0, {0, 1, 0}),
	 anchor = math.add (p_0, math.transform (R_0, P[10])),
      },      

      lowerpivot = joints.hinge {
	 name = "Suspension linkage pivot",

	 axis = math.transform (R_0, {0, 1, 0}),
	 anchor = math.add (p_0, math.transform (R_0, P[16])),
	 
	 arm = bodies.point {
	    name = "Suspension linkage arm",

	    position = math.add (p_0, math.transform (R_0, P[16])),
	    velocity = math.transform (R_0, {V_0, 0, 0}),

	    mass = {
	       1e-3,
	       {0, 0, 0},
	       {1e-6, 0, 0,
		0, 1e-6, 0,
		0, 0, 1e-6},
	    },

	    pivot = joints.hinge {
	       name = "Suspension linkage pivot",

	       axis = math.transform (R_0, {0, 1, 0}),
	       anchor = math.add (p_0, math.transform (R_0, P[14])),
	       
	       arm = bodies.point {
		  name = "Suspension linkage arm",

		  position = math.add (p_0, math.transform (R_0, P[14])),
		  velocity = math.transform (R_0, {V_0, 0, 0}),

		  mass = {
		     1e-3,
		     {0, 0, 0},
		     {1e-6, 0, 0,
		      0, 1e-6, 0,
		      0, 0, 1e-6},
		  },

		  rightpivot = joints.hinge {
		     name = "Suspension linkage pivot",

		     axis = math.transform (R_0, {0, 1, 0}),
		     anchor = math.add (p_0, math.transform (R_0, P[13])),
		  },

		  leftpivot = joints.hinge {
		     name = "Rear damper pivot",

		     axis = math.transform (R_0, {0, 1, 0}),
		     anchor = math.add (p_0, math.transform (R_0, P[15])),
		     
		     lowershock = bodies.point {
			name = "Rear damper rod",
			position = math.add (p_0, math.transform (R_0, P[15])),
			velocity = math.transform (R_0, {V_0, 0, 0}),

			mass = {
			   1e-3,
			   {0, 0, 0},
			   {1e-6, 0, 0,
			    0, 1e-6, 0,
			    0, 0, 1e-6},
			},

			slider = joints.slider {
			   name = "Rear damper piston",

			   axis = math.transform (R_0, {-0.13619, 0, -0.99068}),

			   stops = {blueprint.rearlocks, {0, 0.2}, 0.3},
			   
			   stiffness = blueprint.rearstiffness,
			   damping = blueprint.reardamping,
			   preload = blueprint.rearpreload,

			   step = function (self)
			   	     local l, V = unpack(self.state)
			   	     local F

			   	     -- if l <= self.stops[1][1] then
			   	     -- 	print "rear bottomed out!"
			   	     -- elseif l >= self.stops[1][2] then
			   	     -- 	print "rear topped out!"
			   	     -- end

			   	     F = self.stiffness * (l - self.preload)

				     self.motor = {0,
						   (V < 0 and
						    self.damping[1] or
						    self.damping[2]) * V}

			   	     physics.addforce (self.parent,
			   			       math.scale(self.axis, -F))
			   	     physics.addforce (self.uppershock,
			   			       math.scale(self.axis, F))
			   	  end,

			   uppershock = bodies.point {
			      name = "Rear damper cylinder",
			      position = math.add (p_0, math.transform (R_0, P[11])),
			      velocity = math.transform (R_0, {V_0, 0, 0}),
			
			      mass = {
				 1e-3,
				 {0, 0, 0},
				 {1e-6, 0, 0,
				  0, 1e-6, 0,
				  0, 0, 1e-6},
			      },

			      pivot = joints.hinge {
				 name = "Rear damper pivot",

				 axis = math.transform (R_0, {0, 1, 0}),
				 anchor = math.add (p_0, math.transform (R_0, P[11])),
			      },
			   },
			},
		     },
		  },
	       },
	    },
	 },
      },

      link = function (self)
		self.upperpivot.reference = self.parent.rear
		self.lowerpivot.arm.pivot.arm.rightpivot.reference = self.parent.rear
		self.lowerpivot.arm.pivot.arm.leftpivot.lowershock.slider.uppershock.pivot.reference = self.parent.rear
	     end,
   },

   -- The front frame.

   front = bodies.point {
      name = "Front frame",

      position = math.add (p_0, math.transform (R_0, P[3])),
      orientation = math.concatenate(R_0, transforms.euler (0, -math.deg(theta), 0)),
      velocity = math.transform (R_0, {V_0, 0, 0}),

      mass = {
	 M[1][1],
	 {0, 0, 0},
	 {M[1][2], M[1][5], M[1][6], 
	  M[1][5], M[1][3], M[1][7], 
	  M[1][6], M[1][7], M[1][4]}
      },

      servo = motors.euler {
	 name = "Steering servo",

      	 axes = {math.transform(R_0, {0, 0, 1}),
		 nil,
		 math.transform(R_0, {1, 0, 0})},

	 stops = {
	    {{0, 0}, physics.spring (unpack(blueprint.ridercoupling)), 0},
	 },
      },

      stem = joints.universal {
	 name = "Steering stem",

	 anchor = math.add (p_0, math.transform (R_0, P[2])),
	 axes = {math.transform (R_0, steeraxis),
		 math.transform (R_0, twistaxis)},

	 stops = {
	    {{-0.5 * blueprint.steeringtravel,
	      0.5 * blueprint.steeringtravel}, {0, 0.2}, 0.1},
	    {{0, 0}, physics.spring(blueprint.chassisstiffness,
	    			    blueprint.chassisdamping), 0},
	 },

	 damping = blueprint.steeringdamping,

	 step = function(self)
	 	   -- This implements the steering damper.

	 	   self.motor = {
	 	      {0, math.abs(self.damping * self.state[3])},
	 	      {0, 0}
	 	   }
	 end,
      },

      slider = joints.slider {
	 name = "Front damper piston",

	 axis = math.transform (R_0, steeraxis),
	 stops = {blueprint.frontlocks, {0, 0.2}, 0.3},
	 
	 stiffness = blueprint.frontstiffness,
	 damping = blueprint.frontdamping,
	 preload = blueprint.frontpreload,

	 step = function (self)
	 	   local l, V = unpack(self.state)
	 	   local F

	 	   -- if l <= self.stops[1][1] then
	 	   --    print "front bottomed out!"
	 	   -- elseif l >= self.stops[1][2] then
	 	   --    print "front topped out!"
	 	   -- end

	 	   F = self.stiffness * (l - self.preload)

		   self.motor = {0, (V < 0 and
				     self.damping[1] or
				     self.damping[2]) * V}

	 	   physics.addforce (self.parent, math.scale(self.axis, -F))
	 	   physics.addforce (self.legs, math.scale(self.axis, F))
	 	end,

	 legs = bodies.point {
	    name = "Fork tubes",

	    position = math.add (p_0, math.transform (R_0, P[4])),
	    orientation = math.concatenate(R_0, transforms.euler (0, -math.deg(theta), 0)),
	    velocity = math.transform (R_0, {V_0, 0, 0}),


	    radius = 0.03,
	    length = 0.3,

	    mass = {
	       M[2][1],
	       {0, 0, 0},
	       {M[2][2], M[2][5], M[2][6], 
		M[2][5], M[2][3], M[2][7], 
		M[2][6], M[2][7], M[2][4]}
	    },

	    spindle = joints.hinge {
	       name = "Front wheel spindle",

	       axis = math.transform (R_0, {0, 1, 0}),
	       anchor = math.add (p_0, math.transform (R_0, P[5])),

	       wheel = automotive.wheel {
		  name = "Front wheel",

	       	  position = math.add (p_0, math.transform (R_0, P[5])),
	       	  orientation = R_0,
		  velocity = math.transform (R_0, {V_0, 0, 0}),  

	       	  iswheel = true,
	       	  tag = 2,

	       	  mass = {
		     M[6][1],
		     {0, 0, 0},
		     {M[6][2], M[6][5], M[6][6], 
		      M[6][5], M[6][3], M[6][7], 
		      M[6][6], M[6][7], M[6][4]}
	       	  },

		  resistance = blueprint.frontresistance,
		  radii = blueprint.frontradii,
		  elasticity = blueprint.frontelasticity,
		  load = blueprint.frontload,
		  relaxation = blueprint.frontrelaxation,

		  longitudinal = blueprint.frontlongitudinal,  	    
		  lateral = blueprint.frontlateral,
		  moment = blueprint.frontmoment,

		  scaling = blueprint.frontscaling,
	       },
	    },
	 },
      },
      
      link = function (self)
		self.stem.reference = self.parent.rear
		self.servo.reference = self.parent.rear.waist.torso
	     end,
   },

   -- The rear frame including rider.

   rear = bodies.box {
      name = "Rear frame",

      position = math.add (p_0, math.transform (R_0, P[7])),
      orientation = R_0,
      velocity = math.transform (R_0, {V_0, 0, 0}),

      isvehicle = true,

      size = math.scale({0.5, 0.1, 0.35}, 1),

      mass = {
	 M[3][1],
	 {0, 0, 0},
	 {M[3][2], M[3][5], M[3][6], 
	  M[3][5], M[3][3], M[3][7], 
	  M[3][6], M[3][7], M[3][4]}
      },

      standswitch = switches.toggle {
      	 step = function (self)
      		       -- Deploy stand below 5 kmh.

      		       self.state = math.length(self.parent.velocity) <
      		                    units.kilometersperhour(5)
      		    end,
	 
      	 on = frames.node {
      	    link = function (self)
      	       -- print ("Sidestand on.")

      	       self.ancestry[2].standmotor = motors.euler {
		  name = "Side-stand",
      		  axes = {transforms.fromnode (self.ancestry[2], {1, 0, 0}),
      			  transforms.fromnode (self.ancestry[2], {0, 1, 0}),
      			  transforms.fromnode (self.ancestry[2], {0, 0, 1})},
		  
      		  stops = {{{0, 0}, physics.spring (5000, 500), 0},
      		  	   nil,
      		  	   nil},

      	       }
      	    end,
	    
      	    unlink = function (self)
      			-- print ("Sidestand off.")
			
      			self.ancestry[2].standmotor = nil
      		     end,
      	 }, 
      },

      waist = joints.universal {
	 name = "Rider waist",

	 anchor = math.add (p_0, math.transform (R_0, P[8])),
	 axes = {math.transform (R_0, {1, 0, 0}),
		 math.transform (R_0, {0, 0, 1})},

	 stops = {
	    {{0, 0}, physics.spring (unpack(blueprint.riderroll)), 0},
	    {{0, 0}, physics.spring (unpack(blueprint.rideryaw)), 0},
	 },

	 torso = bodies.box {
	    name = "Rider upper body",

	    position = math.add (p_0, math.transform (R_0, P[9])),
	    orientation = R_0,
	    velocity = math.transform (R_0, {V_0, 0, 0}),

	    isvehicle = true,

	    size = math.scale({0.6, 0.24, 0.15}, 1),
	    
	    mass = {
	       M[4][1],
	       {0, 0, 0},
	       {M[4][2], M[4][5], M[4][6], 
		M[4][5], M[4][3], M[4][7], 
		M[4][6], M[4][7], M[4][4]}
	    },
	 },
      },

      observer = options.external and frames.gimbal {
	 options.side and frames.observer {
	    position = {0, -3, 0},
	    orientation = transforms.relue (0, 90, -90),
	 } or frames.observer {
	    position = {-5, options.rear and 0 or 1.5, 1},
	    orientation = transforms.relue (90, 90, -90),
	 },
      } or frames.observer {
	 position = {0.4, 0, 1},
	 orientation = transforms.relue (90, 90, -90),
      },

      engine = automotive.fourstroke {
	 name = "Engine",

      	 axis = math.transform (R_0, {0, 1, 0}),
      	 anchor = math.add (p_0, math.transform (R_0, P[7])),

	 displacement = blueprint.displacement,
	 cylinders = blueprint.cylinders,
	 intake = blueprint.intake,
	 volumetric = blueprint.volumetric,
	 thermal = blueprint.thermal,
	 exchange = blueprint.exchange,
	 friction = blueprint.friction,

	 peak = blueprint.peak,

	 flywheel = bodies.point {
	    name = "Flywheel",

	    position = math.add (p_0, math.transform (R_0, P[7])),
	    orientation = R_0,
	    spin = math.transform (R_0,
				   {0, units.rotationsperminute(1300), 0}),

      	    mass = {
      	       1e-3,
      	       {0, 0, 0},
      	       {1e-6, 0, 0,
      		0, blueprint.inertia, 0,
      		0, 0, 1e-6}
      	    },

	    clutch = joints.hinge {
	       name = "Clutch",

	       axis = math.transform (R_0, {0, 1, 0}),
	       anchor = math.add (p_0, math.transform (R_0, P[7])),
	       motor = {0, 1 / 0},
 
	       mainshaft = bodies.point {
		  name = "Mainshaft",

		  position = math.add (p_0, math.transform (R_0, P[7])),
		  orientation = R_0,
		  spin = math.transform (R_0,
					 {0, units.rotationsperminute(1300), 0}),

		  mass = {
		     1e-3,
		     {0, 0, 0},
		     {1e-6, 0, 0,
		      0, 1e-6, 0,
		      0, 0, 1e-6}
		  },

		  gearbox = joints.gearing {
		     name = "Gearbox",

		     ratio = nil,
		     
		     link = function (self)
			       self.reference = self.ancestry[5].countershaft.sprocket
			    end,
		  }
	       }, 
	    },
	 },

	 exhaust = resources.loop "slipstream/waves/engine.lc" {
	    reference = 1 / 0,

	    gain = 0,

	    prepare = function (self)
	 		 local m
	 
	 		 m = (math.clamp((1256.6 - self.parent.state[2]) / 418.88,
	 				 0, 1) *
	 		      math.clamp((self.parent.state[2] - 418.88) / 418.88,
	 				 0, 1))

	 		 self.gain = (0.4 + 0.6e-5 * self.parent.state[3])--*m
	 
	 		 self.pitch = self.parent.state[2] / 837.76
	 	      end
	 },
      },

      countershaft = joints.hinge {
	 name = "Countershaft",

      	 -- The countershaft axis location is calculated as an offset
      	 -- from the swingarm pivot location.

      	 axis = math.transform (R_0, {0, 1, 0}),
      	 anchor = math.add (p_0, math.transform (R_0, P[17])),

      	 sprocket = bodies.point {
      	    position = math.add (p_0, math.transform (R_0, P[17])),
      	    orientation = R_0,

      	    mass = {
      	       1e-3,
      	       {0, 0, 0},
      	       {1e-6, 0, 0,
      		0, 1e-6, 0,
      		0, 0, 1e-6}
      	    },

      	    chain = automotive.chain {
      	       axis = {0, 1, 0},
	       radii = blueprint.sprockets,
	 
      	       link = function (self)
      	    		 self.reference = self.ancestry[4].swingarm.spindle.wheel--.damper.sprocket
      	    	      end,
      	    }
      	 },
      },

      aerodynamics = frames.node {
	 drag = blueprint.drag,
	 lift = blueprint.lift,
	 pitch = blueprint.pitch,
	 area = blueprint.area,
	 
	 step = function (self)
		   local rho, C_D, C_L, C_P, f_A, V

		   V = math.length (self.parent.velocity)
		   rho = 1.225

		   physics.addrelativeforce (self.parent,
					     {-0.5 * self.drag * rho *
					      self.area * V * V, 0, 0},
					     P[1])

		   physics.addrelativeforce (self.parent,
					     {0, 0, 0.5 * self.lift * rho *
					      self.area * V * V},
					     P[1])

		   physics.addrelativetorque (self.parent,
					      {0, -0.5 * self.pitch * rho *
					       self.area * 1.41 * V * V, 0})
		end,
      }
   },

   control = slipstream.control {
      rubberband = slipstream.rubberband {
	 anchor = blueprint.anchor or {0.5, 0.75},

	 -- Steering control.

	 steering = slipstream.cascade {
	    axis = 1,

	    slipstream.module {
	       update = function (self)
			   local V, gamma
			   
			   V = math.length(parts.rearframe.velocity)
			   gamma = math.abs(parts.rearwheel.state[1])
			   
			   self.output = (blueprint.steeringsensitivity[1] +
					  V * blueprint.steeringsensitivity[2]) /
			   (1 + blueprint.steeringsensitivity[3] *
			    math.abs(gamma / math.pi * 3)) *
			-self.input
		  end
	    },

	    slipstream.lag {lag = blueprint.steeringlag},
	    slipstream.blackbox {channel = 1},

	    slipstream.module {
	       update = function (self)
			   local beta = math.abs(parts.frontwheel.state[3])
			   local N = parts.frontwheel.state[4]
			   local a = blueprint.loadregion or {100, 0}
			   local b = blueprint.sideslipregion or {0.25, 0.3}

			   if N < a[1] then
			      self.input = self.input * math.clamp((a[2] - N) / (a[2] - a[1]), 0, 1)
			   end

			   if beta > b[1] then
			      self.input = self.input * math.clamp((b[2] - beta) / (b[2] - b[1]), 0, 1)
			   end
			   
			   if math.length (parts.rearframe.velocity) >
			      units.kilometersperhour(5) then
			      physics.addtorque (parts.servo.parent,
						 math.scale(parts.servo.axes[1], self.input))

			      physics.addtorque (parts.servo.reference,
			      			 math.scale(parts.servo.axes[1], -self.input))
			   end
		        end,
	    },
	 },

	 -- Throttle control.

	 throttle = slipstream.cascade {
	    axis = 2,

	    slipstream.module {
	       update = function (self)
			   self.output = tonumber(options.throttle) or
			   math.clamp(self.input *
				      blueprint.throttlesensitivity, 0, 1)
		        end,
	    },

	    slipstream.lag {lag = blueprint.throttlelag},

	    slipstream.blackbox {channel = 2},
	    slipstream.module {
	       update = function (self)
			   parts.engine.throttle = self.input
		        end,
	    },
	 },

	 limiter = slipstream.cascade {
	    slipstream.module {
	       engaged = false,

	       update = function (self)
			   if blueprint.limiterthreshold and
			      blueprint.limiterhysteresis then
			      limit = units.rotationsperminute(blueprint.limiterthreshold + (self.engaged and -1 or 1) * blueprint.limiterhysteresis)
			   else
			      limit = 1 / 0
			   end
			   
			   if parts.engine.state[2] > limit then
			      self.output = 0
			      self.engaged = true
			   else
			      self.output = 1
			      self.engaged = false
			   end
			end
	    },

	    slipstream.module {
	       update = function (self)
			   parts.engine.spark = self.input > 0
		        end,
	    }
	 },

	 -- Front brake control.

	 frontbrake = slipstream.cascade {
	    axis = 2,

	    slipstream.module {
	       update = function (self)
			   self.output = tonumber(options.brake) or
			   math.clamp(-self.input *
				      blueprint.frontbrakesensitivity,
				      0, 1 / 0)
		        end,
	    },

	    slipstream.lag {lag = blueprint.frontbrakelag},
	    slipstream.blackbox {channel = 3},
	    
	    slipstream.module {
	       update = function (self)
			   parts.frontwheel.parent.motor = {0, self.input}
		        end,
	    },
	 },

	 -- Rear brake control.

	 rearbrake = slipstream.cascade {
	    axis = 2,

	    slipstream.module {
	       update = function (self)
			   self.output = tonumber(options.brake) or
			   math.clamp(-self.input *
				      blueprint.rearbrakesensitivity,
				      0, 1 / 0)
		        end,
	    },

	    slipstream.lag {lag = blueprint.rearbrakelag},
	    slipstream.blackbox {channel = 4},

	    slipstream.module {
	       update = function (self)
			   parts.rearwheel.parent.motor = {0, self.input}
		        end,
	    },
	 },
      },

      -- Clutch control.

      clutch = slipstream.cascade {
      	 axis = 2,

      	 slipstream.module {
      	    update = function (self)
      			self.output = 0
      		     end,
      	 },

      	 slipstream.blackbox {channel = 5},

      	 slipstream.module {
      	    peak = blueprint.peak,

      	    update = function (self)
      			parts.clutch.motor = {0, self.peak * (1 - self.input)}
      		     end,
      	 },
      },

      -- Gearbox control.
      
      gearbox = slipstream.cascade {
	 primary = blueprint.primary,
	 gears = blueprint.gearbox,

	 slipstream.module {

	    frames.event {
	       buttonpress = function (self, button)
				if button == 1 and
				   self.parent.output < #self.ancestry[2].gears then
				   self.parent.output = self.parent.output + 1
				end

				if button == 3 and self.parent.output > 0 then
				   self.parent.output = self.parent.output - 1
				end
			     end
	    }
	 },

	 slipstream.blackbox {channel = 6},

	 slipstream.module {
	    update = function (self)
			local q = self.parent.gears[self.input]

			parts.gearbox.gear = self.input
			parts.gearbox.ratio = q and q * self.parent.primary
		     end,
	 },
      },
   },

   profiler = frames.node {
      laps = {},

      link = function (self)
		dynamics.collision.probe =
		   function (a, b)
		      local track, sampler
		      local last, sum, top, samples = -1, 0, 0, 0

		      if a.istrack or b.istrack then
			 track = a.istrack and a or b
			 sampler = track.sampler

			 self.step =
			    function (self)
			       local p = sampler[self.parent.rear.position]
			       local now = p and p[1] or -1
			       local speed = math.length(self.parent.rear.velocity)
			       
			       if last ~= now then
				  if last == #track - 1 and 
				     now == 0 then
				     table.insert (self.laps,
						   {configuration.time[1],
						    sum / samples,
						    top})
				  end

				  last, top, sum, samples = now, 0, 0, 0
			       end

			       top = speed > top and speed or top
			       sum = sum + speed
			       samples = samples + 1
			    end
			 
			 dynamics.collision.probe = nil
		      end
		   end
	     end,

      unlink = function (self)
		  -- Don't overwrite recorded profiles
		  -- when replaying.

		  if not session.laps then
		     session.laps = self.laps
		  end
	       end
   },

   display = not options.nowidgets and widgets.display {
      size = {graphics.window[1] / graphics.window[2], 1},
      padding = {0.01, 0.01},
      align = {-1, -1},
      
      widgets.frame {
	 color = {1.0, 0.8, 0.2},
	 radius = 0.01,

	 widgets.row {
	    size = {0, 1},
	    align = {0, -1},
	    padding = {0.01, 0.01},

	    [2] = widgets.column {
	       padding = {0.01, 0},

	       face = widgets.clock {
		  color = {1.0, 0.8, 0.2},
		  opacity = 0.6,

		  radius = {0.07, 0.05},
		  range = {0, 16000},
		  spacing = {200, 5},
		  spread = {-4 * math.pi / 5, 4 * math.pi / 5},

		  traverse = function (self)
				self.reading = math.abs(parts.engine.state[2]) / math.pi * 30
			     end,

		  markings = frames.transform {
		     [1] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">0</span>"
		     },

		     [3] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">2</span>"
		     },

		     [5] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">4</span>"
		     },

		     [7] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">6</span>"
		     },

		     [9] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">8</span>"
		     },

		     [11] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">10</span>"
		     },

		     [13] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">12</span>"
		     },

		     [15] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">14</span>"
		     },

		     [17] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">16</span>"
		     },
		  },

		  hands = frames.transform {
		     [1] = shapes.line {
			color = {1.0, 0.8, 0.2},

			[1] = {0, -0.1},
			[2] = {0, 0.75},
			[3] = {0.05, 0},
			[4] = {0, -0.1},
			[5] = {-0.05, 0},
			[6] = {0, 0.75},
		     },
		  },
	       },

	       label = widgets.layout {
		  color = math.scale ({1.0, 0.8, 0.2}, 1.2),
		  padding = {0.01, 0.01},

		  text = "<span font=\"Sans 10\" color=\"White\">Tachometer</span>",
	       }
	    },

	    [3] = widgets.layout {
	       traverse = function (self)
			     local text
			     local laps = self.ancestry[4].profiler.laps

			     text = string.format ("<span font=\"Sans 20\" color=\"White\">Speed: %d</span>\n<span font=\"Sans 14\" color=\"White\">Lean: %.1f\194\176</span>\n<span font=\"Sans 14\" color=\"White\">Gear: %s</span>",
							 math.length(parts.rearframe.velocity) / units.kilometersperhour(1),
							 parts.rearwheel.state[1] / units.degrees(1),
							 parts.gearbox.gear == 0 and "N" or parts.gearbox.gear)

			     if #laps > 0 then
				local time
				local m, s = math.modf((configuration.time[1] - laps[#laps][1]) / 60)
				
				-- Break up time into mm:ss.

				if m > 0 then
				   time = string.format ("%d' %.2f\"", m, s * 60)
				else
				   time = string.format ("%2.2f\"", s * 60)
				end

				text = text .. string.format ("\n<span font=\"Sans 14\" color=\"White\">Lap: %s</span>", time)
			     end

			     self.text = text
			  end,
	    },

	    [1] = widgets.column {
	       padding = {0.01, 0},

	       face = widgets.clock {
		  color = {1.0, 0.8, 0.2},
		  opacity = 0.6,
		  
		  radius = {0.06, 0.04},
		  range = {0, 100},
		  spacing = {2, 10},
		  spread = {-4 * math.pi / 5, 4 * math.pi / 5},

		  traverse = function (self)
				self.reading = parts.engine.state[3] / 1000
			     end,

		  markings = frames.transform {
		     [1] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">0</span>"
		     },

		     [2] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">20</span>"
		     },

		     [3] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">40</span>"
		     },

		     [4] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">60</span>"
		     },

		     [5] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">80</span>"
		     },

		     [6] = widgets.layout {
			text = "<span font=\"Sans 7\" color=\"White\">100</span>"
		     },
		  },

		  hands = frames.transform {
		     [1] = shapes.line {
			color = {1.0, 0.8, 0.2},

			[1] = {0, -0.1},
			[2] = {0, 0.75},
			[3] = {0.05, 0},
			[4] = {0, -0.1},
			[5] = {-0.05, 0},
			[6] = {0, 0.75},
		     },
		  },
	       },

	       label = widgets.layout {
		  color = math.scale ({1.0, 0.8, 0.2}, 1.2),
		  padding = {0.01, 0.01},

		  text = "<span font=\"Sans 10\" color=\"White\">MAP(kPa)</span>",
	       }
	    },
	 }
      }
   },
	 
   link = function (self)
	     parts = {
		swingarm = self.swingarm,
		rearwheel = self.swingarm.spindle.wheel,
		frontwheel = self.front.slider.legs.spindle.wheel,
		steering = self.front.stem,
		servo = self.front.servo,
		pilot = self.rear.waist.torso,
		rearframe = self.rear,
		engine = self.rear.engine,
		gearbox = self.rear.engine.flywheel.clutch.mainshaft.gearbox,
		clutch = self.rear.engine.flywheel.clutch,
		rearsuspension = self.swingarm.lowerpivot.arm.pivot.arm.leftpivot.lowershock.slider,
		frontsuspension = self.front.slider,
		finaldrive = self.rear.countershaft.sprocket.chain,
		control = self.control,
	     }
	  end,

   unlink = function (self)
	       parts = nil
	    end,
}

-- Setup collision response.

dynamics.collision.vehicle = function (a, b)
				if a.isvehicle or b.isvehicle then
			   -- print ("Other!")
				   return 3, 0.4, 0.1
				end
			     end

return root
