-- Copyright (C) 2010-2011 Papavasileiou Dimitris                           
--                                                                      
-- This program is free software: you can redistribute it and/or modify 
-- it under the terms of the GNU General Public License as published by 
-- the Free Software Foundation, either version 3 of the License, or    
-- (at your option) any later version.                                  
--                                                                      
-- This program is distributed in the hope that it will be useful,      
-- but WITHOUT ANY WARRANTY; without even the implied warranty of       
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        
-- GNU General Public License for more details.                         
--                                                                      
-- You should have received a copy of the GNU General Public License    
-- along with this program.  If not, see <http://www.gnu.org/licenses/>.

require "frames"
require "switches"

function slipstream.control (values)
   self = frames.node {
      phase = 1,
      granularity = 10,

      name = "Control",

      step = function (self)
		if self.phase == self.granularity then
		   -- Update each system.

		   for key, child in children (self) do
		      if child.update then
			 child.update(child)
		      end
		   end
		   
		   self.phase = 1
		else
		   self.phase = self.phase + 1
		end
	     end,

      comparator = frames.node {
	 phase = 1,
	 granularity = 1000,

      	 link = function (self)
      		if session.channels and session.channels[0] then
		   self.entries = session.channels[0]
		else
      		   self.entries = {}
      		end
	      
      		self.index = 1
      	     end,		

	 unlink = function (self)
		     if not session.channels then
			session.channels = {}
		     end

		     if not session.channels[0] and
			#self.entries > 0 then
			session.channels[0] = self.entries
		     end
		  end,

      	 step = function (self)
		if self.phase == self.granularity then
		   if session.channels then
		      if self.index <= #self.entries then
			 self.delta = math.distance (
			    self.entries[self.index],
			    parts.rearframe.position)
			 
			 print(string.format ("Positional replay error is %.4g.", self.delta))
		      else
			 self.step = nil
		      end
		   else
		      self.entries[self.index] = parts.rearframe.position
		   end   
		   
		   self.index = self.index + 1
		   self.phase = 1
		else
		   self.phase = self.phase + 1
		end
     	     end,
      },
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.module (values)
   self = frames.node {
      input = 0,
      output = 0,

      name = "Module",
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.cascade (values)
   self = frames.node {
      input = 0,
      output = 0,

      name = "Cascade",
      
      update = function(self)
		  local i, input

		  -- Gather the results of each system.

		  i, input = 1, self.input

		  while self[i] do
		     self[i].input = input
		     
		     if self[i].update then
			self[i].update(self[i])
		     end

		     input, i = self[i].output, i + 1
		  end
		  
		  self.output = input
	       end
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.rubberband (values)
   self = switches.toggle {
      commands = {0, 0},

      -- The actual mouse cursor.

      cursor = frames.cursor {
	 shapes.points {
	    color = {1, 0.7, 0},
	    opacity = 0.9,
	    width = 6,
	 
	    [1] = {0, 0},
	 },
      },

      -- The rest of the rubber-band.
      
      on = frames.cursor {
	 knob = shapes.points {
	    color = {1, 0.7, 0},
	    opacity = 0.9,
	    width = 6,
	 
	    [1] = {0, 0},
	 },
      
	 vector = shapes.lines {
	    stipple = {3, 0x5555},
	    opacity = 0.8,
	    width = 3,
	 
	    [1] = {0, 0},
	    [2] = {0, 0},
	 },
      },

      link = function (self)
	 self.events = not session.channels and frames.event {
	    motion = function (self, button, x, y)
			if self.parent.state then
			   self.parent.commands = {self.origin[1] - x,
						   self.origin[2] - y}
			   
			   self.parent.on.knob[1] = self.parent.commands
			   self.parent.on.vector[1] = self.parent.commands
			   self.parent.on.vector.color =
			      self.parent.commands[2] < 0 and
			      {1, 0, 0} or {0.1607, 0.9763, 0}
			end
		     end,
	    
	    buttonpress = function (self, button)
			if button == 2 then
			   self.parent.state = not self.parent.state

			   if self.parent.state then
			      self.origin = math.multiply (
				 graphics.window,
				 self.parent.anchor)

			      graphics.grabinput = true
			      graphics.pointer = math.subtract (
				 self.origin, self.parent.commands)
			   else
			      graphics.grabinput = false
			   end
			end
		     end,
	 }
      end,
      
      update = function(self)
		  for key, child in children (self) do
		     if child.update then
			child.input = child.axis and
			   self.commands[child.axis] or 0

			child.update(child)
		     end
		  end   
			
		  self.output = 0
	       end,
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.lag (values)
   self = frames.node {
      input = 0,
      output = 0,
      lag = 0,

      name = "Lag",
      
      step = function (self, h, t)
		if self.lag > 0 then
		   self.output = (self.output +
				  (self.input - self.output) *
			          h / self.lag)
		else
		   self.output = self.input
		end
	     end,
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end

function slipstream.blackbox (values)
   self = frames.node {
      input = 0,
      output = 0,

      name = "Blackbox",
      
      link = function (self)
		-- If channel data already exists in the
		-- blueprint then we replay it, otherwise
		-- just initialize.

		self.entries = {}

		if session.channels and
		   session.channels[self.channel] then
		   local length, count

		   length, count = 0, 0
		   channel = session.channels[self.channel]
		   
		   -- Find the actual channel length.

		   for i, entry in pairs (channel) do
		      if i > length then
			 length = i
		      end
		      
		      count = count + 1
		   end

		   -- Unpack the saved entries and fill in
		   -- empty values.

		   for i = 1, length do
		      if channel[i] then
			 self.entries[i] = channel[i]
		      else
			 self.entries[i] = self.entries[i - 1]
		      end
		   end

		   print (string.format("Loaded %d entries (uncompressed from %d distinct values) for black box channel %d.", #self.entries, count, self.channel))
		end

		self.index = 1
	     end,

      unlink = function (self)
		  if not session.channels then
		     session.channels = {}
		  end

		  if not session.channels[self.channel] and
		     #self.entries > 0 then
		     local count, channel

		     -- Attempt to compress the channel by skipping
		     -- successive entries with constant value.

		     count = 0
		     channel = {}
		     
		     for i = 1, #self.entries do
			if i == 1 or i == #self.entries or
			   self.entries[i] ~= self.entries[i - 1] then
			   channel[i] = self.entries[i]
			   count = count + 1
			end
		     end
		     
		     session.channels[self.channel] = channel

		     print (string.format("Recorded %d entries (compressed to a count of %d) for black box channel %d.", #self.entries, count, self.channel))
		  end
	       end,

      update = function (self)
		  -- When saving user input to the black box we
		  -- convert to decimal notation with 5 decimal digits
		  -- to save space in the resulting serialized files.

		  if not session.channels then
		     self.entries[self.index] = string.format ("%.5f", self.input)
		  elseif self.index == #self.entries then
		     self.update = nil
		  end

		  self.output = tonumber(self.entries[self.index])
		  self.index = self.index + 1
	       end,
   }

   for key, value in pairs(values) do
      self[key] = value
   end

   return self
end
