/*   
 *   $Id: MainWidget.cpp,v 1.62 2006/04/19 21:24:25 rhizome Exp $
 *
 *      Copyright (C) 2004, 2005, 2006 Alex Marandon
 *
 *  This file is part of slag, a pattern-based audio sequencer.
 *
 *  slag is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  slag is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with slag; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#include <qfiledialog.h>
#include <qprogressdialog.h>
#include <qdir.h>
#include <qmap.h> 
#include <qwidgetstack.h> 
#include <qmenubar.h> 
#include <qstatusbar.h> 
#include <qmessagebox.h>
#include <qprogressdialog.h>
#include <qlabel.h>
#include <qaction.h>
#include <qregexp.h>
#include <qsettings.h>
#include <qscrollview.h> 
#include <qlayout.h> 

#include "../constants.h"
#include "../config.h"
#include "../Slag.h"
#include "../audio_engine/Serialize.h"
#include "../audio_engine/Pattern.h"
#include "../audio_engine/Part.h"
#include "../audio_engine/Channel.h"
#include "../audio_engine/Song.h"
#include "../audio_engine/SongFileExceptions.h"
#include "../audio_IO/AudioFileExceptions.h"
#include "StringListEdit.h"
#include "MainWidget.h"
#include "PatternWidget.h"
#include "ChannelPanel.h"
#include "ToolBar.h"
#include "TimeLine.h"
#include "song_properties/song_properties_dialog.h"
#include "preferences/preferences_dialog.h"


MainWidget::MainWidget(Slag* slag, QProgressDialog* progress_dialog) : 
    containerWidget(new QWidget(this)),
    scrollView(new QScrollView(containerWidget)),
    timeline(new TimeLine(containerWidget, 32)),
    scrollWidget(new QWidget(scrollView->viewport())),
    scrollWidgetLayout(new QHBoxLayout(scrollWidget, 0, 0)),
    stack(new QWidgetStack(scrollWidget)),
    channel_panel(new ChannelPanel(scrollWidget, slag)),
    song_properties_dialog(new SongPropertiesDialog),
    preferences_dialog(new PreferencesDialog),
    slag_file_dialog(new QFileDialog(this, "slag file dialog", TRUE)),
    wav_file_dialog(new QFileDialog(this, "wav file dialog", TRUE)),
    slag(slag),
    file_toolbar(new QToolBar( tr("File"), this)),
    toolbar(new ToolBar(this, slag)),
    patternLabel(new QLabel("", containerWidget)),
    file_menu(new QPopupMenu(this)),
    edit_menu(new QPopupMenu(this)),
    options_menu(new QPopupMenu(this)),
    help_menu(new QPopupMenu(this)),
    openLastSong(false),
    config(Config::instance())
{
    createActions();
    createMenus();
    createToolBars();
    createStatusBar();

    readSettings();

    slag_file_dialog->setFilter( tr("Slag files (*.xml *.slag)") );
    wav_file_dialog->setFilter( tr("WAV files (*.wav *.WAV)") );

    scrollView->viewport()->setBackgroundMode(Qt::PaletteBackground);
    scrollView->setHScrollBarMode(QScrollView::AlwaysOff);
    scrollView->addChild(scrollWidget);

    channel_panel->populate();
    scrollWidgetLayout->addWidget(channel_panel);
    scrollWidgetLayout->addWidget(stack);

    QVBoxLayout* vertical_layout = new QVBoxLayout(containerWidget);

    timeline_layout.addSpacing(16);
    patternLabel->setMaximumWidth(229);
    timeline_layout.addWidget(patternLabel);
    timeline_layout.addWidget(timeline);
    timeline_layout.setResizeMode(QLayout::Fixed);

    vertical_layout->addSpacing(5);
    vertical_layout->addLayout(&timeline_layout);
    vertical_layout->addSpacing(5);
    vertical_layout->addWidget(scrollView);

    setCentralWidget(containerWidget);
    songChanged(progress_dialog);
}

void MainWidget::stopAndClose() {
    slag->stop();
    close();
}

void MainWidget::createActions() {

    newAct = createAnAction(tr("&New"), tr("Ctrl+N"), 
            tr("Create a new song file"), SLOT(newFile()), "new.png");

    openAct = createAnAction(tr("&Open..."), "Ctrl+O", 
            tr("Open an existing song file"), SLOT(open()), "open.png");

    saveAct = createAnAction(tr("&Save"), "Ctrl+S", 
            tr("Save the song to disk"), SLOT(save()), "save.png");

    saveAsAct = createAnAction(tr("Save &As"), "Ctrl+Shift+S", 
            tr("Save the song under a new name"), SLOT(saveAs()));

    exitAct = createAnAction(tr("&Quit"), "Ctrl+Q", 
            tr("Exit the application"), SLOT(stopAndClose()));

    globalSettingsAct = createAnAction(tr("Global settings"), "Ctrl-G", 
            tr("Global settings for the application"), SLOT(showPreferences()));

    aboutAct = createAnAction(tr("About Slag"), 0, tr("About this application"), 
            SLOT(about()));
    
    aboutQtAct = new QAction(tr("About &Qt"), 0, this);
    aboutQtAct->setStatusTip(tr("About the Qt library"));
    connect(aboutQtAct, SIGNAL(activated()), qApp, SLOT(aboutQt()));
    
    songPropertiesAct = createAnAction(tr("Song &properties"), "Ctrl-P", 
            tr("Properties of the current song"), SLOT(showSongProperties()));

    exportWavAct = createAnAction(tr("Export &Wav"), "Ctrl+Shift+W", 
            tr("Export song as WAV sound file"), SLOT(exportWav()));

    addChannelAct = createAnAction(tr("Add &track"), "Ctrl+Shift+C", 
            tr("Add a track to the current Song"), SLOT(addChannel()));

    editPatternListAct = createAnAction(tr("Edit pattern list"), "Ctrl-Alt-P", 
            tr("Edit the list of patterns"), SLOT(editPatternList()));

}

QAction* MainWidget::createAnAction(
            const QString& label,
            const QString& shortCut,
            const QString& statusTip,
            const QString& connectionTarget,
            const QString& icon
            ) 
{
    QAction* action = new QAction(tr(label), tr(shortCut), this);
    action->setStatusTip(tr(statusTip));
    if (icon != NULL) action->setIconSet(QPixmap::fromMimeSource(icon));
    connect(action, SIGNAL(activated()), this, connectionTarget);
    return action;
}
                

void MainWidget::createMenus() {

    menuBar()->insertItem( tr("&File"), file_menu );
    newAct->addTo(file_menu);
    openAct->addTo(file_menu);
    saveAct->addTo(file_menu);
    saveAsAct->addTo(file_menu);
    file_menu->insertSeparator();
    exportWavAct->addTo(file_menu);
    file_menu->insertSeparator();
    exitAct->addTo(file_menu);
    for (int i = 0; i < MaxRecentFiles; ++i)
        recentFileIds[i] = -1;

    menuBar()->insertItem( tr("&Edit"), edit_menu );
    addChannelAct->addTo(edit_menu);
    editPatternListAct->addTo(edit_menu);
    //TODO not implemented yet
    //songPropertiesAct->addTo(edit_menu);

    menuBar()->insertItem( tr("&Options"), options_menu );
    globalSettingsAct->addTo(options_menu);

    menuBar()->insertItem( tr("&Help"), help_menu );
    aboutAct->addTo(help_menu);
    aboutQtAct->addTo(help_menu);
}

void MainWidget::createToolBars() {
    newAct->addTo(file_toolbar);
    openAct->addTo(file_toolbar);
    saveAct->addTo(file_toolbar);
}

void MainWidget::closeEvent(QCloseEvent *event) {
    if (maybeSave()) {
        writeSettings();
        event->accept();
    } else {
        event->ignore();
    }
}

void MainWidget::addChannel() {
    slag->song()->addChannel();
}

void MainWidget::editPatternList() {
    QStringList list = slag->song()->patternNames();
    StringListEdit listEdit(list, slag->song()->unamedPatternList());
    listEdit.setAskBeforeRemoving(true);
    listEdit.setCaption(tr("Pattern List"));
    listEdit.setTexts(tr("Add Pattern"), tr("&Name:"), tr("Edit Pattern Name"));
    if (listEdit.exec()) {
        StringItemList itemList = listEdit.list();
        slag->updatePatternList(itemList);
        toolbar->sync();
        songModified();
        // Handle renamed patterns
        for (StringItemList::const_iterator it = itemList.begin(); 
                it != itemList.end(); ++it) {
            const StringItem* patternItem = &(*it);
            if (patternItem->isRenamed && ! patternItem->isNew) {
                pattern_widget_map[patternItem->text()] = 
                    pattern_widget_map[patternItem->originalText];
                pattern_widget_map.remove(patternItem->originalText);
            }
        }
    } else {
        for (QStringList::Iterator it = list.begin(); it != list.end();
                ++it ) {
            slag->song()->freePatternName(*it);
        }
    }
}

void MainWidget::clearStack() {
    /// Delete all layers of the PatternWidget stack
    for (QMap<QString, QWidget*>::iterator it = pattern_widget_map.begin(); 
            it != pattern_widget_map.end(); ++it) {
        delete it.data();
    }
    pattern_widget_map.clear();
}

void MainWidget::buildStack(QProgressDialog* progress_dialog) {
    // Fill the widget stack with PatternWidgets
    PatternList patterns = slag->song()->patterns();
    uint i = progress_dialog->progress();
    for (PatternIterator pattern_it = patterns.begin();
            pattern_it != patterns.end(); ++pattern_it ) {
        addStackedWidget(*pattern_it);
        progress_dialog->setProgress(++i);
        slag->processEvents();
    }
}

void MainWidget::addStackedWidget(Pattern* pattern) {
    PatternWidget* stacked_widget = 
        new PatternWidget(this, slag->song(), pattern);
    stacked_widget->populate();
    
    stack->addWidget(stacked_widget);
    pattern_widget_map[pattern->name()] = stacked_widget;

    connect(channel_panel, SIGNAL(channelMoved(Channel*)),
            stacked_widget, SLOT(syncChannelPosition(Channel*)));
}

void MainWidget::removeStackedWidget(Pattern* pattern) {
    Q_ASSERT(pattern_widget_map.contains(pattern->name()));
    QWidget* stacked_widget = pattern_widget_map[pattern->name()];
    pattern_widget_map.remove(pattern->name());
    stack->removeWidget(stacked_widget);
    delete stacked_widget;
}

MainWidget::~MainWidget() {
    qDebug("Destruction Mainwidget");
}

void MainWidget::customEvent(QCustomEvent* event) {
    if ((int)event->type() == PatternChangeEventType) {
        PatternChangeEvent* patternChangeEvent = (PatternChangeEvent*) event;
        patternWidgetChange(patternChangeEvent->pattern_name);
    } else if ((int)event->type() == SongUpdateEventType) {
        timeline->oneStepForward();
    } else {
        QMainWindow::customEvent(event);
    }
}

void MainWidget::patternWidgetChange(const QString& name) {
    Q_ASSERT(pattern_widget_map.contains(name));
    stack->raiseWidget(pattern_widget_map[name]);
    patternLabel->setText(name);
    qDebug(QString("Current pattern changed to %1.").arg(name));
}

void MainWidget::newFile() {
    if (maybeSave()) {
        openSong(SongFile::defaultFileName);
    }
}

void MainWidget::open() {
    if (maybeSave()) {
        slag_file_dialog->setMode(QFileDialog::ExistingFile);
        slag_file_dialog->setCaption(tr("Open a file"));
        if ( slag_file_dialog->exec() == QDialog::Accepted ) {
            openSong(slag_file_dialog->selectedFile());
        }
    }
}

void MainWidget::openRecentFile(int param) {
    if (maybeSave()) {
        openSong(recentFiles[param]);
    }
}

void MainWidget::openSong(const QString& filename) {
    try {
        disconnect(slag->song(),  SIGNAL(channelAdded(Channel*)),
                channel_panel, SLOT(addChannel(Channel*)));
        slag->openSong(filename);
        channel_panel->clear();
        channel_panel->setSong(slag->song());
        channel_panel->populate();
    } catch(audio_engine::SongOpenException& e) {
        connect(slag->song(),  SIGNAL(channelAdded(Channel*)),
                channel_panel, SLOT(addChannel(Channel*)));
        QMessageBox::critical(this, tr("Slag - Opening song"), e.what());
    }
}

bool MainWidget::save() {
    try {
        if (slag->songFile()->isDefaultSongFile()) {
            return saveAs();
        } else {
            slag->save();
            modifLabel->setText( tr("Saved") );
            modified = false;
            return true;
        }
    } catch (audio_engine::SongSaveException& e) {
        QMessageBox::critical(this, tr("Slag - Saving song"), e.what());
        return false;
    }
}

bool MainWidget::overWriteFile(const QString& filename) {
    if (QFile::exists(filename)) {
        int ret = QMessageBox::warning(this, tr("Slag"),
                tr("File %1 already exists.\n"
                    "Do you want to overwrite it?")
                .arg(QDir::convertSeparators(filename)),
                QMessageBox::Yes | QMessageBox::Default,
                QMessageBox::No | QMessageBox::Escape);
        if (ret == QMessageBox::No) {
            return false;
        } else {
            return true;
        }
    } else {
        return true;
    }
}

bool MainWidget::saveAs() {
    slag_file_dialog->setMode(QFileDialog::AnyFile);
    slag_file_dialog->setCaption(tr("Save as new file"));
    if ( slag_file_dialog->exec() == QDialog::Accepted ) {
        QString filename = slag_file_dialog->selectedFile().stripWhiteSpace();
        if (filename.isEmpty())
            return false;

        // The user shouldn't have to worry about whether he has given the
        // right suffix or not.
        if (! QRegExp("^.+\\.(slag|xml)$").exactMatch(filename) ) 
            filename.append(".slag");

        if (not overWriteFile(filename)) {
            return true;
        }

        const QString old_filename = slag->songFile()->getFileName();
        slag->songFile()->setPath(filename);
        try {
            slag->save();
        } catch (audio_engine::SongSaveException& e) {
            slag->songFile()->setPath(old_filename);
            QMessageBox::critical(this, tr("Slag - Saving song"), e.what());
            return false;
        }
        modified = false;
        slag_file_dialog->rereadDir();
        recentFiles.remove(filename);
        recentFiles.push_front(filename);
        updateRecentFileItems();
        return true;
    } else {
        return false;
    }
}

bool MainWidget::maybeSave() {
    if (modified) {
        int ret = QMessageBox::warning(this, tr("Slag"),
                     tr("The song has been modified.\n"
                        "Do you want to save your changes?"),
                     QMessageBox::Yes | QMessageBox::Default,
                     QMessageBox::No,
                     QMessageBox::Cancel | QMessageBox::Escape);
        if (ret == QMessageBox::Yes)
            return save();
        else if (ret == QMessageBox::Cancel)
            return false;
    }
    return true;
}

void MainWidget::exportWav() {
    const QString popup_title = tr("Slag - WAV export");
    if (config->audioDriverName() == JACK_LABEL && config->jackTrackOutput()) {
        QMessageBox::warning(
                this, popup_title,
                tr("Sorry, WAV export is not implemented when JACK track "
                   "output is enabled.\nPlease use the global settings "
                   "dialog to disable it."));
        return;
    }
    wav_file_dialog->setMode(QFileDialog::AnyFile);
    wav_file_dialog->rereadDir();
    if ( wav_file_dialog->exec() == QDialog::Accepted ) {
        QString filename = wav_file_dialog->selectedFile();
        if (! QRegExp("^.+\\.[wW][aA][vV]$").exactMatch(filename) ) {
            filename.append(".wav");
        }
        if (not overWriteFile(filename)) {
            return;
        }
        QMessageBox::information(
                this, popup_title,
                tr("Audio output will be recorded into %1 \n"
                    "until you stop playback.")
                   .arg(filename)
                );
        try {
            slag->exportWav(filename);
        } catch (audio_IO::OpenException e) {
            QMessageBox::critical(
                    this, popup_title,
                    tr("Unable to open %1 for writing.\n%2")
                       .arg(e.filename, e.sys_msg)
                    );
        }
    }
}

void MainWidget::showSongProperties() {
    song_properties_dialog->show();
}

void MainWidget::songModified() {
    modified = true;
    modifLabel->setText( tr("<b>%1</b>").arg( tr("Unsaved") ) );
}

void MainWidget::songChanged(QProgressDialog* progress_dialog) {
    clearStack();
    buildStack(progress_dialog);
    toolbar->sync();
    QString filename = slag->songFile()->getFileName();

    if (filename != SongFile::defaultFileName) {
        setCaption("Slag - " + slag->songFile()->getBaseName());
        recentFiles.remove(filename);
        recentFiles.push_front(filename);
    } else {
        setCaption("Slag - New Song");
    }
    updateRecentFileItems();

    modified = false;
    modifLabel->setText( tr("Unchanged") );

}

void MainWidget::updateRecentFileItems() {
    while ((int)recentFiles.size() > MaxRecentFiles)
        recentFiles.pop_back();

    for (int i = 0; i < (int)recentFiles.size(); ++i) {
        QString text = tr("&%1 %2")
                       .arg(i + 1)
                       .arg(QFileInfo(recentFiles[i]).fileName());
        if (recentFileIds[i] == -1) {
            if (i == 0)
                file_menu->insertSeparator(file_menu->count() - 2);
            recentFileIds[i] =
                    file_menu->insertItem(text, this,
                                         SLOT(openRecentFile(int)),
                                         0, -1,
                                         file_menu->count() - 2);
            file_menu->setItemParameter(recentFileIds[i], i);
        } else {
            file_menu->changeItem(recentFileIds[i], text);
        }
    }
}

void MainWidget::deleteChannel(Channel* channel) {
    for (QMap<QString, QWidget*>::iterator it = pattern_widget_map.begin(); 
            it != pattern_widget_map.end(); ++it) {
        PatternWidget* pattern_widget = (PatternWidget*)it.data();
        pattern_widget->deleteChannel(channel);
    }
    channel_panel->deleteChannel(channel);
}

void MainWidget::rewind() {
    if ( ! Config::instance()->no_gui() ) {
        timeline->init();
    }
}

void MainWidget::createStatusBar() {
    modifLabel = new QLabel( tr("Unchanged"), this);

    //statusBar()->addWidget(new QLabel(tr(" PLOP ! "), this));
    //statusBar()->addWidget(new QLabel(tr(" Foo "), this), 1);
    statusBar()->addWidget(modifLabel);
}

void MainWidget::writeSettings() {
    QSettings settings;
    settings.setPath("alpage.org", "Slag");
    settings.beginGroup("/Slag");
    settings.writeEntry("/geometry/x", x());
    settings.writeEntry("/geometry/y", y());
    settings.writeEntry("/geometry/width", width());
    settings.writeEntry("/geometry/height", height());
    settings.writeEntry("/recentFiles", recentFiles);
    settings.writeEntry("/openLastSong", openLastSong);
    settings.endGroup();
}

void MainWidget::readSettings() {
    QSettings settings;
    settings.setPath("alpage.org", "Slag");
    settings.beginGroup("/Slag");
    int x = settings.readNumEntry("/geometry/x", 50);
    int y = settings.readNumEntry("/geometry/y", 50);
    int w = settings.readNumEntry("/geometry/width", 930);
    int h = settings.readNumEntry("/geometry/height", 220);
    recentFiles = settings.readListEntry("/recentFiles");
    openLastSong = settings.readBoolEntry("/openLastSong");
    settings.endGroup();

    move(x, y);
    resize(w, h);
}

void MainWidget::showPreferences() {
    
    // Setting preferences dialog fields acording to current config
    preferences_dialog->audioDriverSelected(config->audioDriverName());

    if(openLastSong)
        preferences_dialog->setOpenLastSong(true);
    else
        preferences_dialog->setOpenLastSong(false);

    if (! config->audioDriverName().isEmpty())
        preferences_dialog->setAudioDriver(config->audioDriverName());

    preferences_dialog->setAudioBufferSize(config->buf_size());
    preferences_dialog->setJackMulti(config->jackTrackOutput());
    preferences_dialog->setJackAutoConnect(config->jackAutoconnect());

    if(preferences_dialog->exec() == QDialog::Accepted) {
        openLastSong = preferences_dialog->getOpenLastSong();
        config->buf_size(preferences_dialog->getAudioBufferSize());
        config->setJackTrackOutput(preferences_dialog->getJackMulti());
        config->setAudioDriverName(preferences_dialog->getAudioDriver());
        config->setJackAutoconnect(preferences_dialog->getJackAutoConnect());

        writeSettings();
        config->save();

        slag->restartAudioDriver();
    }
}

void MainWidget::about() {
    QMessageBox::about(this, tr("About Slag"),
            tr("<h2>%1</h2>"
                "<p>A pattern-based audio sequencer.</p>"
                "<p>Copyright &copy; 2005, 2006<br>Alex Marandon &lt;al@alpage.org&gt;<br>Licensed under the GNU GPL.</p>"
                "<p>Included drum kit is \"Drumatic3 sounds\" by opm, from http://freesound.iua.upf.edu/, licensed under the Creative Commons Sampling Plus 1.0 License.</p>"
                ).arg(QString(PACKAGE_STRING)));
}

//EOF
