(************************************************************************)
(* This file is part of SKS.  SKS is free software; you can
   redistribute it and/or modify it under the terms of the GNU General
   Public License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
   USA *)
(***********************************************************************)

open StdLabels
open MoreLabels
open Printf
open Common
open Packet

module Kdb = Keydb.MakeUnrestricted(
  struct 
    let withtxn = !Settings.transactions
    and cache_bytes = !Settings.cache_bytes
    and pagesize = !Settings.pagesize
    and dbdir = !Settings.dbdir
    and dumpdir = !Settings.dumpdir
  end)

let () = Kdb.open_dbs ()

let rec strip_opt list = match list with
    [] -> []
  | None::tl -> strip_opt tl
  | (Some hd)::tl -> hd::(strip_opt tl)


let rec beginning n list = 
  if n = 0 then []
  else match list with
      [] -> []
    | hd::tl -> hd::(beginning (n-1) tl)

let merge_all keys = 
  let keys = Array.to_list keys in
  match keys with
      hd::tl ->
	List.fold_left ~init:hd tl 
	~f:(fun key1 key2 -> match KeyMerge.merge key1 key2 with
		None -> failwith "hit unparseable key"
	      | Some key -> key)
    | [] -> failwith "List too short"

let mergeable key1 key2 = 
  match KeyMerge.merge key1 key2 with
      None -> false
    | Some key -> true

exception KeyFail of string

let ctr = ref 0 
let click () = 
  incr ctr;
  if !ctr mod 100 = 0 
  then (
    printf "%d\n" !ctr;
    flush stdout;
  )

(*
open Bdb
let dbs = Kdb.get_dbs () 
let wordc = Cursor.create dbs.Kdb.word
let keyidc = Cursor.create dbs.Kdb.keyid

let test_cursor c ~key ~data = 
  try 
    Cursor.init_both c ~key ~data [];
    true
  with
      Not_found -> false

let (stream,close) = Kdb.create_hash_skey_stream ()

let rec get_hashes n accum =
  if n = 0 then accum
  else match SStream.next stream with
      None -> accum
    | Some (hash,skeystr) -> 
	get_hashes (n-1) ((hash,Kdb.skey_of_string skeystr)::accum)

let get_hashes n = get_hashes n []

let get_hashkeys n = 
  let pairs = Array.of_list (get_hashes n) in
  Array.sort ~cmp:(fun (x1,y1) (x2,y2) -> compare y1 y2) pairs; 
  Array.map ~f:(fun (hash,skey) -> (hash,Kdb.key_to_metadata 
				      (Kdb.key_of_skey skey))) 
    pairs

let timefunc f a =
  let timer = MTimer.create () in
  MTimer.start timer;
  let rval = f a in
  MTimer.stop timer;
  printf "Time: %f secs\n" (MTimer.read timer);
  rval


let test_keyids hash_mds = 
  Array.sort hash_mds 
    ~cmp:(fun (_,md1) (_,md2) -> compare 
	    md1.Kdb.md_keyid md2.Kdb.md_keyid);
  let ctr = ref 0 in
  Array.iter 
    ~f:(fun (hash,md) -> 
	  if not (test_cursor keyidc ~key:md.Kdb.md_keyid ~data:hash)
	  then (
	    incr ctr;
	    printf "KeyID test failed: (%d) %s\n" !ctr (KeyHash.hexify hash);
	    flush stdout)
       )
    hash_mds
    

let test_words hash_mds = 

  let wordpairs = 
    Array.map hash_mds 
      ~f:(fun (hash,md) ->
	    List.map ~f:(fun word -> (word,hash)) md.Kdb.md_words 
	 )
  in
  let wordpairs = Kdb.flatten_array_of_lists wordpairs in
  Array.sort ~cmp:compare wordpairs;
  let ctr = ref 0 in
  Array.iter wordpairs
    ~f:(fun (word,hash) ->
	  if not (test_cursor wordc ~key:word ~data:hash)
	  then (
	    incr ctr;
	    printf "Word test failed: (%d) %s - %s\n"
	      !ctr word (KeyHash.hexify hash);
	    flush stdout
	  )
       )
    
let test_chunk n =
  let hash_mds = get_hashkeys n in
  if Array.length hash_mds = 0 
  then false
  else (
    test_keyids hash_mds;
    test_words hash_mds;
    true
  )

let n = match !Settings.n with 0 -> 100000 | n -> n
let () = 
  protect ~f:(fun () -> 
		let ctr = ref 0 in
		while
		  incr ctr;
		  printf "Test %d\n" !ctr;  
		  flush stdout;
		  timefunc test_chunk n
		do () done
	     )
    ~finally:(fun () -> 
		printf "closing dbs\n";
		Kdb.close_dbs ())

(*
let () = Kdb.iter ~f:(fun ~key:hash ~data:keystr -> 
			  click ();
			  let key = Kdb.key_of_string keystr in
			  ()
			  (* let words = Kdb.key_extract_words key in
			  let keyid = Fingerprint.keyid_from_key key in
			  if not
			    (List.for_all words
			       ~f:(fun word -> 
				     test_cursor wordc ~key:word ~data:hash)
			     && test_cursor keyidc ~key:keyid ~data:hash)
			  then 
			    printf "failed: %s\n" (KeyHash.hexify hash) *)
		       )
			    
*)
*)
