/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

// STL
#include <iostream>
#include <cmath>
#include <algorithm> // for min_element and max_element

// FLTK
#include <FL/Fl.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Round_Button.H>
#include <FL/Fl_Input.H>
#include <FL/fl_draw.H>
#include <FL/Fl_Double_Window.H>

// SGA
#include "truss2d.h"
#include "triangleio.h"

// Global variables
int      gNNodes;
int      gNRods;
double * gNodes;
int    * gCon;
bool   * gEp;
double * gNbc;
double * gRadii;
double   gMaxRadius;

/////////////////////////////////////////////////////////////////////////////////////////////// MainWindow /////

// MainWindow
class MainWindow : public Fl_Double_Window
{
public:
	// DrawArea structure
	class DrawArea : public Fl_Widget
	{ // {{{
	public:
		// Constructor
		DrawArea (int xmin, int ymin, int width, int height) // Screen coordinates
			: Fl_Widget (xmin,ymin,width,height,0),
			  _truss    (NULL)
		{}

		// Methods
		void SetTruss (Truss2D const * Truss) { _truss=Truss; }

		// Internal methods
		void draw()
		{
			if (_truss!=NULL)
			{
				// Bounding box
				_Xmin = gNodes[0];
				_Ymin = gNodes[1];
				_Xmax = gNodes[0];
				_Ymax = gNodes[1];
				for (int i=1; i<gNNodes; ++i)
				{
					if (gNodes[i*2  ]<_Xmin) _Xmin = gNodes[i*2  ];
					if (gNodes[i*2+1]<_Ymin) _Ymin = gNodes[i*2+1];
					if (gNodes[i*2  ]>_Xmax) _Xmax = gNodes[i*2  ];
					if (gNodes[i*2+1]>_Ymax) _Ymax = gNodes[i*2+1];
				}
				_Xmin-=gMaxRadius; _Ymin-=gMaxRadius; _Xmax+=gMaxRadius; _Ymax+=gMaxRadius;

				// Scale factor
				_sf = (w()<h() ? static_cast<double>((w()-2)/(_Xmax-_Xmin)) : static_cast<double>((h()-2)/(_Ymax-_Ymin)));

				// Clear the background
				fl_color (FL_BACKGROUND_COLOR);
				fl_rectf (x(),y(),w(),h());

				// Draw circles
				for (int i=0; i<gNNodes; ++i)
				{
					int x = _x(gNodes[i*2]);
					int y = _y(gNodes[i*2+1]);
					int r = _l(gRadii[i]);
					fl_color  (FL_RED);
					fl_pie    (x-r, y-r, 2*r+1, 2*r+1, 0,360);
					fl_color  (FL_BLACK);
					fl_circle (x,y,r);
				}

				// Draw truss
				fl_color (FL_BLUE);
				for (int i=0; i<gNRods; ++i)
					fl_line (_x(gNodes[gCon[i*2]*2]), _y(gNodes[gCon[i*2]*2+1]), _x(gNodes[gCon[i*2+1]*2]), _y(gNodes[gCon[i*2+1]*2+1]));
			}
		}

	private:
		// Data
		double          _Xmin;  // Real coordinates
		double          _Ymin;  // Real coordinates
		double          _Xmax;  // Real coordinates
		double          _Ymax;  // Real coordinates
		double          _sf;    // Scale factor: x=xmin+(X-Xmin)*sf and y=ymin+ymax-(Y-Ymin)*sf, where x and y are screen coordinates
		Truss2D const * _truss; // Pointer to the current Truss2D

		// Private methods
		inline int _x (double   X) const { return static_cast<int>((x()+1)        +_sf*(X-_Xmin)); } // X in screen coordinates
		inline int _y (double   Y) const { return static_cast<int>((y()+1)+(h()-2)-_sf*(Y-_Ymin)); } // Y in screen coordinates
		inline int _l (double   L) const { return static_cast<int>(_sf*L)                        ; } // Length in screen coordinates

	}; // class DrawArea }}}

	// Constructor
	MainWindow(int width=1024, int height=824) : Fl_Double_Window (width,height,"Truss2D"), _truss(NULL), _prob(1), _type(0)
	{ // {{{

		// Set global variables
		gNodes = NULL;
		gCon   = NULL;
		gEp    = NULL;
		gNbc   = NULL;
		gRadii = NULL;

		// Add widgets to window
		begin();
			// Inputs
			_wprob = new Fl_Input( 70, 0,  30, 30, "Prob # =");
			char buf[256];
			snprintf(buf, 256, "%d", _prob); _wprob->value(buf);
			// Buttons
			_wattrac  = new Fl_Round_Button(100, 0, 100, 30, "Attraction"); if (_type==-1) _wattrac->set();
			_wboth    = new Fl_Round_Button(200, 0, 100, 30, "Both");       if (_type== 0) _wboth  ->set();
			_wrepuls  = new Fl_Round_Button(300, 0, 100, 30, "Repulsion");  if (_type==+1) _wrepuls->set();
			_wsetprob = new Fl_Button      (400, 0, 100, 30, "Set &Prob");
			_wstep    = new Fl_Button      (500, 0, 100, 30, "&Step");
			_wrun     = new Fl_Button      (600, 0, 100, 30, "&Run" );
			_wquit    = new Fl_Button      (700, 0, 100, 30, "&Quit");
			_wattrac ->callback(_attrac_cb , this);
			_wboth   ->callback(_both_cb   , this);
			_wrepuls ->callback(_repuls_cb , this);
			_wsetprob->callback(_setprob_cb, this);
			_wquit   ->callback(_quit_cb   , this);
			_wrun    ->callback(_run_cb    , this);
			_wstep   ->callback(_step_cb   , this);
			// Draw area
			_wda = new DrawArea(/*xmin*/5, /*ymin*/35 , /*width*/w()-10, /*height*/h()-40);
		end();

		// Set focus
		_wrun->take_focus();

		// Set resizable and show window
		//resizable(this);
		//resizable(_wda);
		show();

	} // }}}

	// Destructor
	~MainWindow()
	{ // {{{
		delete _wattrac;
		delete _wboth;
		delete _wrepuls;
		delete _wprob;
		delete _wquit;
		delete _wsetprob;
		delete _wrun;
		delete _wstep;
		delete _wda;
		if (gNodes!=NULL) delete [] gNodes;
		if (gCon  !=NULL) delete [] gCon;
		if (gEp   !=NULL) delete [] gEp;
		if (gNbc  !=NULL) delete [] gNbc;
		if (gRadii!=NULL) delete [] gRadii;
		if (_truss!=NULL) delete _truss;
	} // }}}

private:
	// Widgets
	Fl_Round_Button * _wattrac;
	Fl_Round_Button * _wboth;
	Fl_Round_Button * _wrepuls;
	Fl_Input        * _wprob;
	Fl_Button       * _wsetprob;
	Fl_Button       * _wquit;
	Fl_Button       * _wrun;
	Fl_Button       * _wstep;
	DrawArea        * _wda; 
	Truss2D         * _truss;
	int               _prob;
	int               _type; // -1: only attraction, 0: both, +1: only repulsion

	// Callbacks (must be static)
	static void _attrac_cb (Fl_Widget * o, void * v) { ((MainWindow*)v)->_attrac (); }
	static void _both_cb   (Fl_Widget * o, void * v) { ((MainWindow*)v)->_both   (); }
	static void _repuls_cb (Fl_Widget * o, void * v) { ((MainWindow*)v)->_repuls (); }
	static void _setprob_cb(Fl_Widget * o, void * v) { ((MainWindow*)v)->_setprob(); }
	static void _run_cb    (Fl_Widget * o, void * v) { ((MainWindow*)v)->_run    (); }
	static void _step_cb   (Fl_Widget * o, void * v) { ((MainWindow*)v)->_step   (); }
	static void _quit_cb   (Fl_Widget * o, void * v) { ((MainWindow*)v)->_quit   (); }

	// Solver type
	inline void _attrac () { _type = -1; _wboth  ->clear(); _wrepuls->clear(); }
	inline void _both   () { _type =  0; _wattrac->clear(); _wrepuls->clear(); }
	inline void _repuls () { _type = +1; _wattrac->clear(); _wboth  ->clear(); }

	// SetProb callback
	inline void _setprob ()
	{ // {{{

		// Clear previous arrays
		if (gNodes!=NULL) delete [] gNodes;
		if (gCon  !=NULL) delete [] gCon;
		if (gEp   !=NULL) delete [] gEp;
		if (gNbc  !=NULL) delete [] gNbc;
		if (gRadii!=NULL) delete [] gRadii;
		if (_truss!=NULL) delete _truss;

		// Input
		_prob = atoi(_wprob->value());

		if (_prob==1) // problem #1
		{ // {{{
			// Set global variables
			gNNodes   = 4;
			gNRods    = 5;
			gNodes    = new double [gNNodes*2];
			gCon      = new int    [gNRods*2 ];
			gEp       = new bool   [gNNodes*2];
			gNbc      = new double [gNNodes*2];
			gRadii    = new double [gNNodes  ];

			// Coordinates
			gNodes[0] = 0.0;
			gNodes[1] = 0.0;
			gNodes[2] = 1.0;
			gNodes[3] = 0.0;
			gNodes[4] = 1.0;
			gNodes[5] = 1.0;
			gNodes[6] = 0.0;
			gNodes[7] = 1.0;

			// Connectivity
			gCon[0]=0;  gCon[1]=1;
			gCon[2]=1;  gCon[3]=2;
			gCon[4]=2;  gCon[5]=3;
			gCon[6]=3;  gCon[7]=0;
			gCon[8]=0;  gCon[9]=2;

			// Boundary conditions
			for (int i=0; i<gNNodes*2; ++i) { gEp[i]=false; gNbc[i]=0; }
			gEp[0]= gEp[1] = gEp[6]= true;

			// Radii
			gRadii[0] = 0.6;
			gRadii[1] = 0.8;
			gRadii[2] = 0.2;
			gRadii[3] = 0.5;
			gMaxRadius = 0.8;
		} // }}}
		else if (_prob==2) // problem #2
		{ // {{{
			// Set global variables
			gNNodes   = 6;
			gNRods    = 10;
			gNodes    = new double [gNNodes*2];
			gCon      = new int    [gNRods*2 ];
			gEp       = new bool   [gNNodes*2];
			gNbc      = new double [gNNodes*2];
			gRadii    = new double [gNNodes  ];

			// Coordinates
			gNodes[ 0] = 0.0;
			gNodes[ 1] = 0.0;
			gNodes[ 2] = 1.0;
			gNodes[ 3] = 0.0;
			gNodes[ 4] = 0.5;
			gNodes[ 5] = 0.5;
			gNodes[ 6] = 0.0;
			gNodes[ 7] = 0.8;
			gNodes[ 8] = 1.0;
			gNodes[ 9] = 0.9;
			gNodes[10] = 0.5;
			gNodes[11] = 1.0;

			// Connectivity
			gCon[ 0]=0;  gCon[ 1]=1;
			gCon[ 2]=1;  gCon[ 3]=4;
			gCon[ 4]=4;  gCon[ 5]=5;
			gCon[ 6]=5;  gCon[ 7]=3;
			gCon[ 8]=3;  gCon[ 9]=0;
			gCon[10]=2;  gCon[11]=0;
			gCon[12]=2;  gCon[13]=1;
			gCon[14]=2;  gCon[15]=4;
			gCon[16]=2;  gCon[17]=5;
			gCon[18]=2;  gCon[19]=3;

			// Boundary conditions
			for (int i=0; i<gNNodes*2; ++i) { gEp[i]=false; gNbc[i]=0; }
			gEp[0] = gEp[1] = gEp[6]= true;

			// Radii
			gRadii[0] = 0.6;
			gRadii[1] = 0.8;
			gRadii[2] = 0.2;
			gRadii[3] = 0.5;
			gRadii[4] = 0.5;
			gRadii[5] = 0.5;
			gMaxRadius = 0.8;
		} // }}}
		else // problem #3
		{ // {{{
			// Set global variables
			gNNodes   = 6;
			gNRods    = 10;
			gNodes    = new double [gNNodes*2];
			gCon      = new int    [gNRods*2 ];
			gEp       = new bool   [gNNodes*2];
			gNbc      = new double [gNNodes*2];
			gRadii    = new double [gNNodes  ];

			// Coordinates
			gNodes[ 0] = 0.0;
			gNodes[ 1] = 0.0;
			gNodes[ 2] = 1.0;
			gNodes[ 3] = 0.0;
			gNodes[ 4] = 0.5;
			gNodes[ 5] = 0.5;
			gNodes[ 6] = 0.0;
			gNodes[ 7] = 0.8;
			gNodes[ 8] = 1.0;
			gNodes[ 9] = 0.9;
			gNodes[10] = 0.5;
			gNodes[11] = 1.0;

			// Connectivity
			gCon[ 0]=0;  gCon[ 1]=1;
			gCon[ 2]=1;  gCon[ 3]=4;
			gCon[ 4]=4;  gCon[ 5]=5;
			gCon[ 6]=5;  gCon[ 7]=3;
			gCon[ 8]=3;  gCon[ 9]=0;
			gCon[10]=2;  gCon[11]=0;
			gCon[12]=2;  gCon[13]=1;
			gCon[14]=2;  gCon[15]=4;
			gCon[16]=2;  gCon[17]=5;
			gCon[18]=2;  gCon[19]=3;

			// Boundary conditions
			for (int i=0; i<gNNodes*2; ++i) { gEp[i]=false; gNbc[i]=0; }
			gEp[0] = gEp[1] = gEp[6]= true;

			// Radii
			gRadii[0] = 0.4;
			gRadii[1] = 0.3;
			gRadii[2] = 0.6;
			gRadii[3] = 0.2;
			gRadii[4] = 0.1;
			gRadii[5] = 0.5;
			gMaxRadius = 0.6;
		} // }}}

		// Truss
		_truss = new Truss2D (gNNodes,gNRods,gNodes,gCon,gEp,gNbc);

		// Draw initial truss
		_wda->SetTruss(_truss);
		_wda->redraw();
		Fl::wait(0);

	} // }}}

	// Run callback
	inline void _run ()
	{ // {{{

		double diff  = 1.0;
		double error = 0.0;
		double tol   = 1.0e-8;
		int    it    = 1;
		int    maxit = 1000;
		while (diff>tol && it<=maxit)
		{
			double err   = _step();
			       diff  = fabs(error-err);
			       error = err;
			it++;
		}
		if (it>=maxit) std::cout << "Did not converge!" << std::endl;
		//std::cout << "it = " << it << std::endl;

	} // }}}

	// Step callback
	inline double _step ()
	{ // {{{

		if (_truss!=NULL)
		{
			// Clear previous U, F and Fint values
			_truss->Initialize ();

			// Calculate "internal" forces due to Circles under/overlapping
			_truss->CirclesForce (1.0, gRadii, gNbc, _type);

			// Solve for these "internal" forces
			_truss->Solve ();

			// Update coordinates
			for (int j=0; j<2*gNNodes; ++j) gNodes[j] += _truss->U()[j];

			// Update truss
			TriangleIO in(gNNodes);
			TriangleIO out;
			for (int j=0; j<gNNodes*2; ++j) in.R().pointlist[j] = gNodes[j];
			triangulate("Qze",in.P(),out.P(),NULL);
			gNRods = out.R().numberofedges;
			delete [] gCon;
			gCon = new int [gNRods*2];
			for (int j=0; j<gNRods*2; ++j) gCon[j] = out.R().edgelist[j];
			_truss->SetNRods (gNRods);
			_truss->SetCon   (gCon);

			// Calculate error
			double error = _truss->CirclesError(gRadii, _type);
			std::cout << "Error = " << _truss->CirclesError(gRadii) << std::endl;

			// Draw
			_wda->redraw();
			Fl::wait(0);

			return error;
		}
		else return -1;

	} // }}}

	// Quit callback
	inline void _quit() { hide(); }

}; // class MainWindow

/////////////////////////////////////////////////////////////////////////////////////////////// MainWindow /////

int main(int argc, char **argv) try
{
	MainWindow win;
	Fl::run();
	return 0;
}
catch (char const * m) // {{{
{
	std::cout << "Fatal: " << m << std::endl;
	exit (1);
}
catch (...)
{
	std::cout << "Some exception (...) ocurred\n";
} //}}} 

// vim:fdm=marker
