/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

// STL
#include <iostream>
#include <cmath>

// SGA
#include "truss2d.h"

using std::cout;
using std::endl;

int main(int argc, char **argv) try
{
	{ // simple truss
		const int nn = 3; // num nodes
		const int nr = 3; // num rods

		double nodes    [nn*2] = {0,0, 10,0, 10,10};
		int    con      [nr*2] = {0,1, 1,2, 2,0};
		double props    [nr  ] = {10,5,20};
		bool   ep       [nn*2] = {true,true, false,true, false,false};
		double ebc      [nn*2] = {0,-0.5, 0,0.4, 0,0};
		double nbc      [nn*2] = {0,0, 0,0, 2,1};
		double ucorrect [nn*2] = {0,-0.5, 0,0.4, -0.5,0.2};
		double fcorrect [nn*2] = {-2,-2, 0,1, 2,1};

		// Solve
		Truss2D truss (nn,nr,nodes,con,ep,nbc,ebc,props);
		truss.Solve ();

		// Check
		double error = 0.0;
		for (int i=0; i<nn*2; ++i) error += fabs(ucorrect[i]-truss.U()[i]);
		for (int i=0; i<nn*2; ++i) error += fabs(fcorrect[i]-truss.F()[i]);
		cout << "Simple truss <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<\n";
		cout << "error = " << error << endl;
		cout << "F     = "; for (int i=0; i<nn*2; ++i) cout << truss.F()   [i] << " "; cout<<endl;
		cout << "U     = "; for (int i=0; i<nn*2; ++i) cout << truss.U()   [i] << " "; cout<<endl;
		cout << "Fint  = "; for (int i=0; i<nn*2; ++i) cout << truss.Fint()[i] << " "; cout<<endl;
		cout << "Res   = "; for (int i=0; i<nn*2; ++i) cout << truss.Res() [i] << " "; cout<<endl<<endl;
	}
	{ // bridge

		const int nn = 12; // num nodes
		const int nr = 21; // num rods

		double nodes    [nn*2] = {0,0, 10,5, 10,0, 20,8, 20,0, 30,9, 30,0, 40,8, 40,0, 50,5, 50,0, 60,0};
		int    con      [nr*2] = {0,2,2,4,4,6,6,8,8,10,10,11,0,1,1,3,3,5,5,7,7,9,9,11,1,2,3,4,5,6,7,8,9,10,1,4,3,6,6,7,8,9};
		double props    [nr  ] = {2,2,2,2,2,2, 10,10,10,10,10,10, 3,3,3,3,3, 1,1,1,1}; for (int i=0; i<nr; ++i) props[i]*=1000.0;
		bool   ep       [nn*2]; for (int i=0; i<2*nn; ++i) ep [i] = false; ep[0]=true; ep[1]=true; ep[23]=true;
		double ebc      [nn*2]; for (int i=0; i<2*nn; ++i) ebc[i] = 0.0;
		double nbc      [nn*2] = {0,0, 0,0, 0,-10, 0,0, 0,-10, 0,0, 0,-16, 0,0, 0,-10, 0,0, 0,-10, 0,0};
		double ucorrect [nn*2] = {0.0, 0.0, 0.809536, -1.775600, 0.280000, -1.792260, 0.899001, -2.291930, 0.560000, -2.316600, 0.847500, -2.385940, 0.847500, -2.421940, 0.795999, -2.291930, 1.135000, -2.316600, 0.885464, -1.775600, 1.415000, -1.792260, 1.695000, 0.0};
		double fcorrect [nn*2] = {0,28, 0,0, 0,-10, 0,0, 0,-10, 0,0, 0,-16, 0,0, 0,-10, 0,0, 0,-10, 0,28};

		// Solve
		Truss2D truss (nn,nr,nodes,con,ep,nbc,ebc,props,true);
		truss.Solve (10);

		// Check
		double error = 0.0;
		for (int i=0; i<nn*2; ++i) error += fabs(ucorrect[i]-truss.U()[i]);
		for (int i=0; i<nn*2; ++i) error += fabs(fcorrect[i]-truss.F()[i]);
		cout << "Bridge <<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<\n";
		cout << "error = " << error << endl;
	}

	return 0;
}
catch (char const * m) // {{{
{
	std::cout << "Fatal: " << m << std::endl;
	exit (1);
}
catch (...)
{
	std::cout << "Some exception (...) ocurred\n";
} //}}} 

// vim:fdm=marker
