/************************************************************************
 * SGA - A C++ library to help develop Simple Genetic Algorithms        *
 * Copyright (C) 2005 Dorival M. Pedroso                                *
 *                                                                      *
 * This program is free software: you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation, either version 3 of the License, or    *
 * any later version.                                                   *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the         *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program. If not, see <http://www.gnu.org/licenses/>  *
 ************************************************************************/

// STL
#include <iostream>
#include <sstream>

// OpenMPI
#include <mpi.h>

// SGA
#include "random.h"
#include "numstreams.h"

using std::cout;
using std::endl;
using SGA::_6;

// The following two functions must be declared before: #include "island.h"
inline void MutationFun (uint32_t & OldGene)
{
	if (Rnd::Flip(0.5)) OldGene /= 2;
	else                OldGene *= 2;
}
inline double ObjectiveFun (uint32_t const G[], uint32_t nGenes)
{
	double sum = 0.0;
	for (uint32_t i=0; i<nGenes; ++i)
		sum += G[i];
	return sum;
}
#include "island.h"
#include "evolver.h"

inline void Report (uint32_t iGeneration, SGA::Island<uint32_t> const & Isl)
{
	uint32_t my_id = MPI::COMM_WORLD.Get_rank(); // my processor ID
	if (my_id==0) // Root
	{
		double best_ov;  Isl.Best (best_ov );
		double worst_ov; Isl.Worst(worst_ov);
		if (iGeneration==0) cout << _6 << "Gen"       << _6 << "BestOV" << _6 << "WorstOV" << endl;
		else                cout << _6 << iGeneration << _6 <<  best_ov << _6 <<  worst_ov << endl;
	}
}

int main(int argc, char **argv) try
{
	// Initialize MPI
	MPI::Init(argc, argv);
	uint32_t my_id = MPI::COMM_WORLD.Get_rank(); // my processor ID

	// Constants
	const uint32_t NINDIV    = 4;  // Number of individuals
	const uint32_t NGENES    = 3;  // Number of genes
	const uint32_t NGENERAT  = 10; // Number of generations
	const uint32_t MIGINTERV = 10; // Migration interval
	const uint32_t NINDIVMIG = 0;  // Number of individuals that migrate
	const uint32_t RPTINTERV = 1;  // Report interval

	// Minimum and maximum for all Genes
	uint32_t Mi[NGENES] = { 1, 20, 300 };
	uint32_t Ma[NGENES] = { 2, 40, 600 };

	// Initialize random numbers generator
	if (argc>1) Rnd::Init(my_id*atoi(argv[1]));
	else        Rnd::Init(1234);

	// My Island
	SGA::Island<uint32_t> isl(NINDIV,NGENES,true,1,2);

	// Generate a random island
	isl.RandomFill (Mi, Ma);

	// Output - Before
	std::ostringstream oss;
	oss << "--------------------------- Before ----------------------------\n";
	oss<<"Isl # "<<my_id<<"\n";  isl.OutIsland(oss);
	cout << oss.str() << endl;

	// Evolution
	for (uint32_t i=0; i<NGENERAT; ++i)
	{
		isl.Reproduction();
		if ((i>0) && (i%MIGINTERV==0)) SGA::Migrate (NINDIVMIG, isl);
		if (          i%RPTINTERV==0 ) Report       (i, isl);
	}

	// Output - After
	oss.str("");
	oss << "---------------------------- After ----------------------------\n";
	oss<<"Isl # "<<my_id<<"\n";  isl.OutIsland(oss);
	cout << oss.str() << endl;

	// End
	MPI::Finalize();
	return 0;
}
catch (char const * m) // {{{
{
	std::cout << "Fatal: " << m << std::endl;
	MPI::Finalize();
	exit (1);
}
catch (...)
{
	std::cout << "Some exception (...) ocurred\n";
	MPI::Finalize();
	exit (1);
} //}}} 

// vim:fdm=marker
